/* 
 * Code generated by Speakeasy (https://speakeasy.com). DO NOT EDIT.
 */
package com.stackone.stackone_client_java.utils.reactive;

import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.stackone.stackone_client_java.utils.AsyncResponse;
import com.stackone.stackone_client_java.utils.Blob;
import com.stackone.stackone_client_java.utils.EventStreamMessage;
import com.stackone.stackone_client_java.utils.SpeakeasyLogger;
import com.stackone.stackone_client_java.utils.StreamingParser;
import com.stackone.stackone_client_java.utils.Utils;

import java.net.http.HttpResponse;
import java.nio.ByteBuffer;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Flow;
import java.util.concurrent.atomic.AtomicLong;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscriber;
import org.reactivestreams.Subscription;

/**
 * A reactive event stream publisher that can handle different protocols (SSE, JSONL)
 * and emits typed events with proper backpressure handling.
 *
 * @param <ResponseT> the AsyncResponse type that contains the event stream
 * @param <ItemT> the type that events are deserialized into
 */
public class EventStream<ResponseT extends AsyncResponse, ItemT> implements Publisher<ItemT> {

    private static final SpeakeasyLogger logger = SpeakeasyLogger.getLogger(EventStream.class);
    
    /**
     * Protocol interface that defines how to parse and process different event stream formats
     */
    public interface Protocol<ParsedT, ItemT> {
        /**
         * Create a new parser instance for this protocol
         */
        StreamingParser<ParsedT> createParser();
        
        /**
         * Process a parsed item and convert it to the target type
         * @param parsed the parsed item from the parser
         * @param objectMapper the ObjectMapper for deserialization
         * @param typeReference the target type reference
         * @return the converted item, or null if this item should be skipped
         * @throws Exception if conversion fails
         */
        ItemT processItem(ParsedT parsed, ObjectMapper objectMapper, TypeReference<ItemT> typeReference) throws Exception;
        
        /**
         * Check if processing should stop (e.g., terminal message encountered)
         * @param parsed the parsed item
         * @return true if processing should stop
         */
        boolean shouldStop(ParsedT parsed);
    }

    private final CompletableFuture<ResponseT> asyncResponseFuture;
    private final TypeReference<ItemT> typeReference;
    private final ObjectMapper objectMapper;
    private final Protocol<?, ItemT> protocol;

    private EventStream(CompletableFuture<ResponseT> asyncResponseFuture,
                       TypeReference<ItemT> typeReference,
                       ObjectMapper objectMapper,
                       Protocol<?, ItemT> protocol) {
        this.asyncResponseFuture = asyncResponseFuture;
        this.typeReference = typeReference;
        this.objectMapper = objectMapper;
        this.protocol = protocol;
        logger.debug("Reactive EventStream initialized for type: {}", typeReference.getType().getTypeName());
    }

    /**
     * Create an EventStream for SSE (Server-Sent Events) format
     */
    public static <ResponseT extends AsyncResponse, ItemT> EventStream<ResponseT, ItemT> forSSE(
            CompletableFuture<ResponseT> asyncResponseFuture,
            TypeReference<ItemT> typeReference,
            ObjectMapper objectMapper,
            String terminalMessage) {
        return new EventStream<>(asyncResponseFuture, typeReference, objectMapper, 
                                new SSEProtocol<>(terminalMessage));
    }

    /**
     * Create an EventStream for JSONL (JSON Lines) format
     */
    public static <ResponseT extends AsyncResponse, ItemT> EventStream<ResponseT, ItemT> forJsonL(
            CompletableFuture<ResponseT> asyncResponseFuture,
            TypeReference<ItemT> typeReference,
            ObjectMapper objectMapper) {
        return new EventStream<>(asyncResponseFuture, typeReference, objectMapper, 
                                new JsonLProtocol<>());
    }

    /**
     * Returns the value of the Content-Type header.
     **/
    public CompletableFuture<String> contentType() {
        return asyncResponseFuture.thenApply(AsyncResponse::contentType);
    }
    
    /**
     * Returns the HTTP status code.
     **/
    public CompletableFuture<Integer> statusCode() {
        return asyncResponseFuture.thenApply(AsyncResponse::statusCode);
    }
    
    /**
     * Returns the raw HTTP response.
     **/
    public CompletableFuture<HttpResponse<Blob>> rawResponse() {
        return asyncResponseFuture.thenApply(AsyncResponse::rawResponse);
    }
    
    /**
     * Returns the AsyncResponse body.
     **/
    public CompletableFuture<ResponseT> body() {
        return asyncResponseFuture;
    }

    @Override
    public void subscribe(Subscriber<? super ItemT> subscriber) {
        if (subscriber == null) {
            throw new NullPointerException("Subscriber cannot be null");
        }

        EventStreamSubscription subscription = new EventStreamSubscription(subscriber);
        subscriber.onSubscribe(subscription);
        // Start the async operation only after onSubscribe has been called
        subscription.start(rawResponse());
    }

    private class EventStreamSubscription implements Subscription {
        private final Subscriber<? super ItemT> subscriber;
        private final AtomicLong demand = new AtomicLong(0);
        private final StreamingParser<?> parser;
        
        private Flow.Subscription upstreamSubscription;
        private volatile boolean cancelled = false;
        private volatile boolean completed = false;

        @SuppressWarnings("unchecked")
        public EventStreamSubscription(Subscriber<? super ItemT> subscriber) {
            this.subscriber = subscriber;
            this.parser = ((Protocol<Object, ItemT>) protocol).createParser();
        }

        public void start(CompletableFuture<HttpResponse<Blob>> httpResponseFuture) {
            // Wait for the CompletableFuture and then subscribe to the Blob
            httpResponseFuture.whenComplete((httpResponse, throwable) -> {
                if (cancelled) {
                    return;
                }
                if (throwable != null) {
                    // Signal error immediately per Reactive Streams specification
                    signalError(throwable);
                    return;
                }

                // Extract Blob from HttpResponse and subscribe to it
                Blob blob = httpResponse.body();
                // Blob.asPublisher() now returns Flow.Publisher directly
                Flow.Publisher<ByteBuffer> flowPublisher;
                try {
                    flowPublisher = blob.asPublisher();
                } catch (Exception e) {
                    // Handle case where blob is already consumed or other errors
                    signalError(e);
                    return;
                }
                flowPublisher.subscribe(new Flow.Subscriber<>() {
                    @Override
                    public void onSubscribe(Flow.Subscription subscription) {
                        if (cancelled) {
                            subscription.cancel();
                            return;
                        }
                        upstreamSubscription = subscription;
                        requestMoreIfNeeded();
                    }

                    @Override
                    public void onNext(ByteBuffer byteBuffer) {
                        if (cancelled || completed) {
                            return;
                        }
                        try {
                            processBuffer(byteBuffer);
                            if (!completed) {
                                requestMoreIfNeeded();
                            }
                        } catch (Exception e) {
                            signalError(e);
                        }
                    }

                    @Override
                    public void onError(Throwable throwable) {
                        signalError(throwable);
                    }

                    @Override
                    public void onComplete() {
                        try {
                            processEndOfStream();
                            signalComplete();
                        } catch (Exception e) {
                            signalError(e);
                        }
                    }
                });
            });
        }

        @Override
        public void request(long n) {
            if (n <= 0) {
                signalError(new IllegalArgumentException("Request amount must be positive"));
                return;
            }
            if (cancelled || completed) {
                return;
            }
            demand.addAndGet(n);
            requestMoreIfNeeded();
        }

        @Override
        public void cancel() {
            if (!cancelled) {
                cancelled = true;
                if (upstreamSubscription != null) {
                    upstreamSubscription.cancel();
                }
            }
        }

        private void processBuffer(ByteBuffer byteBuffer) {
            // Use ByteBuffer directly without copying
            Optional<?> parsedOpt = parser.add(byteBuffer);
            while (parsedOpt.isPresent()) {
                if (!processItem(parsedOpt.get())) {
                    return; // Stop processing (terminal condition or error)
                }
                // Check for additional items in the buffer
                parsedOpt = parser.next();
            }
        }

        @SuppressWarnings("unchecked")
        private boolean processItem(Object parsed) {
            Protocol<Object, ItemT> typedProtocol = (Protocol<Object, ItemT>) protocol;
            
            // Check if processing should stop
            if (typedProtocol.shouldStop(parsed)) {
                signalComplete();
                return false;
            }
            
            // Emit if there's demand
            if (demand.get() > 0) {
                try {
                    ItemT item = typedProtocol.processItem(parsed, objectMapper, typeReference);
                    if (item != null) {
                        demand.decrementAndGet();
                        if (logger.isTraceEnabled()) {
                            logger.trace("Reactive EventStream item emitted");
                        }
                        subscriber.onNext(item);
                    }
                } catch (Exception e) {
                    logger.debug("Error processing reactive EventStream item: {}", e.getMessage());
                    signalError(e);
                    return false; // Signal to stop processing on error
                }
            }
            return true; // Continue processing
        }

        private void requestMoreIfNeeded() {
            if (cancelled || completed) {
                return;
            }

            if (upstreamSubscription != null && demand.get() > 0) {
                upstreamSubscription.request(1);
            }
        }

        private void processEndOfStream() {
            Optional<?> parsedOpt = parser.finish();
            parsedOpt.ifPresent(this::processItem);
        }

        private void signalError(Throwable t) {
            if (!cancelled && !completed) {
                completed = true;
                subscriber.onError(t);
            }
        }

        private void signalComplete() {
            if (!cancelled && !completed) {
                completed = true;
                logger.debug("Reactive EventStream completed");
                subscriber.onComplete();
            }
        }
    }

    /**
     * SSE Protocol implementation
     */
    private static class SSEProtocol<ItemT> implements Protocol<EventStreamMessage, ItemT> {
        private final String terminalMessage;

        public SSEProtocol(String terminalMessage) {
            this.terminalMessage = terminalMessage;
        }

        @Override
        public StreamingParser<EventStreamMessage> createParser() {
            return StreamingParser.forSSE();
        }

        @Override
        public ItemT processItem(EventStreamMessage message, ObjectMapper objectMapper, TypeReference<ItemT> typeReference) {
            // Skip empty data messages
            if (message.data().isEmpty()) {
                return null;
            }
            return Utils.asType(message, objectMapper, typeReference);
        }

        @Override
        public boolean shouldStop(EventStreamMessage message) {
            // Check if this is a terminal message
            return terminalMessage != null && terminalMessage.equals(message.data());
        }
    }

    /**
     * JSONL Protocol implementation
     */
    private static class JsonLProtocol<ItemT> implements Protocol<String, ItemT> {

        @Override
        public StreamingParser<String> createParser() {
            return StreamingParser.forJsonLines();
        }

        @Override
        public ItemT processItem(String jsonLine, ObjectMapper objectMapper, TypeReference<ItemT> typeReference) throws Exception {
            return objectMapper.readValue(jsonLine, typeReference);
        }

        @Override
        public boolean shouldStop(String jsonLine) {
            // JSONL doesn't have terminal messages
            return false;
        }
    }

}