package fi.evolver.script.app;

import java.io.ByteArrayInputStream;
import java.net.URI;
import java.nio.file.Path;
import java.util.Arrays;
import java.util.stream.Stream;

import fi.evolver.script.Download;
import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class Apt {

	/**
	 * Install the specified packages with APT.
	 *
	 * @param packageNames Packages to install.
	 */
	public static void install(String... packageNames) {
		try (Step step = Step.start("Install %s".formatted(String.join(", ", packageNames)))) {
			if (Arrays.stream(packageNames).allMatch(Apt::isInstalled)) {
				step.skip("Already installed");
				return;
			}

			update();
			Shell.sudo(Stream.concat(Stream.of("apt-get", "install", "-y"), Arrays.stream(packageNames)).toList());
		}
	}

	public static void installFromUrl(URI uri) {
		try (Step step = Step.start("Install deb from %s".formatted(uri))) {
			Path debPath = Download.intoTempFile(uri);
			install(debPath.toString());
		}
	}

	public static void purge(String... packageNames) {
		try (Step step = Step.start("Purge %s".formatted(String.join(", ", packageNames)))) {
			Shell.sudo(Stream.concat(Stream.of("apt-get", "purge", "-y"), Arrays.stream(packageNames)).toList());
		}
	}

	/**
	 * Update local package information cache.
	 */
	public static void update() {
		try (Step step = Step.start("Update package lists")) {
			Shell.sudo("apt-get", "update");
		}
	}

	public static boolean isInstalled(String packageName) {
		Shell.Result result = Shell.Command.user("dpkg-query", "--show", "--showformat=${db:Status-Status}", packageName)
				.failOnError(false)
				.run();
		return result.success() && "installed".equals(result.stdout());
	}

	/**
	 * Add a public key for APT.
	 * <p>
	 * Downloads the key in binary format and passes it to gpg for dearmoring.
	 * </p>
	 *
	 * @param keyringName The name of the keyring file.
	 * @param url         URL pointing to the public key.
	 */
	public static void addPublicKey(String keyringName, URI url) {
		install("gpg");

		byte[] pubKeyBytes = Download.contentBytes(url);

		byte[] dearmoredPubkey = Shell.Command.user("gpg", "--dearmor")
				.stdin(new ByteArrayInputStream(pubKeyBytes))
				.binaryStdout(true)
				.run()
				.binaryStdout();

		FileUtils.sudoWrite(
				"/etc/apt/keyrings/%s".formatted(keyringName),
				dearmoredPubkey,
				"root:root",
				"644"
		);
	}

	public static void addSource(String sourceIdentifier, String sourceFileContent) {
		FileUtils.sudoWrite(
				"/etc/apt/sources.list.d/%s.list".formatted(sourceIdentifier), sourceFileContent,
				"root:root",
				"644"
		);
	}

	public static String getArchitecture() {
		return Shell.user("dpkg", "--print-architecture").trim();
	}

}
