package io.github.qsy7.shell.impl.service;

import io.github.qsy7.property.api.annotation.Property;
import io.github.qsy7.shell.api.model.*;
import io.github.qsy7.shell.api.service.ShellExecutionService;
import io.github.qsy7.shell.impl.ShellProcessExecution;
import io.github.qsy7.shell.impl.annotation.EntityEnabled;
import io.github.qsy7.shell.impl.property.InterruptGracePeriodUnits;
import io.github.qsy7.shell.impl.property.InterruptGracePeriodValue;
import io.github.qsy7.shell.impl.util.ShellExecutionUtil;
import java.io.IOException;
import java.time.temporal.ChronoUnit;
import jakarta.inject.Inject;

public class DefaultShellExecution implements ShellExecutionService, AutoCloseable {
  protected boolean shutdown;

  protected final ChronoUnit interruptGracePeriodUnits;
  protected final long interruptGracePeriodValue;

  @Inject
  public DefaultShellExecution(
      @Property(InterruptGracePeriodUnits.class) ChronoUnit interruptGracePeriodUnits,
      @Property(InterruptGracePeriodValue.class) long interruptGracePeriodValue) {

    this.interruptGracePeriodUnits = interruptGracePeriodUnits;
    this.interruptGracePeriodValue = interruptGracePeriodValue;
  }

  @EntityEnabled
  @Override
  public ShellCommand run(ShellCommand shellCommand) throws Exception {
    checkIfShutdown();

    doRun(shellCommand);
    return shellCommand;
  }

  protected void checkIfShutdown() {
    if (shutdown) throw new IllegalStateException("Service is shutting down");
  }

  protected void doRun(ShellCommand shellCommand) throws IOException, InterruptedException {
    new ShellProcessExecution(
            setupProcess(shellCommand),
            shellCommand,
            interruptGracePeriodUnits,
            interruptGracePeriodValue)
        .run();
  }

  protected Process setupProcess(ShellCommand shellCommand) throws IOException {
    if (shellCommand instanceof Chrootable) {
      if (shellCommand instanceof FreeBSDJailShellCommand)
        return doFreeBSDJailChrootProcess(shellCommand);

      return (doChrootProcess(shellCommand));
    }
    return (ShellExecutionUtil.doNonChrootProcess(shellCommand));
  }

  protected Process doFreeBSDJailChrootProcess(ShellCommand shellCommand) throws IOException {
    final Chrootable chrootable = (Chrootable) shellCommand;
    doValidate(shellCommand, chrootable);

    if (shellCommand instanceof EnvironmentAware) {
      final ProcessBuilder processBuilder =
          new ProcessBuilder(ShellExecutionUtil.getProcessBuilderChrootCmdLine(chrootable));
      setEnvironment(
          processBuilder, (EnvironmentAware) shellCommand.getShellCommandEnvironmentProperties());
      return processBuilder.start();
    }
    return (Runtime.getRuntime().exec(ShellExecutionUtil.getChrootCmdLine(chrootable)));

    // once chroot process is running, execute actual command ...
  }

  private static void setEnvironment(
      final ProcessBuilder processBuilder, final EnvironmentAware environmentAware) {
    for (final ShellCommandEnvironmentProperty shellCommandEnvironmentProperty :
        environmentAware.getShellCommandEnvironmentProperties()) {
      setEnvironmentProperty(processBuilder, shellCommandEnvironmentProperty);
    }
  }

  private static void setEnvironmentProperty(
      final ProcessBuilder processBuilder,
      final ShellCommandEnvironmentProperty shellCommandEnvironmentProperty) {
    processBuilder
        .environment()
        .put(shellCommandEnvironmentProperty.getKey(), shellCommandEnvironmentProperty.getValue());
  }

  protected Process doChrootProcess(ShellCommand shellCommand) throws IOException {
    final Chrootable chrootable = (Chrootable) shellCommand;
    doValidate(shellCommand, chrootable);

    if (shellCommand instanceof EnvironmentAware) {
      if (shellCommand instanceof EnvironmentAware) {
        final ProcessBuilder processBuilder =
            new ProcessBuilder(ShellExecutionUtil.getProcessBuilderChrootCmdLine(chrootable));
        setEnvironment(processBuilder, shellCommand);
        return processBuilder.start();
      }
    }
    return (Runtime.getRuntime().exec(ShellExecutionUtil.getChrootCmdLine(chrootable)));

    // once chroot process is running, execute actual command ...
  }

  protected void doValidate(final ShellCommand shellCommand, final Chrootable chrootable) {
    if (chrootable.getChrootPath() == null || chrootable.getChrootPath().isEmpty())
      throw new IllegalStateException(
          "Cannot chroot, chroot path is null:" + shellCommand.getCommandLine());
  }

  @Override
  public void close() {
    shutdown = true;
  }
}
