package io.github.qsy7.shell.impl.service;

import io.github.qsy7.property.api.annotation.Property;
import io.github.qsy7.shell.api.enumeration.BatteryState;
import io.github.qsy7.shell.api.model.BatteryRequest;
import io.github.qsy7.shell.api.model.BatteryStatus;
import io.github.qsy7.shell.api.model.Node;
import io.github.qsy7.shell.api.service.ShellExecutionService;
import io.github.qsy7.shell.api.service.UpowerService;
import io.github.qsy7.shell.impl.property.UpowerTimeout;
import java.time.LocalDateTime;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jakarta.inject.Inject;

public class DefaultUpowerService extends AbstractSingleShellCommandService<BatteryRequest>
    implements UpowerService {
  protected final Node node;

  @Inject
  public DefaultUpowerService(
      ShellCommandBuilder shellCommandBuilder,
      ShellExecutionService shellExecutionService,
      Node node,
      @Property(UpowerTimeout.class) int timeout) {
    super(shellCommandBuilder, shellExecutionService, timeout);
    this.node = node;
  }

  protected String getCommandLine() {
    return "upower -i /org/freedesktop/UPower/devices/battery_BAT0";
  }

  public void doAfter(BatteryRequest batteryRequest) {
    final Pattern statePattern = Pattern.compile(".*(fully-charged|charging|discharging).*");
    final Pattern percentagePattern = Pattern.compile(".*percentage:[\\W]{1,}([\\d]{1,3})%.*");

    String state = null;
    int percentage = -1;
    for (String line :
        batteryRequest.getShellCommand().getOutputs().get(0).getOutput().split("\n")) {
      if (state != null && percentage > 0) {
        break;
      }

      final Matcher stateMatcher = statePattern.matcher(line);

      if (stateMatcher.matches()) {
        state = stateMatcher.group(1);
        continue;
      }

      final Matcher percentageMatcher = percentagePattern.matcher(line);
      if (percentageMatcher.matches()) {
        percentage = Integer.valueOf(percentageMatcher.group(1));
        continue;
      }
    }

    batteryRequest.setBatteryStatus(
        new BatteryStatus(
            node,
            LocalDateTime.now(),
            BatteryState.getFromUpowerString(state),
            Integer.valueOf(percentage),
            batteryRequest.getShellCommand()));
  }
}
