package com.xmfuncoding.fundialog

import android.graphics.Color
import android.graphics.drawable.ColorDrawable
import android.os.Bundle
import android.view.Gravity
import android.view.LayoutInflater
import android.view.View
import android.view.View.OnClickListener
import android.view.ViewGroup
import androidx.annotation.StringRes
import androidx.core.view.isVisible
import androidx.fragment.app.DialogFragment
import androidx.fragment.app.FragmentManager
import androidx.fragment.app.FragmentTransaction
import com.xmfuncoding.fundialog.FunConstants.CANCELABLE_KEY
import com.xmfuncoding.fundialog.FunConstants.CANCEL_TEXT_KEY
import com.xmfuncoding.fundialog.FunConstants.CANCEL_TEXT_RES_KEY
import com.xmfuncoding.fundialog.FunConstants.CONFIRM_TEXT_KEY
import com.xmfuncoding.fundialog.FunConstants.CONFIRM_TEXT_RES_KEY
import com.xmfuncoding.fundialog.FunConstants.MESSAGE_KEY
import com.xmfuncoding.fundialog.FunConstants.MESSAGE_RES_KEY
import com.xmfuncoding.fundialog.FunConstants.NO_TEXT
import com.xmfuncoding.fundialog.FunConstants.TITLE_KEY
import com.xmfuncoding.fundialog.FunConstants.TITLE_RES_KEY
import com.xmfuncoding.fundialog.databinding.FunDialogUpdateBinding

/**
 *  Author:      XmFunCoding
 *  Email:       xmfuncoding@163.com
 *  Date:        2022-10-13 17:02
 *  Description: 版本更新对话框
 */
class FunUpdateDialog : DialogFragment() {

    companion object {
        fun newInstance(
            // title、titleRes 二选一
            title: String? = "",
            @StringRes titleRes: Int = NO_TEXT,

            // message、messageRes 二选一
            message: String? = "",
            @StringRes messageRes: Int = NO_TEXT,

            // confirmText、confirmTextRes 二选一
            confirmText: String? = "",
            @StringRes confirmTextRes: Int = NO_TEXT,

            // cancelText、cancelTextRes 二选一
            cancelText: String? = "",
            @StringRes cancelTextRes: Int = NO_TEXT,

            cancelable: Boolean = true,
        ): FunUpdateDialog {
            val fragment = FunUpdateDialog()
            val bundle = Bundle()
            bundle.putString(TITLE_KEY, title)
            bundle.putInt(TITLE_RES_KEY, titleRes)

            bundle.putString(MESSAGE_KEY, message)
            bundle.putInt(MESSAGE_RES_KEY, messageRes)

            bundle.putString(CONFIRM_TEXT_KEY, confirmText)
            bundle.putInt(CONFIRM_TEXT_RES_KEY, confirmTextRes)

            bundle.putString(CANCEL_TEXT_KEY, cancelText)
            bundle.putInt(CANCEL_TEXT_RES_KEY, cancelTextRes)

            bundle.putBoolean(CANCELABLE_KEY, cancelable)
            fragment.arguments = bundle
            return fragment
        }
    }

    private lateinit var binding: FunDialogUpdateBinding

    private var mTitle: String? = null
    private var mMessage: String? = null
    private var mConfirmText: String? = null
    private var mCancelText: String? = null
    private var mCancelable = false

    private var mOnConfirmClickListener: OnClickListener? = null
    private var mOnCancelClickListener: OnClickListener? = null

    override fun onCreateView(
        inflater: LayoutInflater,
        container: ViewGroup?,
        savedInstanceState: Bundle?,
    ): View {
        binding = FunDialogUpdateBinding.inflate(layoutInflater)
        return binding.root
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)

        mTitle = arguments?.getString(TITLE_KEY)
        mMessage = arguments?.getString(MESSAGE_KEY)
        mConfirmText = arguments?.getString(CONFIRM_TEXT_KEY)
        mCancelText = arguments?.getString(CANCEL_TEXT_KEY)

        if (mTitle.isNullOrEmpty()) {
            mTitle = arguments?.getInt(TITLE_RES_KEY)?.let {
                if (it != NO_TEXT) {
                    context?.getString(it)
                } else {
                    ""
                }
            }
        }
        if (mMessage.isNullOrEmpty()) {
            mMessage = arguments?.getInt(MESSAGE_RES_KEY)?.let {
                if (it != NO_TEXT) {
                    context?.getString(it)
                } else {
                    ""
                }
            }
        }
        if (mConfirmText.isNullOrEmpty()) {
            mConfirmText = arguments?.getInt(CONFIRM_TEXT_RES_KEY)?.let {
                if (it != NO_TEXT) {
                    context?.getString(it)
                } else {
                    ""
                }
            }
        }
        if (mCancelText.isNullOrEmpty()) {
            mCancelText = arguments?.getInt(CANCEL_TEXT_RES_KEY)?.let {
                if (it != NO_TEXT) {
                    context?.getString(it)
                } else {
                    ""
                }
            }
        }

        mCancelable = arguments?.getBoolean(CANCELABLE_KEY) ?: true

        isCancelable = mCancelable

        // 标题
        if (mTitle.isNullOrEmpty()) {
            binding.tvTitle.isVisible = false
        } else {
            binding.tvTitle.isVisible = true
            binding.tvTitle.text = mTitle
        }

        // 内容
        binding.tvMessage.text = mMessage

        binding.btnCancel.isVisible = !mCancelText.isNullOrEmpty()
        binding.viewDivider.isVisible = !mCancelText.isNullOrEmpty()

        binding.btnCancel.text = mCancelText
        binding.btnConfirm.text = mConfirmText

        if (!mCancelText.isNullOrEmpty()) {
            binding.btnConfirm.setBackgroundResource(R.drawable.fun_dialog_bg_button_ripple_bottom_right_10)
        } else {
            binding.btnConfirm.setBackgroundResource(R.drawable.fun_dialog_bg_button_ripple_bottom_10)
        }

        binding.btnCancel.setOnClickListener {
            mOnCancelClickListener?.onClick(it)
            dismiss()
        }

        binding.btnConfirm.setOnClickListener {
            mOnConfirmClickListener?.onClick(it)
//            dismiss() // 强制更新时，点击按钮时不能关闭对话框，所以交由主工程控制
        }
    }

    fun updateConfirmText(text: String = "", textRes: Int = NO_TEXT) {
        if (text.isNotEmpty()) {
            mConfirmText = text
        } else if (textRes != NO_TEXT) {
            mConfirmText = context?.getString(textRes)
        }

        binding.btnConfirm.text = mConfirmText
    }

    override fun onStart() {
        /*
        因为View在添加后,对话框最外层的ViewGroup并不知道我们导入的View所需要的的宽度。 所以我们需要在onStart生命周期里修改对话框尺寸参数
        */
        super.onStart()

        val width = resources.displayMetrics.widthPixels
        val height = resources.displayMetrics.heightPixels

        val window = requireDialog().window
        val attributes = window?.attributes

        if (width < height) {
            // 竖屏
            attributes?.width = (width * 1.0f).toInt()
            attributes?.height = (height * 0.8f).toInt()
        } else {
            // 横屏
            attributes?.width = dp2px(360F).toInt()
            attributes?.height = (height * 0.9f).toInt()
        }

        attributes?.gravity = Gravity.CENTER
        window?.attributes = attributes
        window?.decorView?.background = ColorDrawable(Color.TRANSPARENT)
    }

    fun setOnConfirmClickListener(onConfirmClickListener: View.OnClickListener?) {
        mOnConfirmClickListener = onConfirmClickListener
    }

    fun setOnCancelClickListener(onCancelClickListener: View.OnClickListener?) {
        mOnCancelClickListener = onCancelClickListener
    }

    fun show(manager: FragmentManager) {
        show(manager, "dialog")
    }

    override fun show(manager: FragmentManager, tag: String?) {
        try {
            val ft: FragmentTransaction = manager.beginTransaction()
            ft.add(this, tag)
            ft.commitAllowingStateLoss()
        } catch (e: IllegalStateException) {
            e.printStackTrace()
        }
    }

    fun isShowing(): Boolean {
        return dialog?.isShowing ?: false
    }

    /**
     * Value of dp to value of px.
     *
     * @param dpValue The value of dp.
     * @return value of px
     */
    fun dp2px(dpValue: Float): Float {
        val scale = resources.displayMetrics.density
        return dpValue * scale + 0.5f
    }
}