/*
 * #%L
 * Image processing operations for SciJava Ops.
 * %%
 * Copyright (C) 2014 - 2024 SciJava developers.
 * %%
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * #L%
 */

package org.scijava.ops.image.geom;

import net.imglib2.type.numeric.real.DoubleType;
import org.scijava.function.Computers;
import org.scijava.ops.spi.OpDependency;

import java.util.function.Function;

/**
 * Abstract base for {@code geom.boxivity} operations. Based on
 * http://www.math.uci.edu/icamp/summer/research_11/park/shape_descriptors_survey.pdf,
 * where boxivity is called rectangularity.
 *
 * @author Tim-Oliver Buchholz (University of Konstanz)
 */
public abstract class AbstractBoxivity<I> implements
	Computers.Arity1<I, DoubleType>
{

	@OpDependency(name = "geom.size")
	private Function<I, DoubleType> areaFunc;

	@OpDependency(name = "geom.smallestEnclosingBoundingBox")
	private Function<I, I> smallestEnclosingRectangleFunc;

	/**
	 * TODO
	 *
	 * @param input
	 * @param output
	 */
	@Override
	public void compute(I input, DoubleType output) {
		output.set(areaFunc.apply(input).getRealDouble() / areaFunc.apply(
			smallestEnclosingRectangleFunc.apply(input)).getRealDouble());
	}
}
