/*
 * #%L
 * Image processing operations for SciJava Ops.
 * %%
 * Copyright (C) 2014 - 2024 SciJava developers.
 * %%
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDERS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * #L%
 */

package org.scijava.ops.image.geom.geom2d;

import java.util.function.Function;

import net.imglib2.roi.geom.real.Polygon2D;
import net.imglib2.type.numeric.real.DoubleType;

import org.scijava.function.Computers;
import org.scijava.ops.spi.OpDependency;

/**
 * Generic implementation of {@code geom.eccentricity}. The eccentricity formula
 * is based on "The Eccentricity of a Conic Section" by Ayoub B. Ayoub.
 *
 * @author Tim-Oliver Buchholz, University of Konstanz
 * @implNote op names='geom.eccentricity', label='Geometric (2D): Eccentricity'
 */
public class DefaultEccentricity implements
	Computers.Arity1<Polygon2D, DoubleType>
{

	@OpDependency(name = "geom.minorAxis")
	private Function<Polygon2D, DoubleType> minorAxisFunc;
	@OpDependency(name = "geom.majorAxis")
	private Function<Polygon2D, DoubleType> majorAxisFunc;

	/**
	 * TODO
	 *
	 * @param input
	 * @param eccentricity
	 */
	@Override
	public void compute(final Polygon2D input, final DoubleType eccentricity) {

		final double a = majorAxisFunc.apply(input).get() / 2.0;
		final double b = minorAxisFunc.apply(input).get() / 2.0;

		eccentricity.set(Math.sqrt(1 - Math.pow(b / a, 2)));
	}

}
