/*
VisAD system for interactive analysis and visualization of numerical
data.  Copyright (C) 1996 - 2018 Bill Hibbard, Curtis Rueden, Tom
Rink, Dave Glowacki, Steve Emmerson, Tom Whittaker, Don Murray, and
Tommy Jasmin.

This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Library General Public
License as published by the Free Software Foundation; either
version 2 of the License, or (at your option) any later version.

This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Library General Public License for more details.

You should have received a copy of the GNU Library General Public
License along with this library; if not, write to the Free
Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
MA 02111-1307, USA
*/

/*
 * TextControlWidget.java
 *
 * Created on July 14, 2003, 3:15 PM
 */

package visad.util;

import java.awt.*;
import java.util.Vector;
import java.rmi.*;
import java.io.*;
import java.net.*;
import java.util.zip.*;
import java.util.Enumeration;

import visad.*;
import visad.util.HersheyFont;
import visad.java2d.DisplayImplJ2D;


/**
 *
 * @author  Sylvain Letourneau (Sylvain.letourneau@nrc.ca)
 * This is a VisAD widget for interactive settings of text controls.
 *
 */
public class TextControlWidget extends javax.swing.JPanel {
  private String[] javaFontNames = null;
  private String[] hersheyFontNames = null;
  private String[] visadFontNames = null;
  private String textToPreview = "VisAD";
  //private String textToPreview = "Mf.abci-_='bp";
  private TextControl previewDisplayTC = null;
  private TextControl textControl = null;

  private DisplayImpl display = null;
  private ScalarMap textMap = null;
  private FieldImpl textField = null;
  private TextType textType = null;

  /** Creates new form TextControlWidget */
  public TextControlWidget(TextControl aTextControl)
    throws RemoteException, VisADException {
    initComponents();

    javaFontNames = getJavaFontNames();
    hersheyFontNames = getHersheyFontNames();
    visadFontNames = new String[]{"VisAD Default"};
    fontNamesJList.setListData(javaFontNames);
    fontNamesJList.setSelectedIndex(0);

    String[] sizes = { "8", "9", "10", "11", "12", "14", "16", "18", "24",
                       "36", "48", "72"};
    fontSizesJList.setListData(sizes);
    fontSizesJList.setSelectedIndex(7);

    String[] styles = {
      "Plain",
      "Bold",
      "Italic",
      "BoldItalic"
    };
    fontStylesJList.setListData(styles);
    fontStylesJList.setSelectedIndex(0);

    textToPreviewField.setText(textToPreview);

    // Save initial controls
    textControl = aTextControl;
    if (textControl == null) {
      textControl = new TextControl(null);
    }
    setGUIControls(textControl);

    setupPreviewDisplay();
    if (display == null) {
      System.out.println("Display is null!");
    }

    previewDisplayTC = (TextControl) textMap.getControl();
    updateTextControl();
    displayTextToPreview.add(display.getComponent());
  }

  /** This method is called from within the constructor to
   * initialize the form.
   * WARNING: Do NOT modify this code. The content of this method is
   * always regenerated by the Form Editor.
   */
  private void initComponents() {//GEN-BEGIN:initComponents
    java.awt.GridBagConstraints gridBagConstraints;

    fontTypeButtonGroup = new javax.swing.ButtonGroup();
    fontTypeJPanel = new javax.swing.JPanel();
    javaFontJRadioButton = new javax.swing.JRadioButton();
    hersheyFontjRadioButton = new javax.swing.JRadioButton();
    labelFontJRadioButton = new javax.swing.JRadioButton();
    fontSelectionJPanel = new javax.swing.JPanel();
    jLabel1 = new javax.swing.JLabel();
    jLabel2 = new javax.swing.JLabel();
    jLabel3 = new javax.swing.JLabel();
    jScrollPane1 = new javax.swing.JScrollPane();
    fontNamesJList = new javax.swing.JList();
    jScrollPane2 = new javax.swing.JScrollPane();
    fontStylesJList = new javax.swing.JList();
    jScrollPane3 = new javax.swing.JScrollPane();
    fontSizesJList = new javax.swing.JList();
    previewPanel = new javax.swing.JPanel();
    textToPreviewField = new javax.swing.JTextField();
    displayTextToPreview = new javax.swing.JPanel();
    jScrollPane4 = new javax.swing.JScrollPane();
    textPanelAttributes = new javax.swing.JPanel();
    leftPanel = new javax.swing.JPanel();
    jLabel4 = new javax.swing.JLabel();
    textOrientationField = new javax.swing.JTextField();
    jLabel5 = new javax.swing.JLabel();
    characterRotationField = new javax.swing.JTextField();
    jLabel8 = new javax.swing.JLabel();
    scaleTextField = new javax.swing.JTextField();
    autoSizeJCheckBox = new javax.swing.JCheckBox();
    rightPanel = new javax.swing.JPanel();
    horizJustificationJComboBox = new javax.swing.JComboBox();
    jLabel7 = new javax.swing.JLabel();
    jLabel6 = new javax.swing.JLabel();
    vertJustificationJComboBox = new javax.swing.JComboBox();
    bottomPanel = new javax.swing.JPanel();
    jLabel9 = new javax.swing.JLabel();
    offsetXField = new javax.swing.JTextField();
    offsetYField = new javax.swing.JTextField();
    offsetZField = new javax.swing.JTextField();

    setLayout(new java.awt.GridBagLayout());

    fontTypeJPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.LEFT));

    javaFontJRadioButton.setSelected(true);
    javaFontJRadioButton.setText("Java font");
    fontTypeButtonGroup.add(javaFontJRadioButton);
    javaFontJRadioButton.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          javaFontJRadioButtonActionPerformed(evt);
        }
      });

    fontTypeJPanel.add(javaFontJRadioButton);

    hersheyFontjRadioButton.setText("Hershery font");
    fontTypeButtonGroup.add(hersheyFontjRadioButton);
    hersheyFontjRadioButton.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          hersheyFontjRadioButtonActionPerformed(evt);
        }
      });

    fontTypeJPanel.add(hersheyFontjRadioButton);

    labelFontJRadioButton.setText("VisAD font");
    fontTypeButtonGroup.add(labelFontJRadioButton);
    labelFontJRadioButton.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          labelFontJRadioButtonActionPerformed(evt);
        }
      });

    fontTypeJPanel.add(labelFontJRadioButton);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridwidth = 3;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    add(fontTypeJPanel, gridBagConstraints);

    fontSelectionJPanel.setLayout(new java.awt.GridBagLayout());

    fontSelectionJPanel.setMinimumSize(new java.awt.Dimension(200, 70));
    fontSelectionJPanel.setPreferredSize(new java.awt.Dimension(360, 100));
    jLabel1.setText("Font Name");
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    fontSelectionJPanel.add(jLabel1, gridBagConstraints);

    jLabel2.setText("Style");
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    fontSelectionJPanel.add(jLabel2, gridBagConstraints);

    jLabel3.setText("Size");
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
    fontSelectionJPanel.add(jLabel3, gridBagConstraints);

    jScrollPane1.setMinimumSize(new java.awt.Dimension(160, 40));
    jScrollPane1.setPreferredSize(new java.awt.Dimension(160, 40));
    fontNamesJList.setBorder(new javax.swing.border.CompoundBorder());
    fontNamesJList.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
    fontNamesJList.setSelectedIndex(1);
    fontNamesJList.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
        public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
          fontNamesJListValueChanged(evt);
        }
      });

    jScrollPane1.setViewportView(fontNamesJList);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.weighty = 0.5;
    fontSelectionJPanel.add(jScrollPane1, gridBagConstraints);

    jScrollPane2.setMaximumSize(new java.awt.Dimension(0, 0));
    jScrollPane2.setMinimumSize(new java.awt.Dimension(90, 40));
    jScrollPane2.setPreferredSize(new java.awt.Dimension(100, 40));
    fontStylesJList.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
    fontStylesJList.setSelectedIndex(0);
    fontStylesJList.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
        public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
          fontStylesJListValueChanged(evt);
        }
      });

    jScrollPane2.setViewportView(fontStylesJList);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weighty = 0.5;
    fontSelectionJPanel.add(jScrollPane2, gridBagConstraints);

    jScrollPane3.setMinimumSize(new java.awt.Dimension(100, 40));
    jScrollPane3.setPreferredSize(new java.awt.Dimension(60, 40));
    fontSizesJList.setSelectionMode(javax.swing.ListSelectionModel.SINGLE_SELECTION);
    fontSizesJList.setSelectedIndex(3);
    fontSizesJList.addListSelectionListener(new javax.swing.event.ListSelectionListener() {
        public void valueChanged(javax.swing.event.ListSelectionEvent evt) {
          fontSizesJListValueChanged(evt);
        }
      });

    jScrollPane3.setViewportView(fontSizesJList);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weighty = 0.5;
    fontSelectionJPanel.add(jScrollPane3, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.weighty = 0.3;
    gridBagConstraints.insets = new java.awt.Insets(7, 7, 7, 7);
    add(fontSelectionJPanel, gridBagConstraints);

    previewPanel.setLayout(new java.awt.GridBagLayout());

    previewPanel.setBorder(new javax.swing.border.TitledBorder("Preview"));
    previewPanel.setMinimumSize(new java.awt.Dimension(56, 192));
    textToPreviewField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          textToPreviewFieldActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints.weightx = 0.5;
    previewPanel.add(textToPreviewField, gridBagConstraints);

    displayTextToPreview.setLayout(new java.awt.BorderLayout());

    displayTextToPreview.setBorder(new javax.swing.border.EtchedBorder());
    displayTextToPreview.setPreferredSize(new java.awt.Dimension(300, 300));
    displayTextToPreview.add(jScrollPane4, java.awt.BorderLayout.SOUTH);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.weighty = 0.9;
    gridBagConstraints.insets = new java.awt.Insets(10, 10, 10, 10);
    previewPanel.add(displayTextToPreview, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 3;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.weighty = 0.8;
    gridBagConstraints.insets = new java.awt.Insets(7, 7, 7, 7);
    add(previewPanel, gridBagConstraints);

    textPanelAttributes.setLayout(new java.awt.GridBagLayout());

    leftPanel.setLayout(new java.awt.GridBagLayout());

    jLabel4.setHorizontalAlignment(javax.swing.SwingConstants.LEFT);
    jLabel4.setText("Text Orientation:");
    jLabel4.setPreferredSize(new java.awt.Dimension(120, 16));
    jLabel4.setHorizontalTextPosition(javax.swing.SwingConstants.LEFT);
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(jLabel4, gridBagConstraints);

    textOrientationField.setText("0");
    textOrientationField.setPreferredSize(new java.awt.Dimension(50, 20));
    textOrientationField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          textOrientationFieldActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(textOrientationField, gridBagConstraints);

    jLabel5.setText("Character Rotation:");
    jLabel5.setPreferredSize(new java.awt.Dimension(40, 16));
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(jLabel5, gridBagConstraints);

    characterRotationField.setText("0");
    characterRotationField.setPreferredSize(new java.awt.Dimension(50, 20));
    characterRotationField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          characterRotationFieldActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(characterRotationField, gridBagConstraints);

    jLabel8.setText("Scale:");
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(jLabel8, gridBagConstraints);

    scaleTextField.setText("1.0");
    scaleTextField.setPreferredSize(new java.awt.Dimension(50, 20));
    scaleTextField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          scaleTextFieldActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(scaleTextField, gridBagConstraints);

    autoSizeJCheckBox.setText("auto size");
    autoSizeJCheckBox.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          autoSizeJCheckBoxActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 3;
    gridBagConstraints.gridwidth = 2;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    leftPanel.add(autoSizeJCheckBox, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    textPanelAttributes.add(leftPanel, gridBagConstraints);

    rightPanel.setLayout(new java.awt.GridBagLayout());

    rightPanel.setMinimumSize(new java.awt.Dimension(144, 80));
    rightPanel.setPreferredSize(new java.awt.Dimension(220, 56));
    horizJustificationJComboBox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Left", "Center", "Right" }));
    horizJustificationJComboBox.setPreferredSize(new java.awt.Dimension(90, 25));
    horizJustificationJComboBox.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          horizJustificationJComboBoxActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.insets = new java.awt.Insets(3, 5, 3, 3);
    rightPanel.add(horizJustificationJComboBox, gridBagConstraints);

    jLabel7.setText("Horizontal justification:");
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    rightPanel.add(jLabel7, gridBagConstraints);

    jLabel6.setText("Vertical justification:");
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 2;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    rightPanel.add(jLabel6, gridBagConstraints);

    vertJustificationJComboBox.setModel(new javax.swing.DefaultComboBoxModel(new String[] { "Bottom", "Center", "Top" }));
    vertJustificationJComboBox.setPreferredSize(new java.awt.Dimension(90, 25));
    vertJustificationJComboBox.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          vertJustificationJComboBoxActionPerformed(evt);
        }
      });

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 2;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.insets = new java.awt.Insets(3, 5, 3, 3);
    rightPanel.add(vertJustificationJComboBox, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    textPanelAttributes.add(rightPanel, gridBagConstraints);

    bottomPanel.setLayout(new java.awt.GridBagLayout());

    jLabel9.setText("Offsets ([x, y, z]):");
    bottomPanel.add(jLabel9, new java.awt.GridBagConstraints());

    offsetXField.setText("0");
    offsetXField.setPreferredSize(new java.awt.Dimension(50, 20));
    offsetXField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          offsetXFieldActionPerformed(evt);
        }
      });

    bottomPanel.add(offsetXField, new java.awt.GridBagConstraints());

    offsetYField.setText("0");
    offsetYField.setPreferredSize(new java.awt.Dimension(50, 20));
    offsetYField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          offsetYFieldActionPerformed(evt);
        }
      });

    bottomPanel.add(offsetYField, new java.awt.GridBagConstraints());

    offsetZField.setText("0");
    offsetZField.setPreferredSize(new java.awt.Dimension(50, 20));
    offsetZField.addActionListener(new java.awt.event.ActionListener() {
        public void actionPerformed(java.awt.event.ActionEvent evt) {
          offsetZFieldActionPerformed(evt);
        }
      });

    bottomPanel.add(offsetZField, new java.awt.GridBagConstraints());

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.gridwidth = 2;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.SOUTH;
    gridBagConstraints.weightx = 2.0;
    gridBagConstraints.insets = new java.awt.Insets(3, 3, 3, 3);
    textPanelAttributes.add(bottomPanel, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 0;
    gridBagConstraints.gridy = 2;
    gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
    gridBagConstraints.weightx = 0.5;
    gridBagConstraints.insets = new java.awt.Insets(7, 7, 7, 7);
    add(textPanelAttributes, gridBagConstraints);

  }//GEN-END:initComponents


  private void vertJustificationJComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_vertJustificationJComboBoxActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_vertJustificationJComboBoxActionPerformed

  private void autoSizeJCheckBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_autoSizeJCheckBoxActionPerformed
    updateTextControl();
  }//GEN-LAST:event_autoSizeJCheckBoxActionPerformed

  private void labelFontJRadioButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_labelFontJRadioButtonActionPerformed
    fontNamesJList.setListData(visadFontNames);
    fontNamesJList.setSelectedIndex(0);
    fontSelectionJPanel.remove(jLabel2);
    fontSelectionJPanel.remove(jScrollPane2);
    fontSelectionJPanel.updateUI();
  }//GEN-LAST:event_labelFontJRadioButtonActionPerformed

  private void offsetZFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_offsetZFieldActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_offsetZFieldActionPerformed

  private void offsetYFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_offsetYFieldActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_offsetYFieldActionPerformed

  private void offsetXFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_offsetXFieldActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_offsetXFieldActionPerformed

  private void javaFontJRadioButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_javaFontJRadioButtonActionPerformed
    fontNamesJList.setListData(javaFontNames);
    fontNamesJList.setSelectedIndex(0);
    jLabel1.setVisible(true);
    jLabel2.setVisible(true);
    jLabel3.setVisible(true);

    java.awt.GridBagConstraints gridBagConstraints;
    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTHWEST;
    fontSelectionJPanel.add(jLabel2, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridy = 0;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.anchor = java.awt.GridBagConstraints.NORTH;
    fontSelectionJPanel.add(jLabel3, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 1;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weighty = 0.5;
    fontSelectionJPanel.add(jScrollPane2, gridBagConstraints);

    gridBagConstraints = new java.awt.GridBagConstraints();
    gridBagConstraints.gridx = 2;
    gridBagConstraints.gridy = 1;
    gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
    gridBagConstraints.weighty = 0.5;
    fontSelectionJPanel.add(jScrollPane3, gridBagConstraints);

    fontSelectionJPanel.updateUI();

  }//GEN-LAST:event_javaFontJRadioButtonActionPerformed

  private void textToPreviewFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_textToPreviewFieldActionPerformed
    // Add your handling code here:
    textToPreview = textToPreviewField.getText();
    try {
      Data datum = new Text(textType, textToPreview);
      textField.setSample(0, datum);
    } catch (RemoteException e) {
      System.out.println("Exception: " + e);
    } catch (VisADException e1) {
      System.out.println("Exception: " + e1);
    }
  }//GEN-LAST:event_textToPreviewFieldActionPerformed

  private void scaleTextFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_scaleTextFieldActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_scaleTextFieldActionPerformed

  private void characterRotationFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_characterRotationFieldActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_characterRotationFieldActionPerformed

  private void horizJustificationJComboBoxActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_horizJustificationJComboBoxActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_horizJustificationJComboBoxActionPerformed

  private void textOrientationFieldActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_textOrientationFieldActionPerformed
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_textOrientationFieldActionPerformed

  private void fontSizesJListValueChanged(javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_fontSizesJListValueChanged
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_fontSizesJListValueChanged

  private void fontStylesJListValueChanged(javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_fontStylesJListValueChanged
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_fontStylesJListValueChanged

  private void fontNamesJListValueChanged(javax.swing.event.ListSelectionEvent evt) {//GEN-FIRST:event_fontNamesJListValueChanged
    // Add your handling code here:
    updateTextControl();
  }//GEN-LAST:event_fontNamesJListValueChanged

  private void hersheyFontjRadioButtonActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_hersheyFontjRadioButtonActionPerformed
    // Add your handling code here:
    fontNamesJList.setListData(hersheyFontNames);
    fontNamesJList.setSelectedIndex(0);

    fontSelectionJPanel.remove(jLabel2);
    fontSelectionJPanel.remove(jScrollPane2);
    fontSelectionJPanel.updateUI();
  }//GEN-LAST:event_hersheyFontjRadioButtonActionPerformed

  private Object getSelectedFont() {
    int selectedIndex = fontNamesJList.getSelectedIndex();
    if (selectedIndex == -1) {
      selectedIndex = 0;
      fontNamesJList.setSelectedIndex(selectedIndex);
    }

    if (javaFontJRadioButton.isSelected()) {
      int size;
      try {
        size = Integer.parseInt((String)fontSizesJList.getSelectedValue());
      }
      catch(Exception e)  {
        size = 12;
      }
      return new Font((String)fontNamesJList.getSelectedValue(),
                      fontStylesJList.getSelectedIndex(),size);
    }
    else if (hersheyFontjRadioButton.isSelected()) {
      return new HersheyFont ((String)fontNamesJList.getSelectedValue());
    }
    else {
      // VisAD font
      return null;
    }
  }

  private String[] getJavaFontNames() {
    String[] nameArray = GraphicsEnvironment
      .getLocalGraphicsEnvironment()
      .getAvailableFontFamilyNames();
    Vector nameVector = new Vector(nameArray.length);

    /*    for(int i = 0, j; i < nameArray.length; i++) {
          System.out.println(i + ".  " + nameArray[i]);
          }
    */

    String[] _array = new String[nameVector.size()];
    nameVector.copyInto(_array);
    return nameArray;
  }

  public static String[] getHersheyFontNames() {
    String[] fontNames = null;
    URL url = HersheyFont.class.
      getResource("futural.jhf");
    String protocol = url.getProtocol();
    //System.out.println(url.getFile());
    //System.out.println(url.getProtocol());
    if (protocol.equals("file")) {
      // We are running from the file system (i.e., visad has beeen extracted)
      String file;
      try {
        file = URLDecoder.decode(url.getFile(),"UTF-8");
      } catch (UnsupportedEncodingException uee) { // shouldn't happen
        file = url.getFile();
      }
      File f = new File(file);
      f = f.getParentFile();
      FilenameFilter filter = new FilenameFilter() {
          public boolean accept(File dir, String name) {
            return name.endsWith(".jhf");
          }
        };
      String[] children = f.list(filter);
      if (children == null) {
        return null;
      }
      fontNames = new String[children.length];
      for (int i=0; i<children.length; i++) {
        fontNames[i] =  children[i].substring(0, children[i].length()-4);
      }
    }
    else if (protocol.equals("jar")) {
      // We are running from a jar file
      try {
        JarURLConnection jarURL = (JarURLConnection)url.openConnection();
        //System.out.println(jarURL.getJarEntry());
        //System.out.println(jarURL.getJarFile().getName());
        //System.out.println(jarURL.getEntryName());
        ZipFile zf = new ZipFile(jarURL.getJarFile().getName());
        Enumeration e = zf.entries();
        Vector namesVector = new Vector();
        while (e.hasMoreElements()) {
          ZipEntry ze = (ZipEntry)e.nextElement();
          String name = ze.getName();
          if (name.startsWith("visad/util/") &&
              name.endsWith(".jhf")) {
            namesVector.add(name.substring(11));
          }
        }
        zf.close();
        if (namesVector.size() == 0) {
          return null;
        }
        fontNames = new String[namesVector.size()];
        for (int i=0; i<fontNames.length; i++) {
          String name = (String)namesVector.elementAt(i);
          fontNames[i] =  name.substring(0, name.length()-4);
        }
      }
      catch (Exception  e) {
        System.out.println(e);
      }
    }
    else {
      System.out.println("Cannot locate the HersheyFonts.");
    }
    return fontNames;
  }

   /**
    * Dumps a zip entry into a string.
    * @param ze a ZipEntry
    */
   private String dumpZipEntry(ZipEntry ze) {
       StringBuffer sb=new StringBuffer();
       if (ze.isDirectory()) {
          sb.append("d ");
       } else {
          sb.append("f ");
       }
       if (ze.getMethod()==ZipEntry.STORED) {
          sb.append("stored   ");
       } else {
          sb.append("defalted ");
       }
       sb.append(ze.getName());
       sb.append("\t");
       sb.append(""+ze.getSize());
       if (ze.getMethod()==ZipEntry.DEFLATED) {
          sb.append("/"+ze.getCompressedSize());
       }
       return (sb.toString());
   }


  private void setupPreviewDisplay()
    throws RemoteException, VisADException
  {
    display = new DisplayImplJ2D("Preview_Display_in_Text_Control_Widget");

    textType = TextType.getTextType("Preview_text_type_for_display_in_Text_Control_Widget");
    RealTupleType coord_tuple_type = new RealTupleType(new RealType[]
      {RealType.Latitude, RealType.Longitude});
    FunctionType text_function = new FunctionType(coord_tuple_type, textType);

    Linear2DSet domain = new Linear2DSet(coord_tuple_type, -120.0, 120.0, 1,
                                         -120.0, 120.0, 1);
    textField = new FieldImpl(text_function, domain);
    Data datum = new Text(textType, textToPreview);
    textField.setSample(0, datum);

    display.addMap(new ScalarMap(RealType.Latitude, Display.YAxis));
    display.addMap(new ScalarMap(RealType.Longitude, Display.XAxis));
    textMap = new ScalarMap(textType, Display.Text);
    display.addMap(textMap);

    DataReferenceImpl ref_text_field =
      new DataReferenceImpl("ref_text_field");
    ref_text_field.setData(textField);
    display.addReference(ref_text_field, null);

    // re-sizing the box around the display
    ProjectionControl pc=display.getProjectionControl();
    double[] pcMatrix=pc.getMatrix();
    double percent = .90;
    if (pcMatrix.length > 10) {
      pcMatrix[0]=percent;
      pcMatrix[5]=percent;
      pcMatrix[10]=percent;
    }
    else {
      pcMatrix[0]=percent/.64;
      pcMatrix[3]=-percent/.64;
      pc.setMatrix(pcMatrix);
    }

    // remove the box around the display
    //    display.getDisplayRenderer().setBoxOn(false);

  }

  private void setGUIControls(TextControl tc) {
    if (tc == null) return;

    String str = String.valueOf(tc.getScale());
    scaleTextField.setText(str);

    if (tc.getJustification() == TextControl.Justification.LEFT) {
      horizJustificationJComboBox.setSelectedItem("Left");
    }
    else if (tc.getJustification() == TextControl.Justification.RIGHT) {
      horizJustificationJComboBox.setSelectedItem("Right");
    }
    else {
      horizJustificationJComboBox.setSelectedItem("Center");
    }

    if (tc.getVerticalJustification() == TextControl.Justification.BOTTOM) {
      vertJustificationJComboBox.setSelectedItem("Bottom");
    }
    else if (tc.getVerticalJustification() == TextControl.Justification.TOP) {
      vertJustificationJComboBox.setSelectedItem("Top");
    }
    else {
      vertJustificationJComboBox.setSelectedItem("Center");
    }

    str = String.valueOf((int)(tc.getSize() * 12.0));
    fontSizesJList.setSelectedValue(str, true);
    Object font = tc.getFont();
    if (font instanceof java.awt.Font) {
      javaFontJRadioButton.setSelected(true);
      switch (((java.awt.Font) font).getStyle()) {
      case java.awt.Font.PLAIN:
        fontStylesJList.setSelectedIndex(0);
        break;
      case java.awt.Font.BOLD:
        fontStylesJList.setSelectedIndex(1);
        break;
      case java.awt.Font.ITALIC:
        fontStylesJList.setSelectedIndex(2);
        break;
      case java.awt.Font.BOLD+java.awt.Font.ITALIC:
        fontStylesJList.setSelectedIndex(3);
        break;
      default:
        fontStylesJList.setSelectedIndex(0);
        break;
      }
      javaFontJRadioButtonActionPerformed(null);
    }
    else if (font instanceof visad.util.HersheyFont) {
      hersheyFontjRadioButton.setSelected(true);
      hersheyFontjRadioButtonActionPerformed(null);
    }
    else {
      labelFontJRadioButton.setSelected(true);
      labelFontJRadioButtonActionPerformed(null);
    }

    str = String.valueOf(tc.getRotation());
    textOrientationField.setText(str);
    str = String.valueOf(tc.getCharacterRotation());
    characterRotationField.setText(str);

    double[] offsets = tc.getOffset();
    str = String.valueOf(offsets[0]);
    offsetXField.setText(str);
    str = String.valueOf(offsets[1]);
    offsetYField.setText(str);
    str = String.valueOf(offsets[2]);
    offsetZField.setText(str);

    if (tc.getAutoSize()) {
      autoSizeJCheckBox.setSelected(true);
    }
  }

  private void updateTextControl() {
    try {
      if (textMap == null) return;

      previewDisplayTC = (TextControl) textMap.getControl();
      if (previewDisplayTC != null) {
        String str = scaleTextField.getText();
        double val;
        try {
          val = Double.valueOf(str).doubleValue();
          if (val <= 0.0) {
            val = 1.0;
            scaleTextField.setText("1.0");
          }
        } catch (NumberFormatException e) {
          val = 1.0;
          scaleTextField.setText("1.0");
        }
        previewDisplayTC.setScale(val);
        textControl.setScale(val);

        String justStr = (String) horizJustificationJComboBox.getSelectedItem();
        if (justStr.equals("Left")) {
          previewDisplayTC.setJustification(TextControl.Justification.LEFT);
          textControl.setJustification(TextControl.Justification.LEFT);
        }
        else if (justStr.equals("Right")) {
          previewDisplayTC.setJustification(TextControl.Justification.RIGHT);
          textControl.setJustification(TextControl.Justification.RIGHT);
        }
        else {
          previewDisplayTC.setJustification(TextControl.Justification.CENTER);
          textControl.setJustification(TextControl.Justification.CENTER);
        }

        justStr = (String) vertJustificationJComboBox.getSelectedItem();
        if (justStr.equals("Bottom")) {
          previewDisplayTC.setVerticalJustification(TextControl.Justification.BOTTOM);
          textControl.setVerticalJustification(TextControl.Justification.BOTTOM);
        }
        else if (justStr.equals("Top")) {
          previewDisplayTC.setVerticalJustification(TextControl.Justification.TOP);
          textControl.setVerticalJustification(TextControl.Justification.TOP);
        }
        else {
          previewDisplayTC.setVerticalJustification(TextControl.Justification.CENTER);
          textControl.setVerticalJustification(TextControl.Justification.CENTER);
        }

        previewDisplayTC.setFont(getSelectedFont());
        textControl.setFont(getSelectedFont());
        // Set initial size
        double size;
        try {
          size = (double) Integer.parseInt((String)fontSizesJList.getSelectedValue());
        }
        catch(Exception e)  {
          size = 12.0;
        }
        previewDisplayTC.setSize(size/12.0);
        textControl.setSize(size/12.0);

        str = textOrientationField.getText();
        try {
          val = Double.valueOf(str).doubleValue();
        } catch (NumberFormatException e) {
          val = 0.0;
          textOrientationField.setText("0.0");
        }
        previewDisplayTC.setRotation(val);
        textControl.setRotation(val);
        str = characterRotationField.getText();
        try {
          val = Double.valueOf(str).doubleValue();
        } catch (NumberFormatException e) {
          val = 0.0;
          characterRotationField.setText("0.0");
        }
        previewDisplayTC.setCharacterRotation(val);
        textControl.setCharacterRotation(val);

        double offsetX = 0.0;
        str = offsetXField.getText();
        try {
          offsetX = Double.valueOf(str).doubleValue();
        } catch (NumberFormatException e) {
          offsetXField.setText("0.0");
        }
        double offsetY = 0.0;
        str = offsetYField.getText();
        try {
          offsetY = Double.valueOf(str).doubleValue();
        } catch (NumberFormatException e) {
          offsetYField.setText("0.0");
        }
        double offsetZ = 0.0;
        str = offsetZField.getText();
        try {
          offsetZ = Double.valueOf(str).doubleValue();
        } catch (NumberFormatException e) {
          offsetZField.setText("0.0");
        }
        double[] offsets = new double[]{offsetX, offsetY, offsetZ};
        previewDisplayTC.setOffset(offsets);
        textControl.setOffset(offsets);

        previewDisplayTC.setAutoSize(autoSizeJCheckBox.isSelected());
        textControl.setAutoSize(autoSizeJCheckBox.isSelected());

      } else {
        System.out.println("TextControl is null?");
      }

    } catch (RemoteException e) {
      System.out.println("Exception: " + e);
    } catch (VisADException e1) {
      System.out.println("Exception: " + e1);
    }
  }

  // Variables declaration - do not modify//GEN-BEGIN:variables
  private javax.swing.JLabel jLabel8;
  private javax.swing.JLabel jLabel4;
  private javax.swing.JComboBox vertJustificationJComboBox;
  private javax.swing.JTextField offsetYField;
  private javax.swing.JTextField offsetZField;
  private javax.swing.JScrollPane jScrollPane1;
  private javax.swing.JLabel jLabel1;
  private javax.swing.JLabel jLabel3;
  private javax.swing.JRadioButton labelFontJRadioButton;
  private javax.swing.JTextField textOrientationField;
  private javax.swing.JLabel jLabel2;
  private javax.swing.JScrollPane jScrollPane4;
  private javax.swing.JLabel jLabel9;
  private javax.swing.JScrollPane jScrollPane2;
  private javax.swing.JPanel fontTypeJPanel;
  private javax.swing.JRadioButton hersheyFontjRadioButton;
  private javax.swing.JPanel displayTextToPreview;
  private javax.swing.JPanel leftPanel;
  private javax.swing.JList fontNamesJList;
  private javax.swing.JComboBox horizJustificationJComboBox;
  private javax.swing.JList fontStylesJList;
  private javax.swing.JCheckBox autoSizeJCheckBox;
  private javax.swing.JPanel previewPanel;
  private javax.swing.JPanel fontSelectionJPanel;
  private javax.swing.JRadioButton javaFontJRadioButton;
  private javax.swing.JLabel jLabel7;
  private javax.swing.JTextField offsetXField;
  private javax.swing.JPanel textPanelAttributes;
  private javax.swing.JList fontSizesJList;
  private javax.swing.JTextField characterRotationField;
  private javax.swing.JLabel jLabel6;
  private javax.swing.JTextField scaleTextField;
  private javax.swing.ButtonGroup fontTypeButtonGroup;
  private javax.swing.JTextField textToPreviewField;
  private javax.swing.JScrollPane jScrollPane3;
  private javax.swing.JPanel rightPanel;
  private javax.swing.JPanel bottomPanel;
  private javax.swing.JLabel jLabel5;
  // End of variables declaration//GEN-END:variables

}
