package org.sqlproc.engine;

import java.util.ArrayList;
import java.util.List;

import org.sqlproc.engine.impl.ErrorMsg;

/**
 * General exception class for the SQL Processor parser.
 * 
 * This exception can be raised in the process of META SQL and Mapping rule parsing. It encapsulates ANTLR exceptions.
 * In the runtime (query command execution) only {@link org.hibernate.HibernateException} can be raised.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlEngineException extends RuntimeException {

    /**
     * List of parsing errors.
     */
    private List<ErrorMsg> errors = new ArrayList<ErrorMsg>();

    /**
     * Creates a new instance of the exception in the process of lexer and parser processing.
     * 
     * @param msg
     *            The error message.
     */
    public SqlEngineException(String msg) {
        super(msg);
    }

    /**
     * Creates a new instance of the exception in the process of lexer and parser processing.
     * 
     * @param msg
     *            The error message.
     * @param lexerErrors
     *            The optional list of (ANTLR) lexer errors.
     * @param parserErrors
     *            The optional list of (ANTLR) parser errors.
     */
    public SqlEngineException(String msg, List<ErrorMsg> lexerErrors, List<ErrorMsg> parserErrors) {
        super(msg);
        if (lexerErrors != null && !lexerErrors.isEmpty())
            errors.addAll(lexerErrors);
        if (parserErrors != null && !parserErrors.isEmpty())
            errors.addAll(parserErrors);
    }

    /**
     * Creates a human readable message regarding exceptions raised in the process of lexer and parser processing.
     * 
     * @return The readable message describing exception.
     */
    public String getMessage() {
        StringBuilder sb = new StringBuilder(super.getMessage());
        if (!errors.isEmpty())
            sb.append("\nANTLR messages:");
        for (ErrorMsg errorMsg : errors) {
            sb.append("\n  ").append(errorMsg.toString());
        }
        return sb.toString();
    }
}
