package org.sqlproc.engine.impl.type;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;

import org.hibernate.Hibernate;
import org.hibernate.Query;
import org.hibernate.type.Type;
import org.sqlproc.engine.SqlRuntimeException;
import org.sqlproc.engine.impl.BeanUtils;
import org.sqlproc.engine.impl.SqlUtils;

/**
 * The SQL META Types for ENUMINTEGER.
 * 
 * @author <a href="mailto:Vladimir.Hudec@gmail.com">Vladimir Hudec</a>
 */
public class SqlEnumIntegerType extends SqlGenericType {

    /**
     * {@inheritDoc}
     */
    @Override
    public Type getHibernateType() {
        return Hibernate.INTEGER;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public Class<?>[] getClassTypes() {
        return new Class[] {};
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public String[] getMetaTypes() {
        return new String[] { "EINTEGER", "EINT" };
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setResult(Object resultInstance, String attributeName, Object resultValue, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setResult " + getMetaTypes()[0] + ": resultInstance=" + resultInstance
                    + ", attributeName=" + attributeName + ", resultValue=" + resultValue);
        }
        Class<?> attributeType = BeanUtils.getFieldType(resultInstance.getClass(), attributeName);
        Method m = BeanUtils.getSetter(resultInstance, attributeName, attributeType);
        if (m != null) {
            Object enumInstance = SqlUtils.getValueToEnum(attributeType, resultValue);
            BeanUtils.simpleInvokeMethod(m, resultInstance, enumInstance);
        } else if (ingoreError) {
            logger.error("There's no getter for " + attributeName + " in " + resultInstance + ", META type is "
                    + getMetaTypes()[0]);
        } else {
            throw new SqlRuntimeException("There's no setter for " + attributeName + " in " + resultInstance
                    + ", META type is " + getMetaTypes()[0]);
        }
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public void setParameter(Query query, String paramName, Object inputValue, Class<?> inputType, boolean ingoreError)
            throws SqlRuntimeException {
        if (logger.isTraceEnabled()) {
            logger.trace(">>> setParameter " + getMetaTypes()[0] + ": paramName=" + paramName + ", inputValue="
                    + inputValue + ", inputType=" + inputType);
        }
        if (inputValue == null) {
            query.setParameter(paramName, inputValue, Hibernate.INTEGER);
        } else if (!inputValue.getClass().isEnum()) {
            if (!(inputValue instanceof Collection)) {
                if (ingoreError) {
                    logger.error("Incorrect integer based enum " + inputValue + " for " + paramName);
                } else {
                    throw new SqlRuntimeException("Incorrect integer based enum " + inputValue + " for " + paramName);
                }
            } else {
                List<Integer> vals = new ArrayList<Integer>();
                for (Iterator iter = ((Collection) inputValue).iterator(); iter.hasNext();) {
                    Object val = iter.next();
                    if (!val.getClass().isEnum()) {
                        if (ingoreError) {
                            logger.error("Incorrect integer based enum item " + val + " for " + paramName);
                        } else {
                            throw new SqlRuntimeException("Incorrect integer based enum item " + val + " for "
                                    + paramName);
                        }
                    } else {
                        Object o = SqlUtils.getEnumToValue(val);
                        if (o != null && o instanceof Integer) {
                            vals.add((Integer) o);
                        } else {
                            if (ingoreError) {
                                logger.error("Incorrect integer based enum item value " + o + " for " + paramName);
                            } else {
                                throw new SqlRuntimeException("Incorrect integer based enum item value " + o + " for "
                                        + paramName);
                            }
                        }
                    }
                }
                query.setParameterList(paramName, vals.toArray());
            }
        } else {
            Object o = SqlUtils.getEnumToValue(inputValue);
            if (o != null && o instanceof Integer) {
                query.setInteger(paramName, (Integer) o);
            } else {
                if (ingoreError) {
                    logger.error("Incorrect integer based enum value " + o + " for " + paramName);
                } else {
                    throw new SqlRuntimeException("Incorrect integer based enum value " + o + " for " + paramName);
                }
            }
        }
    }
}
