import org.w3c.dom.Document
import org.w3c.dom.Element
import org.w3c.dom.Node
import org.w3c.dom.NodeList

import javax.xml.parsers.DocumentBuilder
import javax.xml.parsers.DocumentBuilderFactory
import javax.xml.transform.OutputKeys
import javax.xml.transform.Transformer
import javax.xml.transform.TransformerFactory
import javax.xml.transform.dom.DOMSource
import javax.xml.transform.stream.StreamResult

import java.nio.file.Path
import java.nio.file.Paths
import java.net.InetAddress

import groovy.json.JsonBuilder
import groovy.transform.Field

@Field String surefirePluginVersion = "LATEST"

Properties requestProperties = request.properties
Path ymlFileGenerated = Paths.get(request.outputDirectory, 'browserstack_pre_generate.yml')
Path ymlFileOutput = Paths.get(request.outputDirectory, 'browserstack.yml')
Path pomXml = Paths.get(request.outputDirectory, 'pom.xml')
String testObservabilityYmlContent = "# Test Observability is an intelligent test reporting & debugging product. It collects data using the SDK. Read more about what data is collected at https://www.browserstack.com/docs/test-observability/references/terms-and-conditions\n" +
        "# Visit observability.browserstack.com to see your test reports and insights. To disable test observability, specify `testObservability: false` in the key below.\n" +
        "testObservability: true"

def isCommented(String currentLine) {
    StringBuilder currentLineBuilder = new StringBuilder(currentLine);
    while (currentLineBuilder.length() > 0 && Character.isWhitespace(currentLineBuilder.charAt(0))) {
        currentLineBuilder.deleteCharAt(0);
    }
    if(currentLineBuilder.length() > 0 && currentLineBuilder.toString().substring(0,1).equals("#")) {
        return true
    } else {
        return false
    }
}

def isAppAutomate(Properties reqProperties) {
    return reqProperties.stringPropertyNames().contains("BROWSERSTACK_APP") && !reqProperties.getProperty("BROWSERSTACK_APP").toString().equals("BROWSERSTACK_APP") && !reqProperties.getProperty("BROWSERSTACK_APP").toString().equals("null")
}

def defaultBrowserstackYMLFile(Properties reqProperties){
    String sourceProperty = reqProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? reqProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":sdk-archetype-integrate:v1.2" : reqProperties.getProperty("BROWSERSTACK_SOURCE")
    return "# =============================\n" +
            "# Set BrowserStack Credentials\n" +
            "# =============================\n" +
            "# Add your BrowserStack userName and accessKey here or set BROWSERSTACK_USERNAME and\n" +
            "# BROWSERSTACK_ACCESS_KEY as env variables\n" +
            "userName: "+reqProperties.getProperty("BROWSERSTACK_USERNAME")+"\n" +
            "accessKey: "+reqProperties.getProperty("BROWSERSTACK_ACCESS_KEY")+"\n" +
            "\n" +
            "# ======================\n" +
            "# BrowserStack Reporting\n" +
            "# ======================\n" +
            "# The following capabilities are used to set up reporting on BrowserStack:\n" +
            "# Set 'projectName' to the name of your project. Example, Marketing Website\n" +
            "projectName: "+reqProperties.getProperty("PROJECT_NAME")+"\n" +
            "# Set `buildName` as the name of the job / testsuite being run\n" +
            "buildName: "+reqProperties.getProperty("BUILD_NAME")+"\n" +
            "# `buildIdentifier` is a unique id to differentiate every execution that gets appended to\n" +
            "# buildName. Choose your buildIdentifier format from the available expressions:\n" +
            '# ${BUILD_NUMBER} (Default): Generates an incremental counter with every execution\n' +
            '# ${DATE_TIME}: Generates a Timestamp with every execution. Eg. 05-Nov-19:30\n' +
            "# Read more about buildIdentifiers here -> https://www.browserstack.com/docs/automate/selenium/organize-tests\n" +
            "buildIdentifier: '" + '#${BUILD_NUMBER}'+ "' # Supports strings along with either/both "+ '${expression}'+"\n" +
            "# Set `framework` of your test suite. Example, `testng`, `cucumber`, `cucumber-testng`\n" +
            "# This property is needed to send test context to BrowserStack (test name, status)\n" +
            "framework: "+reqProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ "\n" +
            "\n" +
            "# =======================================\n" +
            "# Platforms (Browsers / Devices to test)\n" +
            "# =======================================\n" +
            "# Platforms object contains all the browser / device combinations you want to test on.\n" +
            "# Entire list available here -> (https://www.browserstack.com/list-of-browsers-and-platforms/automate)\n" +
            "platforms:\n" +
            "  - os: OS X\n" +
            "    osVersion: Big Sur\n" +
            "    browserName: Chrome\n" +
            "    browserVersion: latest\n" +
            "  - os: Windows\n" +
            "    osVersion: 10\n" +
            "    browserName: Edge\n" +
            "    browserVersion: latest\n" +
            "  - deviceName: Samsung Galaxy S22 Ultra\n" +
            "    browserName: chrome # Try 'samsung' for Samsung browser\n" +
            "    osVersion: 12.0\n" +
            "    \n" +
            "# =======================\n" +
            "# Parallels per Platform\n" +
            "# =======================\n" +
            "# The number of parallel threads to be used for each platform set.\n" +
            "# BrowserStack's SDK runner will select the best strategy based on the configured value\n" +
            "#\n" +
            "# Example 1 - If you have configured 3 platforms and set `parallelsPerPlatform` as 2, a total of 6 (2 * 3) parallel threads will be used on BrowserStack\n" +
            "#\n" +
            "# Example 2 - If you have configured 1 platform and set `parallelsPerPlatform` as 5, a total of 5 (1 * 5) parallel threads will be used on BrowserStack\n" +
            "parallelsPerPlatform: 1\n" +
            "\n" +
            "source: "+ sourceProperty + "\n" +
            "\n" +
            "# ==========================================\n" +
            "# BrowserStack Local\n" +
            "# (For localhost, staging/private websites)\n" +
            "# ==========================================\n" +
            "# Set browserStackLocal to true if your website under test is not accessible publicly over the internet\n" +
            "# Learn more about how BrowserStack Local works here -> https://www.browserstack.com/docs/automate/selenium/local-testing-introduction\n" +
            "browserstackLocal: true # <boolean> (Default false)\n" +
            "\n" +
            "# Options to be passed to BrowserStack local in-case of advanced configurations\n" +
            "# browserStackLocalOptions:\n" +
            "  # localIdentifier: # <string> (Default: null) Needed if you need to run multiple instances of local.\n" +
            "  # forceLocal: true  # <boolean> (Default: false) Set to true if you need to resolve all your traffic via BrowserStack Local tunnel.\n" +
            "  # Entire list of arguments available here -> https://www.browserstack.com/docs/automate/selenium/manage-incoming-connections\n" +
            "  \n" +
            "# ===================\n" +
            "# Debugging features\n" +
            "# ===================\n" +
            "debug: false # <boolean> # Set to true if you need screenshots for every selenium command ran\n" +
            "networkLogs: false # <boolean> Set to true to enable HAR logs capturing\n" +
            "consoleLogs: errors # <string> Remote browser's console debug levels to be printed (Default: errors)\n" +
            "# Available options are `disable`, `errors`, `warnings`, `info`, `verbose` (Default: errors)\n" +
            "\n"
}

def createDependencyElement(Document document, String surefireVersion) {
    Element dependencyElement = document.createElement("dependency");
    def addElement = { String tagName, String textContent ->
        Element element = document.createElement(tagName);
        element.setTextContent(textContent);
        dependencyElement.appendChild(element);
        dependencyElement.appendChild(document.createTextNode("\n"))
    }
    dependencyElement.appendChild(document.createTextNode("\n"))
    addElement("groupId", "org.apache.maven.surefire");
    addElement("artifactId", "surefire-junit47");
    addElement("version", surefireVersion);
    return dependencyElement;
}

def addSurefireJunit47Dep(Document document, Element element, NodeList artifactList) {
    try {
        Node rootParentNode;
        for(int artifactIndex = 0; artifactIndex < artifactList.getLength(); artifactIndex++){ 
            Node artifactNode = artifactList.item(artifactIndex);
            
            if (artifactNode == null) {
                return;
            }

            Element pluginArtifactsElement = (Element) artifactNode.getParentNode();
            if(artifactNode.getTextContent().equals("maven-surefire-plugin") || artifactNode.getTextContent().equals("maven-failsafe-plugin")) {
                NodeList surefireVersionList = pluginArtifactsElement.getElementsByTagName("version");

                for(int surefireVersionIndex = 0; surefireVersionIndex < surefireVersionList.getLength(); surefireVersionIndex++){
                    Node surefireVersionNode = surefireVersionList.item(surefireVersionIndex);
                    if (surefireVersionNode != null && surefireVersionNode.getParentNode() != null && surefireVersionNode.getParentNode().getNodeName().equals("plugin")) {
                        surefirePluginVersion = surefireVersionNode.getTextContent();
                        break;
                    }
                }

                boolean shouldAddJUnit47Dep = true;
                NodeList pluginArtifactsList = pluginArtifactsElement.getElementsByTagName("artifactId");
                for(int pluginArtifactIndex = 0; pluginArtifactIndex < pluginArtifactsList.getLength(); pluginArtifactIndex++){
                    if (pluginArtifactsList.item(pluginArtifactIndex) != null 
                        && pluginArtifactsList.item(pluginArtifactIndex).getTextContent().equals("surefire-junit47")) {
                        shouldAddJUnit47Dep = false;
                        break;
                    }
                }

                if (shouldAddJUnit47Dep) {
                    NodeList pluginDependenciesList =  element.getElementsByTagName("dependencies");
                    Element dependencyElement = createDependencyElement(document, surefirePluginVersion);
                    if(pluginDependenciesList.item(0) == null){
                        Element dependenciesElement = document.createElement("dependencies");
                        dependenciesElement.appendChild(dependencyElement);
                        element.appendChild(dependenciesElement);
                    } else {
                        def dependenciesElement = pluginDependenciesList.item(0);
                        dependenciesElement.appendChild(dependencyElement);
                    }
                }
            }
        }
    } catch(Exception e) {
        println("It appears that the surefire-junit47 is not appended, causing parallels per platform to not function properly.")
    }
}

def updatePomXml(Properties reqProperties, Path pomXmlPath){
    File file = pomXmlPath.toFile();
    DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
    DocumentBuilder db = dbf.newDocumentBuilder();
    Document document = db.parse(file);
    document.getDocumentElement().normalize();
    NodeList nodeList = document.getElementsByTagName("plugin");
    for(int temp = 0; temp< nodeList.getLength(); temp++){
        Node node = nodeList.item(temp);
        if(node.getNodeType() == Node.ELEMENT_NODE){
            Element element = (Element) node;
            String artifactId = element.getElementsByTagName("artifactId").item(0).getTextContent();
            NodeList artifactList = element.getElementsByTagName("artifactId");
            
            addSurefireJunit47Dep(document, element, artifactList);
            if(artifactId.equalsIgnoreCase("maven-surefire-plugin") || artifactId.equalsIgnoreCase("maven-failsafe-plugin") ){
                NodeList pluginConfigList = element.getElementsByTagName("configuration");
                if(pluginConfigList.item(0) == null) {
                    Element configElement = document.createElement("configuration");
                    Element argLineElement = document.createElement("argLine");
                    argLineElement.setTextContent('-javaagent:"${com.browserstack:browserstack-java-sdk:jar}"');
                    configElement.appendChild(argLineElement);
                    int len = element.getChildNodes().getLength();
                    for(int cln = 0; cln < len; cln++){
                        Node pluginItem = element.getChildNodes().item(cln);
                        if(pluginItem.getNodeType() == Node.ELEMENT_NODE) {
                            Element pluginItemElement = (Element) pluginItem;
                            pluginItemElement.getParentNode().insertBefore(configElement, pluginItemElement.getNextSibling());
                            break;
                        }
                    }
                } else {
                    for(int pluginConfigListItem = 0; pluginConfigListItem< pluginConfigList.getLength(); pluginConfigListItem++){
                        Node pluginListItemNode = pluginConfigList.item(pluginConfigListItem);
                        if(pluginListItemNode.getNodeType() == Node.ELEMENT_NODE) {
                            Element pluginListItemElement = (Element) pluginListItemNode;
                            NodeList argLine = pluginListItemElement.getElementsByTagName("argLine");
                            if(argLine.getLength() > 0){
                                for(int argLineListItem = 0; argLineListItem < argLine.getLength(); argLineListItem++){
                                    Node argLineListItemNode = argLine.item(argLineListItem);
                                    if(argLineListItemNode.getNodeType() == Node.ELEMENT_NODE) {
                                        Element argLineListItemElement = (Element) argLineListItemNode;
                                        if(argLineListItemElement.getTextContent()!= null && argLineListItemElement.getTextContent().contains("browserstack-java-sdk:jar")){
                                            continue;
                                        } else {
                                            argLineListItemElement.setTextContent('-javaagent:"${com.browserstack:browserstack-java-sdk:jar}"\n'.concat(argLineListItemElement.getTextContent()));
                                        }
                                    }
                                }
                            } else {
                                Element argLineElement = document.createElement("argLine");
                                argLineElement.setTextContent('-javaagent:"${com.browserstack:browserstack-java-sdk:jar}"');
                                int len = pluginListItemElement.getChildNodes().getLength();
                                if(len == 0 || (len == 1 && pluginListItemElement.getChildNodes().item(0).getNodeType() == Node.TEXT_NODE)){
                                    pluginListItemElement.appendChild(argLineElement);
                                }
                                for(int cln = 0; cln < len; cln++){
                                    Node pluginItem = pluginListItemElement.getChildNodes().item(cln);
                                    if(pluginItem.getNodeType() == Node.ELEMENT_NODE) {
                                        Element pluginItemElement = (Element) pluginItem;
                                        pluginItemElement.getParentNode().insertBefore(argLineElement, pluginItemElement.getNextSibling());
                                        break;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    Transformer transformer = TransformerFactory.newInstance().newTransformer();

    /* Removing indentation as new lines seen
       when running archetype command programmatically in Eclipse and IntelliJ

       transformer.setOutputProperty(OutputKeys.INDENT, "yes");
     */
    transformer.setOutputProperty("{http://xml.apache.org/xslt}indent-amount", "4");
    StreamResult result = new StreamResult(new FileOutputStream(pomXmlPath.toAbsolutePath().toString()));
    DOMSource source = new DOMSource(document);
    transformer.transform(source,result);
}

def sendSDKEvent(String event, Properties reqProperties) {
    try {
        String userName = reqProperties.getProperty("BROWSERSTACK_USERNAME");
        String accessKey = reqProperties.getProperty("BROWSERSTACK_ACCESS_KEY");
        if(userName == 'YOUR_USERNAME' || accessKey == 'YOUR_ACCESS_KEY'){
            return
        }
        String funnelEventApiUrl = "";
        String BROWSERSTACK_API_URL = 'https://api.browserstack.com';
        String FUNNEL_INSTRUMENTATION_API_URL = 'sdk/v1/event';
        String eventType = event;
        List<String> productList = isAppAutomate(reqProperties) ? ['app-automate'] : ['automate']
        String source  = reqProperties.getProperty("BROWSERSTACK_SOURCE")
        String framework = reqProperties.getProperty("BROWSERSTACK_FRAMEWORK")
        String sourceProperty = ''
        if(isAppAutomate(reqProperties)){
            sourceProperty = source.equalsIgnoreCase("default") ? framework+ ":appium-sdk-archetype-integrate:v1.3" : source
        } else {
            sourceProperty = source.equalsIgnoreCase("default") ? framework+ ":sdk-archetype-integrate:v1.3" : source
        }

        funnelEventApiUrl = "${BROWSERSTACK_API_URL}/${FUNNEL_INSTRUMENTATION_API_URL}";
        def paramsObject = [
            userName: userName,
            accessKey: accessKey,
            event_type: eventType,
            detectedFramework: framework,
            event_properties: [
                language_framework: sourceProperty,
                language: "java",
                languageVersion: System.getProperty("java.version"),
                referrer: "java-archetype/1.3",
                product: productList,
                buildName: reqProperties.getProperty("BUILD_NAME"),
                os: System.getProperty('os.name'),
                hostname: InetAddress.getLocalHost().getHostName()
            ]
        ]
        def jsonPayload = new JsonBuilder(paramsObject).toPrettyString();
        def client = new URL(funnelEventApiUrl).openConnection();
        String credentials = "${userName}:${accessKey}".getBytes().encodeBase64().toString();
        client.setRequestMethod("POST");
        client.setDoOutput(true);
        client.setRequestProperty("Content-Type", "application/json");
        client.setRequestProperty("Authorization", "Basic ${credentials}");
        client.getOutputStream().write(jsonPayload.getBytes("UTF-8"));
    } catch(Exception e) {}
}

try {
    updatePomXml(requestProperties, pomXml);
    sendSDKEvent('SDKSetup', requestProperties);
    def defaultBrowserstackContent = new StringBuilder();
    def observabilityFileContent = new StringBuilder();
    Scanner scGenerated = new Scanner(ymlFileGenerated)
    String currentLine = ''
    boolean observabilityConfigStarted = false
    boolean appAutomateConfigStarted = false
    boolean appAutomateAndroidPlatformsStarted = false
    boolean appAutomateIOSPlatformsStarted = false
    boolean defaultPlatformsStarted = false
    boolean frameworkConfigStarted = false
    boolean testObservabilityContentStarted = false
    while(scGenerated.hasNext()){
        currentLine = scGenerated.nextLine()
        String sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":obs-sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
        if(currentLine.contains("OBSERVABILITY ONLY CONFIG")) {
            /* Dummy comment added to identify observability config from default browserstack config */
            observabilityConfigStarted = true
            continue
        }
        if(currentLine.contains("App Automate ONLY CONFIG Start")){
            /* Dummy comment added to identify app automate config from default browserstack config */
            appAutomateConfigStarted = true
            continue
        } else if(currentLine.contains("App Automate ONLY CONFIG End")){
            /* Dummy comment added to identify app automate config from default browserstack config */
            appAutomateConfigStarted = false
            continue
        } else if(currentLine.contains("Android Platforms Start")){
            /* Dummy comment added to identify app automate android config from default browserstack config */
            appAutomateAndroidPlatformsStarted = true
            continue
        } else if(currentLine.contains("Android Platforms End")){
            /* Dummy comment added to identify app automate android config from default browserstack config */
            appAutomateAndroidPlatformsStarted = false
            continue
        } else if(currentLine.contains("IOS Platforms Start")){
            /* Dummy comment added to identify app automate ios config from default browserstack config */
            appAutomateIOSPlatformsStarted = true
            continue
        } else if(currentLine.contains("IOS Platforms End")){
            /* Dummy comment added to identify app automate ios config from default browserstack config */
            appAutomateIOSPlatformsStarted = false
            continue
        } else if(currentLine.contains("Default Platforms Start")){
            /* Dummy comment added to identify app automate ios config from default browserstack config */
            defaultPlatformsStarted = true
            continue
        } else if(currentLine.contains("Default Platforms End")){
            /* Dummy comment added to identify app automate ios config from default browserstack config */
            defaultPlatformsStarted = false
            continue
        } else if(currentLine.contains("Framework Config Start")){
            frameworkConfigStarted = true
            continue
        } else if(currentLine.contains("Framework Config End")){
            frameworkConfigStarted = false
            continue
        } else if(currentLine.contains("Test Observability Content Start")){
            testObservabilityContentStarted = true
            continue
        } else if(currentLine.contains("Test Observability Content End")){
            testObservabilityContentStarted = false
            continue
        }

        if(observabilityConfigStarted) {
            /* Append contents for observability platform */
            if(currentLine.contains("buildName") && currentLine.contains("browserstack build")) {
                currentLine = "buildName: your unique build or job name goes here"
            }
            if(currentLine.contains("projectName") && currentLine.contains("BrowserStack Samples")) {
                currentLine = "projectName: your unique project name goes here"
            }
            if(currentLine.contains("source:")){
                currentLine = "source: "+ sourceProperty
            }
            if(frameworkConfigStarted){
                if(requestProperties.getProperty("BROWSERSTACK_FRAMEWORK").equalsIgnoreCase("java")){
                    continue
                }
            } else if (testObservabilityContentStarted){
                if(!requestProperties.getProperty("BROWSERSTACK_FRAMEWORK").equalsIgnoreCase("testng")){
                    continue
                }
            }
            observabilityFileContent.append(currentLine).append("\n")
        } else if(appAutomateConfigStarted){
            /* Append contents for app automate config only */
            if(isAppAutomate(requestProperties)){
                defaultBrowserstackContent.append(currentLine).append("\n")
            }
        } else if(appAutomateAndroidPlatformsStarted){
            /* Append contents for app automate android config only */
            if(isAppAutomate(requestProperties) && requestProperties.stringPropertyNames().contains("BROWSERSTACK_PLATFORMS") && requestProperties.getProperty("BROWSERSTACK_PLATFORMS").toString().equals("ANDROID")){
                defaultBrowserstackContent.append(currentLine).append("\n")
            }
        } else if(appAutomateIOSPlatformsStarted){
            /* Append contents for app automate ios config only */
            if(isAppAutomate(requestProperties) && requestProperties.stringPropertyNames().contains("BROWSERSTACK_PLATFORMS") && requestProperties.getProperty("BROWSERSTACK_PLATFORMS").toString().equals("IOS")){
                defaultBrowserstackContent.append(currentLine).append("\n")
            }
        } else if(defaultPlatformsStarted){
            /* Append contents for default platforms only */
            if(!isAppAutomate(requestProperties)){
                defaultBrowserstackContent.append(currentLine).append("\n")
            }
        } else if(frameworkConfigStarted){
            if(!requestProperties.getProperty("BROWSERSTACK_FRAMEWORK").equalsIgnoreCase("java")){
                defaultBrowserstackContent.append(currentLine).append("\n")
            }
        } else if (testObservabilityContentStarted){
            if(requestProperties.getProperty("BROWSERSTACK_FRAMEWORK").equalsIgnoreCase("testng")){
                defaultBrowserstackContent.append(currentLine).append("\n")
            }
        } else {
            /* Append contents for default config only */
            if(currentLine.contains("source:") && isAppAutomate(requestProperties)){
                sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":appium-sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
                currentLine = "source: "+ sourceProperty
            } else if (currentLine.contains("source:")){
                sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
                currentLine = "source: "+ sourceProperty
            }
            defaultBrowserstackContent.append(currentLine).append("\n")
        }
    }

    File outputYMLFile = new File(ymlFileOutput.toString())
    if(outputYMLFile.exists()) {
        def overrideYMLFileContent = new StringBuilder();
        Scanner scOutput
        scOutput = new Scanner(ymlFileOutput)
        currentLine = ''
        boolean overrideUserNameFound = false
        boolean overrideAccessKeyFound = false
        boolean overrideProjectNameFound = false
        boolean overrideBuildNameFound = false
        boolean overrideTestObservabilityFound = false
        boolean overrideBrowserstackAutomationFound = false
        boolean overrideBrowserstackAppFound = false
        boolean overrideSourceFound = false
        String sourceProperty = requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":sdk-archetype-integrate:v1.2"
        while(scOutput.hasNext()){
            currentLine = scOutput.nextLine()
            /* Merging with user's existing yml file */
            if(currentLine.length() && !isCommented(currentLine)) {
                if(currentLine.contains("userName:")) {
                    currentLine = !requestProperties.getProperty("BROWSERSTACK_USERNAME").toString().equals("YOUR_USERNAME") ? "userName: " + requestProperties.getProperty("BROWSERSTACK_USERNAME").toString() : currentLine
                    overrideUserNameFound = true
                } else if(currentLine.contains("accessKey:")) {
                    currentLine = !requestProperties.getProperty("BROWSERSTACK_ACCESS_KEY").toString().equals("YOUR_ACCESS_KEY") ? "accessKey: " + requestProperties.getProperty("BROWSERSTACK_ACCESS_KEY").toString() : currentLine
                    overrideAccessKeyFound = true
                } else if(currentLine.contains("projectName:")) {
                    currentLine = !requestProperties.getProperty("PROJECT_NAME").toString().equals("BrowserStack Samples") && !requestProperties.getProperty("PROJECT_NAME").toString().equals("your unique project name goes here") ? "projectName: " + requestProperties.getProperty("PROJECT_NAME").toString() : currentLine
                    overrideProjectNameFound = true
                } else if(currentLine.contains("buildName:")) {
                    currentLine = !requestProperties.getProperty("BUILD_NAME").toString().equals("browserstack build") && !requestProperties.getProperty("BUILD_NAME").toString().equals("your unique build or job name goes here") ? "buildName: " + requestProperties.getProperty("BUILD_NAME").toString() : currentLine
                    overrideBuildNameFound = true
                } else if(currentLine.contains("testObservability:")) {
                    currentLine = "# Test Observability is an intelligent test reporting & debugging product. It collects data using the SDK. Read more about what data is collected at https://www.browserstack.com/docs/test-observability/references/terms-and-conditions\n" +
                            "# Visit observability.browserstack.com to see your test reports and insights. To disable test observability, specify `testObservability: false` in the key below.\n" +
                            "testObservability: true"
                    overrideTestObservabilityFound = true
                } else if(currentLine.contains("browserstackAutomation:")) {
                    currentLine = "browserstackAutomation: " + requestProperties.getProperty("BROWSERSTACK_AUTOMATION").toString()
                    overrideBrowserstackAutomationFound = true
                } else if(currentLine.contains("app:")) {
                    currentLine = !requestProperties.getProperty("BROWSERSTACK_APP").toString().equals("BROWSERSTACK_APP") ? "app: " + requestProperties.getProperty("BROWSERSTACK_APP").toString() : currentLine
                    overrideBrowserstackAppFound = true
                } else if(currentLine.contains("source:")) {
                    if(isAppAutomate(requestProperties)) {
                        sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":appium-sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
                        currentLine = "source: "+ sourceProperty
                    } else {
                        sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
                        currentLine = "source: "+ sourceProperty
                    }
                    overrideSourceFound = true
                }
            }
            overrideYMLFileContent.append(currentLine).append("\n")
        }

        /* Appending to user's existing yml file if config key not found */
        if(!overrideUserNameFound && !requestProperties.getProperty("BROWSERSTACK_USERNAME").toString().equals("YOUR_USERNAME")) {
            overrideYMLFileContent.append("\n")
            overrideYMLFileContent.append("userName: " + requestProperties.getProperty("BROWSERSTACK_USERNAME").toString()).append("\n")
        }
        if(!overrideAccessKeyFound && !requestProperties.getProperty("BROWSERSTACK_ACCESS_KEY").toString().equals("YOUR_ACCESS_KEY")) {
            overrideYMLFileContent.append("\n")
            overrideYMLFileContent.append("accessKey: " + requestProperties.getProperty("BROWSERSTACK_ACCESS_KEY").toString()).append("\n")
        }
        if(!overrideProjectNameFound && !requestProperties.getProperty("PROJECT_NAME").toString().equals("BrowserStack Samples") && !requestProperties.getProperty("PROJECT_NAME").toString().equals("your unique project name goes here")) {
            overrideYMLFileContent.append("\n")
            overrideYMLFileContent.append("# Set 'projectName' to the name of your project. Example, Marketing Website").append("\n")
            overrideYMLFileContent.append("projectName: " + requestProperties.getProperty("PROJECT_NAME").toString()).append("\n")
        }
        if(!overrideBuildNameFound && !requestProperties.getProperty("BUILD_NAME").toString().equals("browserstack build") && !requestProperties.getProperty("BUILD_NAME").toString().equals("your unique build or job name goes here")) {
            overrideYMLFileContent.append("\n")
            overrideYMLFileContent.append("# Set `buildName` as the name of the job / testsuite being run").append("\n")
            overrideYMLFileContent.append("buildName: " + requestProperties.getProperty("BUILD_NAME").toString()).append("\n")
        }
        if(!overrideTestObservabilityFound) {
            overrideYMLFileContent.append("\n")
            overrideYMLFileContent.append("# Test Observability is an intelligent test reporting & debugging product. It collects data using the SDK. Read more about what data is collected at https://www.browserstack.com/docs/test-observability/references/terms-and-conditions").append("\n")
            overrideYMLFileContent.append("# Visit observability.browserstack.com to see your test reports and insights. To disable test observability, specify `testObservability: false` in the key below.").append("\n")
            overrideYMLFileContent.append("testObservability: true").append("\n")
        }
        if(!overrideBrowserstackAutomationFound && requestProperties.getProperty("BROWSERSTACK_AUTOMATION").toString().equals("false")) {
            overrideYMLFileContent.append("\n")
            overrideYMLFileContent.append("# =======================").append("\n")
            overrideYMLFileContent.append("# Browserstack Automation").append("\n")
            overrideYMLFileContent.append("# =======================").append("\n")
            overrideYMLFileContent.append("# This flag decides whether to run your test suite on Browserstack platform or not").append("\n")
            overrideYMLFileContent.append("browserstackAutomation: " + requestProperties.getProperty("BROWSERSTACK_AUTOMATION").toString()).append("\n")
        }
        if(!overrideBrowserstackAppFound && !requestProperties.getProperty("BROWSERSTACK_APP").toString().equals("BROWSERSTACK_APP")) {
            overrideYMLFileContent.append("# =======================================").append("\n")
            overrideYMLFileContent.append("# App").append("\n")
            overrideYMLFileContent.append("# =======================================").append("\n")
            overrideYMLFileContent.append("# Set `app` to use the appliction under test for Appium sessions.").append("\n")
            overrideYMLFileContent.append("# Available options:").append("\n")
            overrideYMLFileContent.append("# app: '/path/to/local/app.apk'").append("\n")
            overrideYMLFileContent.append("# OR").append("\n")
            overrideYMLFileContent.append("# app: 'bs://<app-id>' # App URL returned when uploading the app to BrowserStack manually").append("\n")
            overrideYMLFileContent.append("# Visit https://www.browserstack.com/docs/app-automate/appium/set-up-tests/specify-app for more options").append("\n")
            overrideYMLFileContent.append("app: " + requestProperties.getProperty("BROWSERSTACK_APP").toString()).append("\n")
        }
        if(!overrideSourceFound) {
            overrideYMLFileContent.append("\n")
            if(isAppAutomate(requestProperties)) {
                sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":appium-sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
                overrideYMLFileContent.append("source: "+ sourceProperty)
            } else {
                sourceProperty = requestProperties.getProperty("BROWSERSTACK_SOURCE").equalsIgnoreCase("default") ? requestProperties.getProperty("BROWSERSTACK_FRAMEWORK")+ ":sdk-archetype-integrate:v1.2" : requestProperties.getProperty("BROWSERSTACK_SOURCE")
                overrideYMLFileContent.append("source: "+ sourceProperty)
            }
            overrideYMLFileContent.append("\n")
        }

        PrintWriter writer = new PrintWriter(ymlFileOutput.toFile())
        /* Override contents of browserstack.yml */
        writer.write(overrideYMLFileContent.toString())
        writer.close()
    } else {
        if(requestProperties.getProperty("BROWSERSTACK_AUTOMATION") != null
                && !requestProperties.getProperty("BROWSERSTACK_AUTOMATION").toString().equals("null")
                && requestProperties.getProperty("BROWSERSTACK_AUTOMATION").toString().equals("false")
        ) {
            PrintWriter writer = new PrintWriter(ymlFileOutput.toFile())
            /* Replace contents of browserstack.yml */
            writer.write(observabilityFileContent.toString())
            writer.close()
        } else {
            PrintWriter writer = new PrintWriter(ymlFileOutput.toFile())
            /* Replace contents of browserstack.yml */
            writer.write(defaultBrowserstackContent.toString())
            writer.close()
        }
    }
} catch(Throwable e) {
    PrintWriter writer = new PrintWriter(ymlFileOutput.toFile())
    /* Replace default contents of browserstack.yml */
    String defaultYmlContent = defaultBrowserstackYMLFile(requestProperties).toString();
    if(requestProperties.getProperty("BROWSERSTACK_FRAMEWORK").equalsIgnoreCase("testng")){
        defaultYmlContent.concat(testObservabilityYmlContent);
    }
    writer.write(defaultYmlContent)
    writer.close()
    e.printStackTrace()
} finally {
    new File(ymlFileGenerated.toString()).delete()
}
