package com.btjava.common.utils.office.excel;

import com.btjava.common.utils.office.excel.pojo.ColCallMethod;
import com.btjava.common.utils.office.excel.pojo.ExcelCol;
import com.btjava.common.utils.office.excel.pojo.ExcelData;
import com.btjava.common.utils.text.DataTools;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellStyle;
import org.apache.poi.ss.usermodel.DataValidation;
import org.apache.poi.ss.usermodel.DataValidationConstraint;
import org.apache.poi.ss.usermodel.DataValidationHelper;
import org.apache.poi.ss.usermodel.Font;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.VerticalAlignment;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.ss.util.CellRangeAddressList;
import org.apache.poi.xssf.streaming.SXSSFWorkbook;
import org.apache.poi.xssf.usermodel.XSSFDataValidation;

import java.io.File;
import java.io.FileOutputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;

public class ExcelUtil {
    /**
     * 默认最大行数
     */
    public static final int MAX_ROW_NUM =50000;

    /**
     * 导出Excel
     * @param wb excel对象 SXSSFWorkbook wb = new SXSSFWorkbook(100);
     * @param titleList 表头列表对象
     * @param dataList 数据列表
     * @param headTitle 标题
     * @param sheetName sheet名称
     * @return 组装好的excel
     */
    public static SXSSFWorkbook createWorkBook(SXSSFWorkbook wb, List<ExcelCol> titleList, List<Object> dataList , String headTitle, String sheetName) throws Exception{
        List<ExcelData> list = new ArrayList<ExcelData>(dataList.size());
        dataList.forEach(one->{
            list.add(new ExcelData(one));
        });
        return createWorkBook(wb,titleList,list,headTitle,sheetName,0);
    }
    /**
     * 导出Excel
     * @param wb excel对象 SXSSFWorkbook wb = new SXSSFWorkbook(100);
     * @param titleList 表头列表对象
     * @param dataList 数据列表
     * @param headTitle 标题
     * @param sheetName sheet名称
     * @param type 无业务意义，仅用于区别两个方法
     * @return 组装好的excel
     */
    public static SXSSFWorkbook createWorkBook(SXSSFWorkbook wb, List<ExcelCol> titleList, List<ExcelData> dataList , String headTitle, String sheetName,Integer type) throws Exception{
        Sheet sheeet =null;
        if(StringUtils.isNotBlank(sheetName)){
            sheeet =wb.createSheet(sheetName);
        }else{
            sheeet =wb.createSheet();
        }
        // 设置文件标题
        Cell headCell= sheeet.createRow(0).createCell(0);
        CellStyle headStyle = wb.createCellStyle();
        int startRow =0;
        if(StringUtils.isNotBlank(headTitle)){
            //设置字体
            Font font = wb.createFont();
            font.setColor(Font.COLOR_NORMAL);
            font.setBold(true);
            //设置样式
            headStyle.setFont(font);
            headStyle.setVerticalAlignment(VerticalAlignment.CENTER);
            headCell.setCellStyle(headStyle);
            headCell.setCellValue(headTitle);
            //合并单元格
            CellRangeAddress region = new CellRangeAddress(0, 1, 0, titleList.size()==0?0:titleList.size()-1);
            //设置内容起始行
            startRow =2;
            sheeet.addMergedRegion(region);
        }
        //设置表头
        Row titleRow =sheeet.createRow(startRow);
        for(int i=0;i<titleList.size();i++){
            ExcelCol col = titleList.get(i);
            Cell cell=titleRow.createCell(i);
            //设置表头值
            cell.setCellValue(col.getTitile());
            if(!Objects.isNull(col.getCellStyleMap())){
                Map<String,CellStyle> cellStyleMap = col.getCellStyleMap();
                if(cellStyleMap.containsKey(ExcelCol.STYLE_KEY_ONLY_TITLE)){
                    //优先取ONLY_TITLE
                    cell.setCellStyle(cellStyleMap.get(ExcelCol.STYLE_KEY_ONLY_TITLE));
                }else if(cellStyleMap.containsKey(ExcelCol.STYLE_KEY_All)){
                    cell.setCellStyle(cellStyleMap.get(ExcelCol.STYLE_KEY_All));
                }
            }
            //设置列宽
            if(!Objects.isNull(col.getColumnWidth()) && col.getColumnWidth() >0){
                sheeet.setColumnWidth(i,col.getColumnWidth()*256);
            }else{
                //默认10个字长度
                sheeet.setColumnWidth(i,10*256);
            }
        }
        //填入数据
        for(int i =0;i<dataList.size();i++){
            // 获取数据对象
            ExcelData excelData = dataList.get(i);
            //将对象转为Map
            Map<String, Object> map  = DataTools.objectToMap(excelData.getData());
            Row row =sheeet.createRow(i+3);
            for(int j=0;j<titleList.size();j++){
                // 获取表头
                ExcelCol col = titleList.get(j);
                // 创建cell
                Cell cell=row.createCell(j);
                if(!Objects.isNull(col.getCellStyleMap())){
                    Map<String,CellStyle> cellStyleMap = col.getCellStyleMap();
                    if(excelData.getStyleFlag() && cellStyleMap.containsKey(ExcelCol.STYLE_KEY_ONLY_CONDITION)){
                        //优先取ONLY_TITLE
                        cell.setCellStyle(cellStyleMap.get(ExcelCol.STYLE_KEY_ONLY_TITLE));
                    }else if(cellStyleMap.containsKey(ExcelCol.STYLE_KEY_All)){
                        cell.setCellStyle(cellStyleMap.get(ExcelCol.STYLE_KEY_All));
                    }
                }
                if(!Objects.isNull(col.getColCallMethod())){
                    ColCallMethod colCallMethod = col.getColCallMethod();
                    //设置值
                    cell.setCellValue(formatCellValue(col.getColCallMethod(),map.get(col.getField()),i));
                }else{
                    //设置值
                    cell.setCellValue(Objects.isNull(map.get(col.getField()))?"":map.get(col.getField()).toString());
                }
            }
        }
        return wb;
    }

    /**
     * 设置下拉框
     * @param formulaString 下拉框数组
     * @param firstRow 起始行 从0开始
     * @param lastRow 终止行
     * @param firstCol 起始列
     * @param lastCol 终止列
     * @return
     */
    public Sheet setDropDownBox(Sheet sheet,String[] formulaString,int firstRow, int lastRow,int firstCol,int lastCol){
        //加载下拉列表内容
        DataValidationHelper helper = sheet.getDataValidationHelper();
        DataValidationConstraint constraint = helper.createExplicitListConstraint(formulaString);
        //设置下拉框位置
        CellRangeAddressList addressList = null;
        addressList = new CellRangeAddressList(firstRow, lastRow, firstCol, lastCol);
        DataValidation dataValidation = helper.createValidation(constraint, addressList);
        //处理Excel兼容性问题
        if(dataValidation instanceof XSSFDataValidation){
            //数据校验
            dataValidation.setSuppressDropDownArrow(true);
            dataValidation.setShowErrorBox(true);
        }else{
            dataValidation.setSuppressDropDownArrow(false);
        }
        sheet.addValidationData(dataValidation);
        return sheet;
    }
    private static String formatCellValue(ColCallMethod colCallMethod,Object obj,int rowNum){
        // 获取传入值的下标(默认下标为0)
        int colIndex = Objects.isNull(colCallMethod.getColIndexNum())|| colCallMethod.getColIndexNum()<0?0:colCallMethod.getColIndexNum();
        Object[] new_param =null;
        if(rowNum==0){
            //获取参数对象
            Object[] param = colCallMethod.getParams();
            new_param = new Object[Objects.isNull(param)?1:param.length+1];
            // 混入参数的逻辑
            if(Objects.isNull(param)){
                new_param[0]=obj;
            }else{
                for(int i=0;i<new_param.length;i++){
                    if(i<colIndex){
                        new_param[i]=param[i];
                    }if(i==colIndex){
                        new_param[i]=obj;
                    }else{
                        new_param[i] =param[i-1];
                    }
                }
            }
        }else{
            new_param =colCallMethod.getParams();
            new_param[colIndex]=obj;
        }
        //传入新参数
        colCallMethod.setParams(new_param);
        return Objects.isNull(DataTools.callMethod(colCallMethod))?(Objects.isNull(obj)?null:obj.toString()):DataTools.callMethod(colCallMethod).toString();
    }

    /**
     *
     * @param wb workBoot对象
     * @param cacheFilePath 缓存文件地址
     * @param fileName 文件名称，不到后缀
     * @return 输出流对象
     * @throws Exception
     */
    public static FileOutputStream export(SXSSFWorkbook wb,String cacheFilePath,String fileName) throws Exception{
        if(!StringUtils.isNotBlank(fileName)){
            int rodom = (int) ((Math.random() * 9 + 1) * 1000);
            // 拼接excel名称
            StringBuilder sb = new StringBuilder();
            sb.append(System.currentTimeMillis());
            sb.append(rodom);
            fileName = sb.toString();
        }
        File folderFile = new File(cacheFilePath);
        File newFile = new File(cacheFilePath+File.separator+fileName+".xlsx");
        /** 根据真实路径创建目录 **/
        if (!folderFile.exists()) {
            folderFile.mkdirs();
        }
        FileOutputStream fOut = new FileOutputStream(newFile);//将数据写入Excel
        wb.write(fOut);
        fOut.flush(); // 刷新缓冲区
        fOut.close();
        return fOut;
    }
}
