package com.btjava.common.utils.text;

import com.btjava.common.utils.enums.DateEnum;
import com.btjava.common.utils.enums.ResultCode;
import com.btjava.common.utils.exception.CustomException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.regex.Pattern;

/**
 * 日期类常用转换
 */
public class DateUtils {
    private static final Logger logger = LoggerFactory.getLogger(DateUtils.class);
    private static final String DATE_FORMAT_STR ="yyyy-MM-dd";
    private static final String DATETIME_FORMAT_STR="yyyy-MM-dd HH:mm:ss";
    private static final String DATE_FORMAT_STR_ZN ="yyyy年MM月dd日";
    private static final String DATETIME_FORMAT_STR_ZN="yyyy年MM月dd日 HH:mm:ss";
    private static final String MID_MONTH = "04,06,09,11";
    private static final String LITTLE_MONTH = "02";
    /**
     * 获取指定日期对于的中文时间字符串
     * @param date 指定日期
     * @return 时间字符串(yyyy年MM月dd日 HH:mm:ss)
     */
    public static String getDateTimeStrZN(Date date){
        return dateToStr(date,DATETIME_FORMAT_STR_ZN);
    }
    /**
     * 获取指定日期对于的时间字符串
     * @param date 指定日期
     * @return 时间字符串(yyyy-MM-dd HH:mm:ss)
     */
    public static String getDateTimeStr(Date date){
        return dateToStr(date,DATETIME_FORMAT_STR);
    }
    /**
     * 获取指定日期对于的中文时间字符串
     * @param date 指定日期
     * @return 日期字符串(yyyy年MM月dd日)
     */
    public static String getDateStrZN(Date date){
        return dateToStr(date,DATE_FORMAT_STR_ZN);
    }
    /**
     * 获取指定日期对于的时间字符串
     * @param date 指定日期
     * @return 日期字符串(yyyy-MM-dd)
     */
    public static String getDateStr(Date date){
        return dateToStr(date,DATE_FORMAT_STR);
    }

    /**
     * 日期类型转换指定格式的字符串
     * @param date 日期
     * @param format 格式化字符串
     * @return 指定格式的时间字符串
     */
    public static String dateToStr(Date date,String format){
        SimpleDateFormat simple = new SimpleDateFormat(format);
        return simple.format(date);
    }
    /**
     * 获取时间戳
     * @param date 日期
     * @return 时间戳
     */
    public static Integer getTimestamp(Date date){
        return (int)date.getTime()/1000;
    }

    /**
     * 字符串转换成日期
     * @param dateStr 日期字符串
     * @param format 日期格式化字符串
     * @return 日期
     */
    public static Date getDateByStr(String dateStr,String format){
        //字符串和日期字符串是否匹配
        if(!checkDateStr(dateStr,format)){
            throw new CustomException(ResultCode.DATA_FORMAT_ERROR.getCode(),ResultCode.DATA_FORMAT_ERROR.getMessage());
        }
        SimpleDateFormat simple = new SimpleDateFormat(format);
        Date result =null;
        try{
            result = simple.parse(dateStr);
        }catch (Exception e){
            logger.warn("字符串转换成日期失败！message:{}",e.getMessage());
            e.printStackTrace();
            throw new CustomException(ResultCode.SYSTEM_ERROR.getCode(),ResultCode.SYSTEM_ERROR.getMessage());
        }
        return result;
    }

    /**
     * 日期加减
     * @param date 日期
     * @param day 加减天数
     * @return 计算后的日期
     */
    public static Date dateAddDay(Date date,int day){
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.DAY_OF_MONTH,day);
        return cal.getTime();
    }
    /**
     * 月份加减
     * @param date 日期
     * @param month 加减月数
     * @return 计算后的日期
     */
    public static Date dateAddMonth(Date date,int month){
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.MONTH,month);
        return cal.getTime();
    }
    /**
     * 年份加减
     * @param date 日期
     * @param year 加减年数
     * @return 计算后的日期
     */
    public static Date dateAddYear(Date date,int year){
        Calendar cal = Calendar.getInstance();
        cal.setTime(date);
        cal.add(Calendar.YEAR,year);
        return cal.getTime();
    }
    /**
     * 两个日期比较
     * @param date1
     * @param date2
     * @return date1是否大于date2
     */
    public static Boolean compareTo(Date date1,Date date2){
        return date1.getTime() - date2.getTime()>=0;
    }

    /**
     * 计算两个日期之间的差值(带方向)
     * @param date1
     * @param date2
     * @return 计算两个日期之间的差值(年和月不精确)，使用DateEnum枚举获取结果
     */
    public static Map<String,Integer> getDifference(Date date1, Date date2){
        Map<String,Integer> map = new HashMap<String,Integer>();
        Long dateTime1 = date1.getTime();
        Long dateTime2 = date2.getTime();
        //一共相差秒数
        Integer full_second = (int)((dateTime1-dateTime2)/1000);
        map.put(DateEnum.FULL_SECOND.getCode(),full_second);
        //一共相差分钟数
        Integer full_minute =full_second/60;
        map.put(DateEnum.FULL_MINUTE.getCode(),full_second);
        //秒数余数
        Integer second =full_second%60;
        map.put(DateEnum.SECOND.getCode(),second);
        Integer full_hour =full_minute/60;
        map.put(DateEnum.FULL_HOUR.getCode(),full_hour);
        Integer minute =full_minute%60;
        map.put(DateEnum.MINUTE.getCode(),full_second);
        Integer full_Day =full_hour/24;
        map.put(DateEnum.FULL_DAY.getCode(),full_Day);
        Integer hour =full_hour%24;
        map.put(DateEnum.HOUR.getCode(),hour);
        Integer full_month = full_Day/31;
        map.put(DateEnum.FULL_MONTH.getCode(),full_month);
        Calendar cal1 = Calendar.getInstance();
        cal1.setTime(date1);
        Calendar cal2 = Calendar.getInstance();
        cal1.setTime(date2);
        map.put(DateEnum.FULL_YEAR.getCode(),cal1.get(Calendar.YEAR)-cal2.get(Calendar.YEAR));
        return map;
    }
    /**
     * 检查日期格式
     * @param dateStr 时间字符串
     * @param format 格式化字符串(目前只支持yyyy、MM、dd、HH、mm、ss校验)
     * @return 传入字符串是否和日期格式匹配
     */
    public static boolean checkDateStr(String dateStr,String format){
        boolean flag = true;
        //获取年份的位置
        int yearIndex = format.indexOf("yyyy");
        //获取月份的位置
        int monthIndex = format.indexOf("MM");
        //获取日期的位置
        int dayIndex = format.indexOf("dd");
        //获取小时的位置
        int hourIndex = format.indexOf("HH");
        //获取分钟的位置
        int minuteIndex =format.indexOf("mm");
        //获取秒钟的位置
        int secondIndex = format.indexOf("ss");
        if(yearIndex>=0){
            int num =4;
            String str = dateStr.substring(yearIndex,yearIndex+num);
            if(!Pattern.matches("^\\d{"+num+"}$",str)){
                return false;
            }
        }
        if(monthIndex>=0){
            int num =2;
            String str = dateStr.substring(monthIndex,monthIndex+num);
            if(!Pattern.matches("^\\d{"+num+"}$",str)|| Integer.parseInt(str)<=0 || Integer.parseInt(str)>12){
                return false;
            }
        }
        if(dayIndex>=0){
            int num =2;
            int day =31;
            if(monthIndex>=0){
                String monthStr = dateStr.substring(monthIndex,monthIndex+num);
                if(MID_MONTH.indexOf(monthStr)>=0){
                    day=30;
                }else if(LITTLE_MONTH.indexOf(monthStr)>=0){
                    day=29;
                }
            }
            String str = dateStr.substring(dayIndex,dayIndex+num);
            if(!Pattern.matches("^\\d{"+num+"}$",str)|| Integer.parseInt(str)<=0 || Integer.parseInt(str)>day){
                return false;
            }
        }
        if(hourIndex>=0){
            int num =2;
            String str = dateStr.substring(hourIndex,hourIndex+num);
            if(!Pattern.matches("^\\d{"+num+"}$",str)|| Integer.parseInt(str)<=0 || Integer.parseInt(str)>24){
                return false;
            }
        }
        if(minuteIndex>=0){
            int num =2;
            String str = dateStr.substring(minuteIndex,minuteIndex+num);
            if(!Pattern.matches("^\\d{"+num+"}$",str)|| Integer.parseInt(str)<=0 || Integer.parseInt(str)>60){
                return false;
            }
        }
        if(secondIndex>=0){
            int num =2;
            String str = dateStr.substring(secondIndex,secondIndex+num);
            if(!Pattern.matches("^\\d{"+num+"}$",str)|| Integer.parseInt(str)<=0 || Integer.parseInt(str)>60){
                return false;
            }
        }
        return flag;
    }

    /**
     * 获取传入日期的零点
     * @param day
     * @return 零点 yyyy-MM-dd 00:00:00
     */
    public static Date getDayBeginTime(Date day){
        Calendar cal =Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        cal.set(Calendar.HOUR_OF_DAY,0);
        cal.set(Calendar.MINUTE,0);
        cal.set(Calendar.SECOND,0);
        return  cal.getTime();
    }

    /**
     * 获取当前日期的零点
     * @return 当前日期零点 yyyy-MM-dd 00:00:00
     */
    public static Date getDayBeginTime(){
        return getDayBeginTime(null);
    }

    /**
     * 获取传入日期的最后一秒
     * @param day
     * @return yyyy-MM-dd 23:59:59
     */
    public static Date getDayEndTime(Date day){
        Calendar cal =Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        cal.set(Calendar.HOUR_OF_DAY,23);
        cal.set(Calendar.MINUTE,59);
        cal.set(Calendar.SECOND,59);
        return  cal.getTime();
    }

    /**
     * 获取当前日期的最后一秒
     * @return yyyy-MM-dd 23:59:59
     */
    public static Date getDayEndTime(){
        return  getDayEndTime(null);
    }

    /**
     * 获取传入月份的第一天
     * @param day
     * @return yyyy-MM-01 HH:mm:ss
     */
    public static Date getFirstDayByMonth(Date day){
        Calendar cal =Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        cal.set(Calendar.DAY_OF_MONTH,1);
        return  cal.getTime();
    }
    /**
     * 获取当前月份的第一天
     * @return yyyy-MM-01 HH:mm:ss
     */
    public static Date getFirstDayByMonth(){
        return  getFirstDayByMonth(null);
    }

    /**
     * 获取传入月份的最后一天
     * @param day
     * @return yyyy-MM-[28|29|30|31] HH:mm:ss
     */
    public static Date getLastDayByMonth(Date day){
        Calendar cal =Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        cal.add(Calendar.MONTH,1);
        cal.set(Calendar.DAY_OF_MONTH,1);
        cal.add(Calendar.DAY_OF_MONTH,-1);
        return  cal.getTime();
    }
    /**
     * 获取当前月份的最后一天
     * @return yyyy-MM-[28|29|30|31] HH:mm:ss
     */
    public static Date getLastDayByMonth(){
        return  getLastDayByMonth(null);
    }

    /**
     * 获取传入月份开始时间
     * @param day
     * @return yyyy-MM-01 00:00:00
     */
    public static Date getBeginTimeByMonth(Date day){
        return getDayBeginTime(getFirstDayByMonth(day));
    }
    /**
     * 获取当前月份开始时间
     * @return yyyy-MM-01 00:00:00
     */
    public static Date getBeginTimeByMonth(){
        return getBeginTimeByMonth(null);
    }
    /**
     * 获取传入月份结束时间
     * @param day
     * @return yyyy-MM-01 23:59:59
     */
    public static Date getEndTimeByMonth(Date day){
        return getDayEndTime(getLastDayByMonth(day));
    }
    /**
     * 获取当前月份结束时间
     * @return yyyy-MM-01 23:59:59
     */
    public static Date getEndTimeByMonth(){
        return getEndTimeByMonth(null);
    }

    /**
     * 获取传入日期所在的星期几
     * @param day
     * @param num 1-7
     * @return
     */
    public static Date getWeekDay(Date day,Integer num){
        Calendar cal = Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        if(Objects.isNull(num) || num<1 || num >7){
            logger.warn("DateUtils-getMondayDate,num不在正确的取值范围内.num:{}",num);
            throw new CustomException(ResultCode.DATA_ERROR.getCode(),ResultCode.DATA_ERROR.getMessage());
        }
        if(num<=6){
            num =num+1;
        }else if(num ==7){
            num =1;
        }
        if(num ==1){
            cal.add(Calendar.DAY_OF_MONTH,7);
        }
        cal.set(Calendar.DAY_OF_WEEK,num);
        return cal.getTime();
    }
    /**
     * 获取所在日期所在的星期几
     * @param num 1-7
     * @return
     */
    public static Date getWeekDay(Integer num){
        return getWeekDay(null,num);
    }

    /**
     * 获取传入日期是星期几
     * @param day
     * @return 一、二、三、四、五、六、日
     */
    public static String getWeekDayZN(Date day){
        Calendar cal = Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        int num =cal.get(Calendar.DAY_OF_WEEK);
        Map<Integer,String> map =new HashMap<Integer,String>();
        map.put(1,"日");
        map.put(2,"一");
        map.put(3,"二");
        map.put(4,"三");
        map.put(5,"四");
        map.put(6,"五");
        map.put(7,"六");
        return map.get(num);
    }
    /**
     * 获取当前日期是星期几
     * @return 一、二、三、四、五、六、日
     */
    public static String getWeekDayZN(){
        return getWeekDayZN(null);
    }

    /**
     * 获取传入日期是星期几
     * @param day
     * @return 1,2,3,4,5,6,7
     */
    public static int getWeekDayNum(Date day){
        Calendar cal = Calendar.getInstance();
        if(!Objects.isNull(day)){
            cal.setTime(day);
        }
        int num =cal.get(Calendar.DAY_OF_WEEK);
        if(num==1){
            num =7;
        }else{
            num =num -1;
        }
        return num;
    }

    /**
     * 获取当前日期是星期几
     * @return 1,2,3,4,5,6,7
     */
    public static int getWeekDayNum(){
        return getWeekDayNum(null);
    }
}
