package com.codeupsoft.base.common;

import java.util.Objects;
import lombok.extern.slf4j.Slf4j;

/**
 * 环境工具.
 *
 * @author Liu,Dongdong
 */
@Slf4j
public class Profile {

  /** 当前环境. */
  public static String CURRENT = getCurrentActive();

  /**
   * 判断当前是否为开发环境.
   *
   * @return true表示是开发环境，false表示不是
   */
  public static boolean isDev() {
    return ProfileEnv.DEV.name().equalsIgnoreCase(CURRENT);
  }

  /**
   * 判断当前是否为生产环境.
   *
   * @return true表示是生产环境，false表示不是
   */
  public static boolean isProd() {
    return ProfileEnv.PROD.name().equalsIgnoreCase(CURRENT);
  }

  /**
   * 判断当前是否为测试环境.
   *
   * @return true表示是测试环境，false表示不是
   */
  public static boolean isTest() {
    return ProfileEnv.TEST.name().equalsIgnoreCase(CURRENT);
  }

  /**
   * 判断当前是否为集成环境.
   *
   * @return true表示是集成环境，false表示不是
   */
  public static boolean isItg() {
    return ProfileEnv.ITG.name().equalsIgnoreCase(CURRENT);
  }

  /**
   * 判断当前是否为用户验收测试环境.
   *
   * @return true表示是UAT环境，false表示不是
   */
  public static boolean isUat() {
    return ProfileEnv.UAT.name().equalsIgnoreCase(CURRENT);
  }

  /**
   * 判断当前是否为本地环境.
   *
   * @return true表示是本地环境，false表示不是
   */
  public static boolean isLocal() {
    return ProfileEnv.LOCAL.name().equalsIgnoreCase(CURRENT);
  }

  /**
   * 获取当前激活的环境配置.
   *
   * <p>优先级顺序如下： 1. 启动参数中的 SPRING_APPLICATION_ARGS 2. 系统属性 spring.profiles.active 3. 环境变量
   * spring_profiles_active 4. 默认使用 LOCAL 环境
   *
   * @return 当前激活的环境名称
   */
  private static String getCurrentActive() {
    // 尝试从启动参数中获取
    final String args = System.getenv("SPRING_APPLICATION_ARGS");
    if (Objects.nonNull(args)) {
      final String[] split = args.split(" ");
      for (String str : split) {
        if (str.toLowerCase().startsWith("--spring.profiles.active=")) {
          return str.split("=")[1];
        }
      }
    }
    // 尝试从系统属性中获取
    final String systemProperty = System.getProperty("spring.profiles.active");
    if (Objects.nonNull(systemProperty) && !systemProperty.isEmpty()) {
      return systemProperty;
    }
    // 尝试从环境变量中获取
    final String envProperty = System.getenv("spring_profiles_active");
    if (Objects.nonNull(envProperty) && !envProperty.isEmpty()) {
      return envProperty;
    }
    // 设置默认启动环境
    final String localProfile = ProfileEnv.LOCAL.name().toLowerCase();
    System.setProperty("spring.profiles.active", localProfile);
    return localProfile;
  }

  /** 支持的环境枚举. */
  public enum ProfileEnv {
    /** 开发环境. */
    DEV,
    /** 测试环境. */
    TEST,
    /** 生产环境. */
    PROD,
    /** 集成环境. */
    ITG,
    /** 用户验收测试环境. */
    UAT,
    /** 本地环境. */
    LOCAL;
  }
}
