package com.codeupsoft.base.common.enums;

import java.util.Objects;
import lombok.AllArgsConstructor;
import lombok.Getter;

/**
 * 字符串形式的布尔值枚举，用于表示真/假状态.
 *
 * <p>该枚举定义了多种表示“真”和“假”的字符串形式，包括：
 *
 * <ul>
 *   <li>{@code TRUE} 和 {@code FALSE}
 *   <li>{@code Y} 和 {@code N}
 *   <li>{@code 1} 和 {@code 0}
 * </ul>
 *
 * <p>这些常量可用于在不同系统或模块之间统一布尔值的字符串表示.
 *
 * @author Liu,Dongdong
 */
@AllArgsConstructor
@Getter
@SuppressWarnings("unused")
public enum StrTrueValidated implements BaseEnum<String> {
  STR_TRUE("TRUE", "是"),
  STR_FALSE("FALSE", "否"),
  STR_SYMBOL_Y("Y", "是"),
  STR_SYMBOL_N("N", "否"),
  STR_ONE("1", "是"),
  STR_ZERO("0", "否"),
  ;

  private final String value;
  private final String description;

  /**
   * 判断给定的字符串是否代表“真”.
   *
   * <p>此方法会忽略大小写进行比较，并认为以下值为“真”： {@code "TRUE"}, {@code "1"}, {@code "Y"}（不区分大小写）.
   *
   * @param value 待判断的字符串，可以为 {@code null}
   * @return 如果字符串代表“真”，则返回 {@code true}；否则返回 {@code false}
   */
  public static boolean isTrue(String value) {
    if (Objects.isNull(value)) {
      return false;
    }
    return STR_TRUE.getValue().equalsIgnoreCase(value)
        || STR_ONE.getValue().equalsIgnoreCase(value)
        || STR_SYMBOL_Y.getValue().equalsIgnoreCase(value);
  }
}
