package com.codeupsoft.base.common.exception;

import com.codeupsoft.base.common.response.R;
import com.codeupsoft.base.common.response.Response;
import java.io.Serial;
import lombok.Getter;

/**
 * 自定义基础异常类，用于封装业务异常信息.
 *
 * @author Liu,Dongdong
 * @since 1.0.0
 */
@SuppressWarnings("unused")
public class CodeupSoftBaseException extends RuntimeException {

  @Serial private static final long serialVersionUID = 989881684047262229L;

  @Getter private final R<Void> response;

  /**
   * 构造函数，使用默认错误码和指定的错误消息.
   *
   * @param message 错误消息
   */
  public CodeupSoftBaseException(String message) {
    super(message);
    response = R.error(message);
  }

  /**
   * 构造函数，使用默认错误码、指定的错误消息和原因.
   *
   * @param message 错误消息
   * @param cause 异常原因
   */
  public CodeupSoftBaseException(String message, Throwable cause) {
    super(message, cause);
    response = R.error(message);
  }

  /**
   * 构造函数，使用指定的错误码和错误消息.
   *
   * @param code 错误码
   * @param message 错误消息
   */
  public CodeupSoftBaseException(int code, String message) {
    super(message);
    response = R.error(code, message);
  }

  /**
   * 构造函数，使用指定的错误码、错误消息和原因.
   *
   * @param code 错误码
   * @param message 错误消息
   * @param cause 异常原因
   */
  public CodeupSoftBaseException(int code, String message, Throwable cause) {
    super(message, cause);
    response = R.error(code, message);
  }

  /**
   * 构造函数，基于响应对象构建异常.
   *
   * @param response 响应对象
   */
  public CodeupSoftBaseException(Response<Integer, Void> response) {
    super(response.getMessage());
    this.response = new R<>(response.getCode(), response.getMessage(), null);
  }

  /**
   * 构造函数，基于响应对象和异常原因构建异常.
   *
   * @param response 响应对象
   * @param cause 异常原因
   */
  public CodeupSoftBaseException(Response<Integer, Void> response, Throwable cause) {
    super(response.getMessage(), cause);
    this.response = new R<>(response.getCode(), response.getMessage(), null);
  }
}
