package com.codeupsoft.base.common.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;
import lombok.experimental.UtilityClass;

/**
 * 树结构工具类. 提供构建树、遍历树、查找节点等常用操作方法.
 *
 * @author Liu,Dongdong
 */
@SuppressWarnings("unused")
@UtilityClass
public class TreeUtils {

  /**
   * 将列表结构转换为树形结构.
   *
   * @param list 所有节点列表
   * @param idGetter 获取节点ID的函数
   * @param parentIdGetter 获取节点父ID的函数
   * @param childrenSetter 设置节点子节点列表的函数
   * @param rootParentId 根节点的父ID值
   * @param <T> 节点类型
   * @return 树形结构列表
   */
  public static <T> List<T> listToTree(
      List<T> list,
      Function<T, Object> idGetter,
      Function<T, Object> parentIdGetter,
      TreeChildrenSetter<T> childrenSetter,
      Object rootParentId) {
    if (list == null || list.isEmpty()) {
      return new ArrayList<>();
    }

    List<T> tree = new ArrayList<>();
    for (T node : list) {
      if (Objects.equals(parentIdGetter.apply(node), rootParentId)) {
        tree.add(node);
        buildTree(node, list, idGetter, parentIdGetter, childrenSetter);
      }
    }
    return tree;
  }

  /**
   * 递归构建树形结构.
   *
   * @param parent 父节点
   * @param list 所有节点列表
   * @param idGetter 获取节点ID的函数
   * @param parentIdGetter 获取节点父ID的函数
   * @param childrenSetter 设置节点子节点列表的函数
   * @param <T> 节点类型
   */
  private static <T> void buildTree(
      T parent,
      List<T> list,
      Function<T, Object> idGetter,
      Function<T, Object> parentIdGetter,
      TreeChildrenSetter<T> childrenSetter) {
    List<T> children = new ArrayList<>();
    Object parentId = idGetter.apply(parent);

    for (T node : list) {
      if (Objects.equals(parentIdGetter.apply(node), parentId)) {
        children.add(node);
        buildTree(node, list, idGetter, parentIdGetter, childrenSetter);
      }
    }

    childrenSetter.setChildren(parent, children);
  }

  /**
   * 在树结构中查找满足条件的节点.
   *
   * @param treeList 树节点列表
   * @param condition 查找条件
   * @param childrenGetter 获取子节点列表的函数
   * @param <T> 节点类型
   * @return 满足条件的节点，未找到返回null
   */
  public static <T> T findNode(
      List<T> treeList, Predicate<T> condition, Function<T, List<T>> childrenGetter) {
    if (treeList == null || treeList.isEmpty()) {
      return null;
    }

    for (T node : treeList) {
      if (condition.test(node)) {
        return node;
      }

      List<T> children = childrenGetter.apply(node);
      if (children != null && !children.isEmpty()) {
        T found = findNode(children, condition, childrenGetter);
        if (found != null) {
          return found;
        }
      }
    }

    return null;
  }

  /**
   * 遍历树结构并对每个节点执行操作.
   *
   * @param treeList 树节点列表
   * @param visitor 节点访问器
   * @param childrenGetter 获取子节点列表的函数
   * @param <T> 节点类型
   */
  public static <T> void traverseTree(
      List<T> treeList, TreeVisitor<T> visitor, Function<T, List<T>> childrenGetter) {
    if (treeList == null || treeList.isEmpty()) {
      return;
    }

    for (T node : treeList) {
      visitor.visit(node);
      List<T> children = childrenGetter.apply(node);
      if (children != null && !children.isEmpty()) {
        traverseTree(children, visitor, childrenGetter);
      }
    }
  }

  /**
   * 过滤树结构，返回满足条件的节点组成的新树.
   *
   * @param treeList 树节点列表
   * @param condition 过滤条件
   * @param childrenGetter 获取子节点列表的函数
   * @param <T> 节点类型
   * @return 过滤后的新树
   */
  public static <T> List<T> filterTree(
      List<T> treeList, Predicate<T> condition, Function<T, List<T>> childrenGetter) {
    if (treeList == null || treeList.isEmpty()) {
      return new ArrayList<>();
    }

    List<T> result = new ArrayList<>();
    for (T node : treeList) {
      List<T> children = childrenGetter.apply(node);
      List<T> filteredChildren = filterTree(children, condition, childrenGetter);

      // 如果节点满足条件或有满足条件的子节点，则保留该节点
      if (condition.test(node) || !filteredChildren.isEmpty()) {
        // 这里需要创建节点副本或者有其他方式来创建新节点
        // 此处简化处理，实际使用时需要根据具体情况实现
        result.add(node);
      }
    }

    return result;
  }

  /**
   * 树节点子节点设置器接口.
   *
   * @param <T> 节点类型
   */
  @FunctionalInterface
  public interface TreeChildrenSetter<T> {
    /**
     * 设置节点的子节点列表.
     *
     * @param node 节点
     * @param children 子节点列表
     */
    void setChildren(T node, List<T> children);
  }

  /**
   * 树节点访问器接口.
   *
   * @param <T> 节点类型
   */
  @FunctionalInterface
  public interface TreeVisitor<T> {
    /**
     * 访问节点.
     *
     * @param node 节点
     */
    void visit(T node);
  }
}
