package com.comparizen.client;

import com.cedarsoftware.util.io.JsonReader;
import com.comparizen.client.model.TestRunStatus;
import com.comparizen.client.model.TestRunStatusResponse;
import com.comparizen.client.util.ImageType;
import com.comparizen.client.util.MultipartBodyPublisher;
import com.comparizen.client.util.UnknownMimeTypeException;
import lombok.Builder;

import java.awt.image.BufferedImage;
import java.io.IOException;
import java.math.BigInteger;
import java.net.InetSocketAddress;
import java.net.ProxySelector;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.net.http.HttpResponse.BodyHandlers;
import java.nio.file.Path;
import java.util.Map;
import java.util.Random;
import java.util.stream.Collectors;

@SuppressWarnings("unused")
public class ComparizenClient {

    private static final String INVALID_JSON_ERROR_MESSAGE = "Received unexpected response from Comparizen server. Check your internet connection. " +
            "If this problem persist, see the Comparizen website if there's a newer version of this client.";

    @SuppressWarnings("FieldCanBeLocal")
    private final String DEFAULT_URL = "https://app.comparizen.com";

    private final HttpClient client;
    private final String url;
    private final String apiKey;

    @Builder
    private ComparizenClient(String apiKey, InetSocketAddress proxy, String customURL) {
        this.apiKey = apiKey;

        HttpClient.Builder builder = HttpClient.newBuilder();
        if (proxy != null) {
            builder.proxy(ProxySelector.of(proxy));
        }
        this.url = customURL == null ? DEFAULT_URL : customURL;
        this.client = builder.build();
    }

    /**
     * Creates a new Comparizen test run
     *
     * @param projectId the project's ID where the test run should be created.
     * @return the new test run's ID
     * @throws ComparizenClientException when the test run could not be created
     */
    public String createTestRun(String projectId) throws ComparizenClientException {
        return this.createTestRun(projectId, null);
    }

    /**
     * Creates a new Comparizen test run
     *
     * @param projectId the project's ID where the test run should be created.
     * @param name      a name describing this test run (optional)
     * @return the new test run's ID
     * @throws ComparizenClientException when the test run could not be created
     */
    public String createTestRun(String projectId, String name) throws ComparizenClientException {
        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(url + "/rest/testrun"))
                    .POST(HttpRequest.BodyPublishers.ofString("{" + // \\\\\
                            (name == null ? "" : "\"name\": \"" + name.replaceAll("\"", "\\\\\"") + "\"," ) +
                            "\"projectId\": \"" + projectId + "\"," +
                            "\"apiKey\": \"" + apiKey + "\"" +
                            "}"))
                    .header("Content-Type", "application/json")
                    .build();

            HttpResponse<String> response = client.send(request, BodyHandlers.ofString());
            int responseStatusCode = response.statusCode();
            if (responseStatusCode >= 400) {
                throw new ComparizenClientException("Something went wrong while creating a test run: " +
                        "server responded with status code " + responseStatusCode + ": " + response.body());

            }
            return getStringFromResponse(response, "id");
        } catch (final InterruptedException | IOException e) {
            throw new ComparizenClientException("Something went wrong while creating a test run: " +
                    "an exception occurred while communicating with the Comparizen server: " + e.getMessage(), e);
        }
    }

    /**
     * Finalizes a Comparizen test run. Finalizing a test run marks the test run as 'completed' and
     * will prevent the test run from accepting new screenshots.
     *
     * @param testRunId the test run's ID
     * @throws ComparizenClientException when the test run could not be finalized
     */
    public void finalizeTestRun(String testRunId) throws ComparizenClientException {
        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(url + "/rest/testrun/" + testRunId + "/finalize"))
                    .POST(HttpRequest.BodyPublishers.ofString("{" +
                            "\"apiKey\": \"" + apiKey + "\"" +
                            "}"))
                    .header("Content-Type", "application/json")
                    .build();

            HttpResponse<String> response = client.send(request, BodyHandlers.ofString());
            int responseStatusCode = response.statusCode();
            if (responseStatusCode >= 400) {
                throw new ComparizenClientException("Something went wrong while finalizing test run: " +
                        "server responded with status code " + responseStatusCode + ": " + response.body());

            }
        } catch (final InterruptedException | IOException e) {
            throw new ComparizenClientException("Something went wrong while finalizing test run: " +
                    "an exception occurred while communicating with the Comparizen server: " + e.getMessage(), e);
        }
    }

    /**
     * Adds a new comparison (screenshot) to a test run
     *
     * @param testRunId the test run's ID
     * @param name      the name of the screenshot. This screenshot will be compared to a screenshot from the testrun project's baseline with the same name.
     * @param file      path to the screenshot file
     * @return the ID of the newly created comparison
     * @throws ComparizenClientException when something went wrong while uploading the screenshot or creating the comparison
     */
    public String createComparison(String testRunId, String name, Path file) throws ComparizenClientException {
        return performUpload(testRunId, name, file, null, null);
    }

    /**
     * Adds a new comparison (screenshot) to a test run
     *
     * @param testRunId     the test run's ID
     * @param name          the name of the screenshot. This screenshot will be compared to a screenshot from the testrun project's baseline with the same name.
     * @param bufferedImage A BufferedImage containing image data
     * @param imageType     an ImageType.
     * @return the ID of the newly created comparison
     * @throws ComparizenClientException when something went wrong while uploading the screenshot or creating the comparison
     */
    public String createComparison(String testRunId, String name, BufferedImage bufferedImage, ImageType imageType) throws ComparizenClientException {
        return performUpload(testRunId, name, bufferedImage, imageType, null);
    }

    /**
     * Adds a new comparison (screenshot) to a test run
     *
     * @param testRunId  the test run's ID
     * @param name       the name of the screenshot. This screenshot will be compared to a screenshot from the testrun project's baseline with the same name.
     * @param file       path to the screenshot file
     * @param properties additional properties for this comparison, such as tag names.
     * @return the ID of the newly created comparison
     * @throws ComparizenClientException when something went wrong while uploading the screenshot or creating the comparison
     */
    public String createComparison(String testRunId, String name, Path file, ComparisonProperties properties) throws ComparizenClientException {
        return performUpload(testRunId, name, file, null, properties);
    }

    /**
     * Adds a new comparison (screenshot) to a test run
     *
     * @param testRunId     the test run's ID
     * @param name          the name of the screenshot. This screenshot will be compared to a screenshot from the testrun project's baseline with the same name.
     * @param bufferedImage A BufferedImage containing image data
     * @param imageType     an ImageType.
     * @param properties    additional properties for this comparison, such as tag names.
     * @return the ID of the newly created comparison
     * @throws ComparizenClientException when something went wrong while uploading the screenshot or creating the comparison
     */
    public String createComparison(String testRunId, String name, BufferedImage bufferedImage, ImageType imageType, ComparisonProperties properties) throws ComparizenClientException {
        return performUpload(testRunId, name, bufferedImage, imageType, properties);
    }

    private String performUpload(String testRunId, String name, Object bufferedImageOrPath, ImageType forcedImageType, ComparisonProperties properties) throws ComparizenClientException {
        if (!(bufferedImageOrPath instanceof BufferedImage) && !(bufferedImageOrPath instanceof Path)) {
            throw new ComparizenClientException("Screenshot upload can only be done using a Path or BufferedImage object");
        }

        try {
            String boundary = "===imagecompare-upload-boundary-" + new BigInteger(256, new Random()).toString();
            Map<Object, Object> postParams = Map.of(
                    "apiKey", apiKey,
                    "testRunId", testRunId,
                    "name", name,
                    "tags", extractCommaSeparatedTags(properties),
                    "file", bufferedImageOrPath
            );
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(url + "/rest/comparison"))
                    .header("Content-Type", "multipart/form-data;boundary=" + boundary)
                    .header("content-transfer-encoding", "binary")
                    .POST(MultipartBodyPublisher.ofMimeMultipartData(postParams, boundary, forcedImageType))
                    .build();

            HttpResponse<String> response = client.send(request, BodyHandlers.ofString());
            int responseStatusCode = response.statusCode();
            if (responseStatusCode >= 400) {
                throw new ComparizenClientException("Something went wrong while uploading a screenshot: " +
                        "server responded with status code " + responseStatusCode + ": " + response.body());
            }

            return getStringFromResponse(response, "comparisonId");
        } catch (final InterruptedException | IOException e) {
            throw new ComparizenClientException("Something went wrong while uploading a screenshot: " +
                    "an exception occurred while communicating with the Comparizen server: " + e.getMessage(), e);
        } catch (final UnknownMimeTypeException e) {
            throw new ComparizenClientException("Something went wrong while uploading a screenshot: " +
                    "an invalid or unknown mime type was attached to the upload content: " + e.getMessage(), e);
        }

    }

    /**
     * Retrieves a test run's status
     *
     * @param testRunId the test run's ID
     * @return a TestRunStatusResponse describing the test run's status
     * @throws ComparizenClientException when something went wrong while retrieving the test run's status
     */
    public TestRunStatusResponse getTestRunStatus(final String testRunId) throws ComparizenClientException {
        try {
            HttpRequest request = HttpRequest.newBuilder()
                    .uri(URI.create(url + "/rest/testrun/" + testRunId + "/status?apiKey=" + this.apiKey))
                    .GET()
                    .build();

            HttpResponse<String> response = client.send(request, BodyHandlers.ofString());
            int responseStatusCode = response.statusCode();
            if (responseStatusCode >= 400) {
                throw new ComparizenClientException("Something went wrong while retrieving test run status: " +
                        "server responded with status code " + responseStatusCode + ": " + response.body());

            }

            @SuppressWarnings("rawtypes") final Map responseMap = JsonReader.jsonToMaps(response.body());
            if (responseMap == null || !responseMap.containsKey("status") || !responseMap.containsKey("url")) {
                throw new ComparizenClientException(INVALID_JSON_ERROR_MESSAGE);
            }

            return TestRunStatusResponse.builder()
                    .testRunStatus(TestRunStatus.fromString(String.valueOf(responseMap.get("status"))))
                    .url(String.valueOf(responseMap.get("url")))
                    .build();
        } catch (final InterruptedException | IOException e) {
            throw new ComparizenClientException("Something went wrong while retrieving test run status: " +
                    "an exception occurred while communicating with the Comparizen server: " + e.getMessage(), e);
        }
    }

    /**
     * Waits until a test run is either finalized or the given timeout is passed.
     * This method will poll the Comparizen service every 500 ms.
     *
     * @param testRunId a test run's ID
     * @param timeoutMs the maximum amount of milliseconds to wait.
     * @return a TestRunStatusResponse
     * @throws ComparizenClientException when timeout exceeded or something went wrong while retrieving the test run status
     * @throws InterruptedException      when the current Thread is interrupted
     */
    @SuppressWarnings("BusyWait")
    public TestRunStatusResponse waitUntilTestRunResult(final String testRunId, final long timeoutMs) throws ComparizenClientException, InterruptedException {
        long startTime = System.currentTimeMillis();

        while (startTime + timeoutMs >= System.currentTimeMillis()) {
            TestRunStatusResponse response = getTestRunStatus(testRunId);

            if (response.getTestRunStatus() != TestRunStatus.PROCESSING) {
                return response;
            }
            Thread.sleep(500);
        }

        throw new ComparizenClientException("Test run with id " + testRunId + " was still processing after " + timeoutMs + " ms passed.");
    }

    private String extractCommaSeparatedTags(final ComparisonProperties props) {
        if (props == null || props.getTagNames() == null) {
            return "";
        }

        return props.getTagNames().stream()
                .map(t -> t.replaceAll(",", ""))
                .collect(Collectors.joining(","));
    }

    @SuppressWarnings("rawtypes")
    private String getStringFromResponse(final HttpResponse<String> response, final String fieldKey) throws ComparizenClientException {
        try {
            final Map responseMap = JsonReader.jsonToMaps(response.body());
            if (responseMap == null || !responseMap.containsKey(fieldKey)) {
                throw new ComparizenClientException(INVALID_JSON_ERROR_MESSAGE);
            }

            return String.valueOf(responseMap.get(fieldKey));
        } catch (Exception e) {
            throw new ComparizenClientException(INVALID_JSON_ERROR_MESSAGE);
        }
    }


}
