package com.comparizen.client.util;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.net.http.HttpRequest;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.Random;

public class MultipartBodyPublisher {

    public static HttpRequest.BodyPublisher ofMimeMultipartData(Map<Object, Object> data,
                                                                String boundary, ImageType forcedImageType) throws IOException, UnknownMimeTypeException {
        var resultBytes = new ArrayList<byte[]>();
        byte[] fieldSeparator = ("--" + boundary + "\r\nContent-Disposition: form-data; name=")
                .getBytes(StandardCharsets.UTF_8);

        for (Map.Entry<Object, Object> entry : data.entrySet()) {
            resultBytes.add(fieldSeparator);

            if (entry.getValue() instanceof Path) {
                Path path = (Path) entry.getValue();
                String mimeType =  forcedImageType != null ? forcedImageType.getMimeType() : Optional.ofNullable(Files.probeContentType(path)).orElseThrow(UnknownMimeTypeException::new);
                resultBytes.add(("\"" + entry.getKey() + "\"; filename=\"" + path.getFileName()
                        + "\"\r\nContent-Type: " + mimeType + "\r\n\r\n").getBytes(StandardCharsets.UTF_8));
                resultBytes.add(Files.readAllBytes(path));
                resultBytes.add("\r\n".getBytes(StandardCharsets.UTF_8));
            }
            if (entry.getValue() instanceof BufferedImage) {
                if (forcedImageType == null) {
                    throw new UnknownMimeTypeException("Missing 'imageType' for BufferedImage. When using BufferedImage type, you must specifiy an ImageType as well.");
                }
                BufferedImage image = (BufferedImage) entry.getValue();
                String fileName = new BigInteger(256, new Random()).toString();

                resultBytes.add(("\"" + entry.getKey() + "\"; filename=\"" + fileName
                        + "\"\r\nContent-Type: " + forcedImageType.getMimeType() + "\r\n\r\n").getBytes(StandardCharsets.UTF_8));

                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                ImageIO.write(image, forcedImageType.getImageIoType(), baos);

                baos.flush();
                resultBytes.add(baos.toByteArray());
                baos.close();

                resultBytes.add("\r\n".getBytes(StandardCharsets.UTF_8));
            }
            else {
                resultBytes.add(("\"" + entry.getKey() + "\"\r\n\r\n" + entry.getValue() + "\r\n")
                        .getBytes(StandardCharsets.UTF_8));
            }
        }
        resultBytes.add(("--" + boundary + "--").getBytes(StandardCharsets.UTF_8));
        return HttpRequest.BodyPublishers.ofByteArrays(resultBytes);
    }
}
