/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.callbacks;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * An extension of {@link Callback} that allows many callbacks for each call
 * context.
 * <P>
 * The server may return zero, one, or many results for a call; each result will
 * be supplied to one of the callback methods. When no further results are
 * expected for the call context, {@link #onClose()} will be called. If a
 * context is closed prematurely (for example, due to the session being closed),
 * {@link #onError(ErrorReason)} will be called. One of {@code onClose} or
 * {@code onError} will be the final callback made for a call context.
 *
 * @author DiffusionData Limited
 * @since 5.1
 */
public interface Stream extends Callback {

    /**
     * Notification that a stream context was closed normally.
     * <P>
     * No further calls will be made for the stream context.
     */
    void onClose();

    /**
     * Abstract default callback.
     * <P>
     * This simply logs onClose calls at 'debug' level. This method may be
     * overridden to provide more specific processing.
     */
    abstract class Default
        extends Callback.Default
        implements Stream {

        private static final Logger LOG =
            LoggerFactory.getLogger(Stream.Default.class);

        @Override
        public void onClose() {
            LOG.debug("{} stream closed", this);
        }
    }
}
