/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features;

import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.types.Credentials;
import com.pushtechnology.diffusion.client.types.GlobalPermission;
import com.pushtechnology.diffusion.client.types.PathPermission;
import com.pushtechnology.diffusion.client.types.TopicPermission;

/**
 * This feature provides a client session with the ability to change its
 * associated principal as well as to query permissions assigned to it.
 *
 * <H3>Access control</H3>
 * There are no permissions requirements associated with this feature. The
 * client provides credentials that authenticate the new principal.
 *
 * <H3>Accessing the feature</H3>
 * This feature may be obtained from a {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * Security security = session.feature(Security.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface Security extends Feature {

    /**
     * Change the security principal associated with the session.
     *
     * @param principal the new principal name
     * @param credentials the credentials authenticating the new principal
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If authentication succeeded and the session principal has been
     *         changed, the CompletableFuture will complete successfully with a
     *         {@link Boolean#TRUE TRUE} result.
     *
     *         <p>
     *         If authentication failed because the {@code principal} was
     *         unknown or the {@code credentials} were invalid, the session's
     *         principal has not changed and the CompletableFuture will complete
     *         successfully with a {@link Boolean#FALSE FALSE} result.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<Boolean> changePrincipal(
        String principal,
        Credentials credentials);

    /**
     * Change the security principal associated with the session.
     * <p>
     * If authentication fails, the session's current principal will not be
     * changed.
     *
     * @param principal the new principal name
     * @param credentials the credentials authenticating the new principal
     * @param callback called on success or failure
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void changePrincipal(
        String principal,
        Credentials credentials,
        ChangePrincipalCallback callback);

    /**
     * Change the security principal associated with the session.
     * <p>
     * If authentication fails, the session's current principal will not be
     * changed.
     *
     * @param principal the new principal name
     * @param credentials credentials authenticating the new principal
     * @param context passed to the callback
     * @param callback called on success or failure
     * @param <C> the context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void changePrincipal(
        String principal,
        Credentials credentials,
        C context,
        ChangePrincipalContextCallback<C> callback);

    /**
     * Query the global permissions assigned to the calling session.
     *
     * @return a CompletableFuture which completes when the response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully with the session's global permissions.
     *
     * @since 6.3
     */
    CompletableFuture<Set<GlobalPermission>> getGlobalPermissions();

    /**
     * Query the path permissions assigned to the calling session on a given
     * path.
     *
     * @param path the path to query for permissions
     * @return a CompletableFuture which completes when the response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully with the session's permissions for
     *         {@code path}.
     *
     * @since 6.3
     * @deprecated since 6.5. Use {@link #getPermissionsForPath(String)} instead.
     */
    @Deprecated
    CompletableFuture<Set<TopicPermission>> getPathPermissions(String path);

    /**
     * Query the path permissions assigned to the calling session on a given
     * path.
     *
     * @param path the path to query for permissions
     * @return a CompletableFuture which completes when the response is received
     *         from the server.
     *
     *         <p>
     *         If the request was successful, the CompletableFuture will
     *         complete successfully with the session's permissions for
     *         {@code path}.
     *
     * @since 6.5
     */
    CompletableFuture<Set<PathPermission>> getPermissionsForPath(String path);

    /**
     * The callback interface for
     * {@link Security#changePrincipal(String, Credentials, ChangePrincipalCallback)}
     * .
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface ChangePrincipalCallback extends Callback {

        /**
         * This will be called when the session principal has been successfully
         * changed.
         */
        void onPrincipalChanged();

        /**
         * This will be called if an attempt to change the session principal
         * failed because authentication failed at the server. This could be
         * because the named principal was unknown or the supplied credentials
         * were invalid.
         */
        void onAuthenticationFailure();

        /**
         * Default implementation of {@link ChangePrincipalCallback}.
         * <P>
         * This simply logs onPrincipalChanged callback at 'debug' level
         * onAuthenticationFailure at 'warn' level. These methods may be
         * overridden as required.
         */
        class Default
            extends Callback.Default
            implements ChangePrincipalCallback {

            private static final Logger LOG =
                LoggerFactory.getLogger(ChangePrincipalCallback.Default.class);

            @Override
            public void onPrincipalChanged() {
                LOG.debug("{} - Principal changed", this);
            }

            @Override
            public void onAuthenticationFailure() {
                LOG.warn("{} - Principal authentication failed", this);
            }

        }
    }

    /**
     * The callback interface for
     * {@link Security#changePrincipal(String, Credentials, Object, ChangePrincipalContextCallback)}
     * .
     * <P>
     * Use this alternative to {@link ChangePrincipalCallback} to associate some
     * arbitrary context object with each call.
     *
     * @param <C> the context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface ChangePrincipalContextCallback<C> extends ContextCallback<C> {

        /**
         * Notification that a request to change the session principal was
         * successful.
         *
         * @param context the context object supplied when making the call. May
         *        be {@code null}
         */
        void onPrincipalChanged(C context);

        /**
         * This will be called if an attempt to change the session principal
         * failed because authentication failed at the server. This could be
         * because the named principal was unknown or the supplied credentials
         * were invalid.
         *
         * @param context the context object supplied when making the call. May
         *        be {@code null}
         */
        void onAuthenticationFailure(C context);

        /**
         * Default implementation of {@link ChangePrincipalContextCallback}.
         * <P>
         * This simply logs onPrincipalChanged callback at 'debug' level
         * onAuthenticationFailure at 'warn' level. These methods may be
         * overridden as required.
         *
         * @param <C> context object type
         */
        class Default<C>
            extends ContextCallback.Default<C>
            implements ChangePrincipalContextCallback<C> {

            private static final Logger LOG =
                LoggerFactory
                    .getLogger(ChangePrincipalContextCallback.Default.class);

            @Override
            public void onPrincipalChanged(C context) {
                LOG.debug("{} - Principal changed, context={}", this, context);
            }

            @Override
            public void onAuthenticationFailure(C context) {
                LOG.warn(
                    "{} - Principal authentication failed, context={}",
                    this,
                    context);
            }
        }
    }
}
