/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/

package com.pushtechnology.diffusion.client.features;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.Diffusion;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl.IncompatibleExistingTopicException;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl.IncompatibleParentTopicException;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl.InvalidTopicPathException;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl.InvalidTopicSpecificationException;
import com.pushtechnology.diffusion.client.features.control.topics.TopicControl.TopicLicenseLimitException;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.topics.details.TopicSpecification;
import com.pushtechnology.diffusion.client.topics.details.TopicType;


/**
 * An update stream provides a method for updating a specific topic.
 *
 * <p>
 * An update stream is associated with a specific topic.
 * Update streams are created using {@link TopicUpdate#newUpdateStreamBuilder()}.
 * It can be created with an optional {@link UpdateConstraint constraint}. An
 * optional {@link TopicSpecification} can be provided on creation.
 * The type of the topic must match the type of values passed to the update stream.
 * The existence of the topic, its type and the constraint are validated lazily
 * by the first {@link UpdateStream#set} or {@link UpdateStream#validate}
 * operation. Subsequent operations issued before the first operation
 * completes will be deferred until the completion of the first operation.
 * <p>
 * An update stream can be used to send any number of updates. It sends a
 * sequence of updates for a specific topic to the server. If supported by the
 * data type, updates will be sent to the server as a stream of binary deltas.
 * An update stream does not prevent other sessions from updating the topic. If
 * exclusive access is required update streams should be used with
 * {@link com.pushtechnology.diffusion.client.session.Session.SessionLock session locks}
 * as constraints.
 * <p>
 * Once validated an update stream can be invalidated. An invalidated
 * update stream rejects the operations applied to it. The update stream
 * will be invalidated if:
 * <ul>
 *     <li>the topic is removed
 *     <li>another update stream is created for the same topic
 *     <li>the topic is updated to a new value by anything other than the stream
 *     <li>the session does not have the
 *     {@link com.pushtechnology.diffusion.client.types.PathPermission#UPDATE_TOPIC
 *     UPDATE_TOPIC} permission
 *     <li>an operation fails because of cluster repartitioning
 * </ul>
 * <p>
 * Update streams are thread-safe.
 *
 * @param <T> type of the value
 * @author DiffusionData Limited
 * @since 6.2
 */
public interface UpdateStream<T> {
    /**
     * Sets the topic to a specified value.
     * <p>
     * The {@code null} value can only be passed to the {@code value}
     * parameter when updating {@link TopicType#STRING string},
     * {@link TopicType#INT64 int64}, or {@link TopicType#DOUBLE double} topics.
     * <p>
     * When a {@link TopicType#STRING string}, {@link TopicType#INT64 int64}, or
     * {@link TopicType#DOUBLE double} topic is set to {@code null}, the topic
     * will be updated to have no value. If a previous value was present
     * subscribers will receive a notification that the new value is
     * {@code null}. New subscribers will not receive a value notification.
     * <p>
     * The first call to this method may fail with
     * {@link NoSuchTopicException} or {@link IncompatibleTopicException}.
     * Subsequent calls may fail with {@link InvalidUpdateStreamException}.
     * Any call can fail with {@link ClusterRoutingException},
     * {@link PermissionsException} or {@link SessionClosedException}.
     * <p>
     * If a {@link UpdateConstraint constraint} was provided when creating the
     * update stream, the first call to this method may also fail with
     * {@link UnsatisfiedConstraintException}.
     * <p>
     * If the update stream was created with a {@link TopicSpecification},
     * the first call to this method may also fail with
     * {@link IncompatibleExistingTopicException} and it will not fail with
     * {@link NoSuchTopicException}.
     * <p>
     * If this method fails all subsequent calls to {@link #set} or
     * {@link #validate} will fail with {@link InvalidUpdateStreamException}.
     *
     * @param value the value. Update streams for
     *              {@link TopicType#STRING string},
     *              {@link TopicType#INT64 int64}, and
     *              {@link TopicType#DOUBLE double}
     *              topics accept {@code null}, as described above. Using
     *              null with other topic types is an error and will result
     *              in a {@link NullPointerException}.
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchTopicException} &ndash; if there is no topic
     *         bound to {@code path};
     *         <li>{@link IncompatibleTopicException} &ndash; if updates
     *         cannot be applied to the topic, for example if a topic view has
     *         bound a reference topic to the path;
     *         <li>{@link IncompatibleTopicStateException} &ndash; if the topic
     *         is managed by a component (such as fan-out) that prohibits
     *         updates from the caller;
     *         <li>{@link UnsatisfiedConstraintException} &ndash; if the
     *         {@code constraint} is not satisfied by the topic {@code path};
     *         <li>{@link InvalidUpdateStreamException} &ndash; the update
     *         stream has been invalidated;
     *         <li>{@link IncompatibleParentTopicException} &ndash; a topic
     *         could not be added because a topic at a parent path is
     *         incompatible;
     *         <li>{@link InvalidTopicPathException} &ndash; {@code topicPath}
     *         is not a valid topic path;
     *         <li>{@link InvalidTopicSpecificationException} &ndash; the
     *         specification is invalid, possibly because mandatory properties
     *         not supplied;
     *         <li>{@link TopicLicenseLimitException} &ndash; the
     *         topic could not be added as it would breach a licensing limit;
     *         <li>{@link ClusterRoutingException} &ndash; if the operation failed
     *         due to a transient cluster error;
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have the {@code MODIFY_TOPIC} or
     *         {@code UPDATE_TOPIC} permission for {@code path};
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     */
    CompletableFuture<TopicCreationResult> set(T value);

    /**
     * Return the latest value of the topic set using this update stream.
     * <p>
     * The returned value reflects the last value that has been set, before it
     * is sent to the server.
     * <p>
     * If the server rejects a set operation, the topic value will not change
     * and this update stream will be invalidated.
     *
     * @return the cached value of the topic
     * @throws IllegalStateException if called before the first call to
     *                               {@link #set}
     */
    T get();

    /**
     * Validates the update stream.
     * <p>
     * Update streams are validated lazily when
     * {@link #set setting the value}. This method allows the stream to be
     * validated before a value needs to be set.
     * <p>
     * If the update stream has not been validated yet, calling this method
     * checks the topic exists, the topic type is correct, the constraint is
     * satisfied and the session has permission to update the topic. Once
     * it has been validated calling this method checks the topic has not been
     * removed, no other stream has been created for the topic, the value
     * of the topic has not been changed by anything else and the session
     * still has permission to update the topic.
     * <p>
     * This method may fail with {@link IncompatibleExistingTopicException}
     * if it is the first call to {@link #validate}, {@link #set} has not
     * been called and a {@link TopicSpecification topic specification} was
     * provided when creating the update stream, otherwise it will never
     * fail with this cause.
     * <p>
     * The first call to this method may fail with
     * {@link NoSuchTopicException} or {@link IncompatibleTopicException}.
     * Subsequent calls may fail with {@link InvalidUpdateStreamException}.
     * Any call can fail with {@link ClusterRoutingException},
     * {@link PermissionsException} or {@link SessionClosedException}.
     * <p>
     * If a {@link UpdateConstraint constraint} was provided when creating the
     * update stream, the first call to this method may also fail with
     * {@link UnsatisfiedConstraintException}.
     * <p>
     * If the update stream was created with a {@link TopicSpecification},
     * the first call to this method may also fail with
     * {@link IncompatibleExistingTopicException} and it will not fail with
     * {@link NoSuchTopicException}.
     * <p>
     * If this method fails all subsequent calls to {@link #set} or
     * {@link #validate} will fail with {@link InvalidUpdateStreamException}.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link NoSuchTopicException} &ndash; if there is no topic
     *         bound to {@code path};
     *         <li>{@link IncompatibleTopicException} &ndash; if updates
     *         cannot be applied to the topic, for example if a topic view has
     *         bound a reference topic to the path;
     *         <li>{@link IncompatibleTopicStateException} &ndash; if the topic
     *         is managed by a component (such as fan-out) that prohibits
     *         updates from the caller;
     *         <li>{@link UnsatisfiedConstraintException} &ndash; if the
     *         {@code constraint} is not satisfied by the topic {@code path};
     *         <li>{@link InvalidUpdateStreamException} &ndash; the update
     *         stream has been invalidated;
     *         <li>{@link ClusterRoutingException} &ndash; if the operation failed
     *         due to a transient cluster error;
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have the {@code MODIFY_TOPIC} or
     *         {@code UPDATE_TOPIC} permission for {@code path};
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     */
    CompletableFuture<TopicCreationResult> validate();

    /**
     * Builder for {@link UpdateStream update stream} to use for updating a
     * specific topic. A builder is created using
     * {@link TopicUpdate#newUpdateStreamBuilder}.
     * <p>
     * The type of the topic being updated must match the type derived from the
     * {@code valueClass} parameter.
     *
     * @since 6.9
     */
    interface Builder {

        /**
         * Specifies a TopicSpecification for this update stream.
         * <p>
         * If a topic does not exist at the {@code path} one will be created using
         * the {@code topicSpecification} when the update stream is validated. If a
         * topic does exist, its specification must match {@code topicSpecification},
         * otherwise the operation will fail with
         * {@link IncompatibleTopicException}.
         * <p>
         * Specification is null by default. Calling this method with a null parameter
         * removes the specification for this builder.
         *
         * @param topicSpecification the required specification of the topic
         * @return this builder
         */
        Builder specification(TopicSpecification topicSpecification);

        /**
         * Specifies an update constraint for this update stream.
         * <p>
         * Constraints can be created using {@link Diffusion#updateConstraints}}.
         * <p>
         * Constraints can be applied to the setting of a value and creation of an
         * update stream. Constraints describe a condition that must be satisfied for
         * the operation to succeed. The constraints are evaluated on the server. The
         * available constraints are: an active session lock, the absence of a topic,
         * the current value of the topic being updated, and a part of the current
         * value of the topic being updated.
         * <p>
         * Constraint is Unconstrained by default. Calling this method with a null parameter
         * resets the constraint for this builder.
         *
         * @param updateConstraint the constraint that must be satisfied for the update
         * stream to be validated
         * @return this builder
         */
        Builder constraint(UpdateConstraint updateConstraint);


        /**
         * Resets the builder to its default parameters.
         *
         * @return this builder
         */
        Builder reset();

        /**
         * Creates an {@link UpdateStream} to use for updating a
         * specific topic.
         * <p>
         * The type of the topic being updated must match the type derived from the
         * {@code valueClass} parameter.
         * <p>
         * Update streams send a sequence of updates for a specific topic. The
         * updates may be delivered to the server as binary deltas. They do not
         * provide exclusive access to the topic. If exclusive access is required
         * update streams should be used with
         * {@link Session.SessionLock session locks} as constraints.
         * <p>
         * Streams are validated lazily when the first {@link UpdateStream#set} or
         * {@link UpdateStream#validate} operation is completed. Once validated a
         * stream can be invalidated, after which it rejects future
         * updates.
         *
         * @param path the path of the topic
         * @param valueClass the type of the values expected by the update stream
         * @param <T> type of the values expected by the update stream
         *
         * @return an update stream
         */
        <T> UpdateStream<T> build(String path, Class<T> valueClass);
    }
}

