/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.clients;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.callbacks.Registration;
import com.pushtechnology.diffusion.client.callbacks.Stream;
import com.pushtechnology.diffusion.client.security.authentication.Authenticator;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.session.SessionException;
import com.pushtechnology.diffusion.client.types.GlobalPermission;

/**
 * This feature allows a client session to authenticate the credentials of other
 * sessions.
 * <P>
 * Sessions are authenticated by a chain of authentication handlers. A client
 * session can participate in the authentication process by creating an
 * authentication handler and registering it with the server.
 * <P>
 * Each authentication handler is registered under a particular handler name.
 * For registration to succeed, the server's security configuration must include
 * a matching {@code control-authentication-handler} entry for the name,
 * otherwise registration will fail and the authentication handler will be
 * closed immediately.
 * <P>
 * Each client session can register a single authentication handler for a given
 * handler name.
 * <P>
 * For each authentication event, the server will use its configuration to
 * determine the handler priority order. The server may call authentication
 * handlers in serial or parallel. The server may stop the authentication
 * process as soon as it has an allow or deny response from an authentication
 * handler and all higher priority authentication handlers have abstained.
 * <P>
 * For a configured control authentication handler, the server will select a
 * single handler from those registered for the handler name. If no
 * authenticators are currently registered, the server will consult the next
 * configured authentication handler in priority order.
 * <H3>Access control</H3>
 * <P>
 * In order to register an authentication handler a session needs both
 * {@link GlobalPermission#REGISTER_HANDLER REGISTER_HANDLER} and
 * {@link GlobalPermission#AUTHENTICATE AUTHENTICATE} permissions.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * AuthenticationControl authenticationControl = session.feature(AuthenticationControl.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface AuthenticationControl extends Feature {

    /**
     * Register an authentication handler for client authentication events.
     *
     * @param handlerName the handler name which must match an entry in the
     *        server's security configuration
     *
     * @param authenticator specifies the authentication handler
     *
     * @return a CompletableFuture that completes when the authentication
     *         handler has been registered, returning a {@link Registration}
     *         which can be used to unregister the authentication handler.
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does
     *         not have {@code REGISTER_HANDLER} or {@code AUTHENTICATE}
     *         permission;
     *
     *         <li>{@link SessionException} &ndash; will occur if the server
     *         configuration does not contain a
     *         {@code control-authentication-handler} element with the given
     *         name.
     *         </ul>
     *
     * @since 6.2
     */
    CompletableFuture<Registration> setAuthenticationHandler(
        String handlerName,
        ControlAuthenticator authenticator);

    /**
     * A control authenticator.
     * @since 6.2
     */
    interface ControlAuthenticator extends Authenticator, Stream {
    }

}
