/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.features.control.topics;

import static org.slf4j.LoggerFactory.getLogger;

import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.pushtechnology.diffusion.client.callbacks.Callback;
import com.pushtechnology.diffusion.client.callbacks.ContextCallback;
import com.pushtechnology.diffusion.client.callbacks.Registration;
import com.pushtechnology.diffusion.client.callbacks.Stream;
import com.pushtechnology.diffusion.client.features.HandlerConflictException;
import com.pushtechnology.diffusion.client.features.InvalidFilterException;
import com.pushtechnology.diffusion.client.features.NoSuchSessionException;
import com.pushtechnology.diffusion.client.features.TopicTreeHandler;
import com.pushtechnology.diffusion.client.features.control.clients.ClientControl.SessionPropertiesListener;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.session.SessionException;
import com.pushtechnology.diffusion.client.session.SessionId;
import com.pushtechnology.diffusion.client.topics.TopicSelector;
import com.pushtechnology.diffusion.client.topics.TopicSelectors;
import com.pushtechnology.diffusion.client.topics.details.TopicType;
import com.pushtechnology.diffusion.client.types.ErrorReport;
import com.pushtechnology.diffusion.client.types.GlobalPermission;
import com.pushtechnology.diffusion.client.types.PathPermission;

/**
 * This feature allows a session to subscribe or unsubscribe other sessions to
 * topics. It also provides a mechanism for handling requests to subscribe to
 * routing topics (deprecated).
 * <P>
 * Requests to subscribe sessions to topics can be submitted at any time even if
 * the topics do not exist at the server. {@link TopicSelector Topic selectors}
 * are used on subscription to match against topics at the server. The session
 * will become subscribed to any topics that exist at the server that match the
 * selector (unless they are already subscribed, or the session has insufficient
 * permission). The subscription request is also retained at the server so that
 * if any newly created topics match the selector, the session will then become
 * subscribed to it (unless a subsequent unsubscription cancels it).
 * <P>
 * Specific sessions may be subscribed/unsubscribed if the {@link SessionId} is
 * known.
 * <P>
 * Subscriptions may also be requested using 'session filters' (see
 * {@link Session} for a full description of session filters), where all
 * sessions that satisfy a particular filter expression will be
 * subscribed/unsubscribed. The filter is only evaluated once against the
 * current sessions that exist at the time - it is not retained and applied to
 * any sessions that are created later. In order to be notified of new sessions
 * as they are created {@link SessionPropertiesListener session properties
 * listeners} can be used and those sessions subscribed as required based upon
 * their session properties.
 *
 * <p>
 * A handler for routing topics belonging to a branch of the topic tree can be
 * added using
 * {@link #addRoutingSubscriptionHandler(String, SubscriptionControl.RoutingSubscriptionRequest.RoutingHandler)
 * addRoutingSubscriptionHandler()}. When another session subscribes to one of
 * the routing topics, the handler will be called to determine the appropriate
 * source topic.
 *
 * <H3>Access control</H3> To subscribe other sessions to topics, a session must
 * have {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} permission, and
 * {@link PathPermission#SELECT_TOPIC SELECT_TOPIC} permission for the path
 * prefix of the topic selector used for subscription. The subscribed sessions
 * will only be subscribed to matching topics for which they have
 * {@link PathPermission#READ_TOPIC READ_TOPIC} permission.
 * <p>
 * To unsubscribe other sessions, a session must have
 * {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} permission.
 * <P>
 * To register a {@link #addRoutingSubscriptionHandler routing subscription
 * handler} the session needs {@link GlobalPermission#VIEW_SESSION
 * VIEW_SESSION}, {@link GlobalPermission#MODIFY_SESSION MODIFY_SESSION} and
 * {@link GlobalPermission#REGISTER_HANDLER REGISTER_HANDLER} permissions.
 * <P>
 * When handling a subscription request to a routing topic via a routing handler
 * the routing session needs {@link PathPermission#READ_TOPIC READ_TOPIC}
 * permission to both the routing topic being subscribed to and the target topic
 * that is assigned.
 * <P>
 * Operations that identify sessions using a session filter require the
 * {@link GlobalPermission#VIEW_SESSION VIEW_SESSION} permission.
 *
 * <H3>Accessing the feature</H3> This feature may be obtained from a
 * {@link Session session} as follows:
 *
 * <pre>
 * <code>
 * SubscriptionControl subsControl = session.feature(SubscriptionControl.class);
 * </code>
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface SubscriptionControl extends Feature {
    /**
     * Adds a handler for {@link TopicType#ROUTING routing} topics that belong
     * to a branch of the topic tree. When another session subscribes to one of
     * the routing topics, the handler will be called to determine the
     * appropriate source topic.
     *
     * <p>
     * For each subscription to the routing topic, the server will select a
     * handler registered for the most specific branch. If multiple handlers are
     * registered for the same branch, the server will select one of them
     * arbitrarily.
     *
     * @param topicPath identifies the branch of the topic tree to associate the
     *        handler with
     *
     * @param handler the handler to use for routing topics at or below the
     *        specified path (unless there is a handler registered for a more
     *        specific topic path)
     *
     * @return a CompletableFuture that completes when the handler is registered
     *         with the server.
     *
     *         <p>
     *         If registration was successful, the CompletableFuture will
     *         complete successfully with a {@link Registration} which can be
     *         used to unregister the handler.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link HandlerConflictException} &ndash; if the session has
     *         already registered a routing subscription handler for
     *         {@code topicPath};
     *
     *         <li>{@link PermissionsException} &ndash; if the session does
     *         not have {@code REGISTER_HANDLER} permission;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does
     *         not have {@code VIEW_SESSION} permission;
     *
     *         <li>{@link PermissionsException} &ndash; if the session does
     *         not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.0
     * @deprecated since 6.7
     *             <p>
     *             Routing topics are deprecated. The more powerful
     *             {@link SessionTrees} feature should be used in their place.
     */
    @Deprecated
    CompletableFuture<Registration> addRoutingSubscriptionHandler(
        String topicPath,
        RoutingSubscriptionRequest.RoutingHandler handler);

    /**
     * Adds a handler for {@link TopicType#ROUTING routing} topics that belong
     * to a branch of the topic tree. When another session subscribes to one of
     * the routing topics, the handler will be called to determine the
     * appropriate source topic.
     *
     * <p>
     * The CompletableFuture-based alternative
     * {@link SubscriptionControl#addRoutingSubscriptionHandler(String, com.pushtechnology.diffusion.client.features.control.topics.SubscriptionControl.RoutingSubscriptionRequest.RoutingHandler)
     * addRoutingSubscriptionHandler(String, RoutingHandler)} should be
     * preferred since it provides better error reporting.
     *
     * <p>
     * For each subscription to the routing topic, the server will select a
     * handler registered for the most specific branch. If multiple handlers are
     * registered for the same branch, the server will select one of them
     * arbitrarily.
     *
     * @param topicPath identifies the branch of the topic tree to associate the
     *        handler with
     *
     * @param handler the handler to use for routing topics at or below the
     *        specified path (unless there is a handler registered for a
     *        more specific topic path)
     *
     * @deprecated since 6.7
     *             <p>
     *             Routing topics are deprecated. The more powerful
     *             {@link SessionTrees} feature should be used in their place.
     */
    @Deprecated
    void addRoutingSubscriptionHandler(
        String topicPath,
        RoutingSubscriptionRequest.Handler handler);

    /**
     * Subscribe another session to topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribe(SessionId, TopicSelector)} with a selector parsed using
     * {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void
     *         to provide forward compatibility with future iterations of this
     *         API that may provide a non-null result with a more specific
     *         result type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code SELECT_TOPIC} permission for the
     *         path prefix of the selector expression ;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId};
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     * @since 6.0
     */
    CompletableFuture<?> subscribe(
        SessionId sessionId,
        String topics)
        throws IllegalArgumentException;

    /**
     * Subscribe another session to topics.
     *
     * <p>
     * New subscriptions will be established for existing topics that match the
     * provided topic selector and for which the subscribed session has
     * {@code READ_TOPIC} permission. The topic selector will be added to the
     * topic selections of the subscribed session, and re-evaluated when new
     * topics are added or the session's security roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics identifies the topics to subscribe to
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void
     *         to provide forward compatibility with future iterations of this
     *         API that may provide a non-null result with a more specific
     *         result type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code SELECT_TOPIC} permission for the
     *         path prefix of the selector expression ;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId};
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     * @since 6.0
     */
    CompletableFuture<?> subscribe(
        SessionId sessionId,
        TopicSelector topics);

    /**
     * Subscribe another session to topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribe(SessionId, TopicSelector, SubscriptionCallback)} with a
     * selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void subscribe(
        SessionId sessionId,
        String topics,
        SubscriptionCallback callback)
        throws IllegalArgumentException;

    /**
     * Subscribe another session to topics.
     *
     * <p>
     * New subscriptions will be established for existing topics that match the
     * provided topic selector and for which the subscribed session has
     * {@code READ_TOPIC} permission. The topic selector will be added to the
     * topic selections of the subscribed session, and re-evaluated when new
     * topics are added or the session's security roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics identifies the topics to subscribe to
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void subscribe(
        SessionId sessionId,
        TopicSelector topics,
        SubscriptionCallback callback);

    /**
     * Subscribe another session to topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribe(SessionId, TopicSelector, Object, SubscriptionContextCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void subscribe(
        SessionId sessionId,
        String topics,
        C context,
        SubscriptionContextCallback<C> callback)
        throws IllegalArgumentException;

    /**
     * Subscribe another session to topics.
     *
     * <p>
     * New subscriptions will be established for existing topics that match the
     * provided topic selector and for which the subscribed session has
     * {@code READ_TOPIC} permission. The topic selector will be added to the
     * topic selections of the subscribed session, and re-evaluated when new
     * topics are added or the session's security roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param sessionId identifies the session to subscribe
     *
     * @param topics identifies the topics to subscribe to
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void subscribe(
        SessionId sessionId,
        TopicSelector topics,
        C context,
        SubscriptionContextCallback<C> callback);

    /**
     * Unsubscribe another session from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribe(SessionId, TopicSelector)} with a selector parsed
     * using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe specified as a
     *        {@link TopicSelector} expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void
     *         to provide forward compatibility with future iterations of this
     *         API that may provide a non-null result with a more specific
     *         result type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId};
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     * @since 6.0
     */
    CompletableFuture<?> unsubscribe(
        SessionId sessionId,
        String topics)
        throws IllegalArgumentException;

    /**
     * Unsubscribe another session from topics.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe from
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully. The result type is any rather than Void
     *         to provide forward compatibility with future iterations of this
     *         API that may provide a non-null result with a more specific
     *         result type.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} permission;
     *
     *         <li>{@link NoSuchSessionException} &ndash; if there is no session
     *         with the given {@code sessionId};
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     * @since 6.0
     */
    CompletableFuture<?> unsubscribe(
        SessionId sessionId,
        TopicSelector topics);

    /**
     * Unsubscribe another session from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribe(SessionId, TopicSelector, SubscriptionCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe specified as a
     *        {@link TopicSelector} expression
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void unsubscribe(
        SessionId sessionId,
        String topics,
        SubscriptionCallback callback)
        throws IllegalArgumentException;

    /**
     * Unsubscribe another session from topics.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe from
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void unsubscribe(
        SessionId sessionId,
        TopicSelector topics,
        SubscriptionCallback callback);

    /**
     * Unsubscribe another session from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribe(SessionId, TopicSelector, Object, SubscriptionContextCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe specified as a
     *        {@link TopicSelector} expression
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void unsubscribe(
        SessionId sessionId,
        String topics,
        C context,
        SubscriptionContextCallback<C> callback)
        throws IllegalArgumentException;

    /**
     * Unsubscribe another session from topics.
     *
     * @param sessionId identifies the session to unsubscribe
     *
     * @param topics the topics to unsubscribe from
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void unsubscribe(
        SessionId sessionId,
        TopicSelector topics,
        C context,
        SubscriptionContextCallback<C> callback);

    /**
     * Callback interface to receive status notifications for subscription and
     * unsubscription operations.
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface SubscriptionCallback extends
        com.pushtechnology.diffusion.client.features.Callback {

        /**
         * Called to indicate that the requested operation has been processed by
         * the server.
         */
        void onComplete();

        /**
         * Default implementation of {@link SubscriptionCallback}.
         * <p>
         * This simply logs onComplete calls at 'debug' level. This method may
         * be overridden to perform more specific processing.
         */
        class Default
            extends
            com.pushtechnology.diffusion.client.features.Callback.Default
            implements SubscriptionCallback {

            private static final Logger LOG =
                getLogger(SubscriptionCallback.Default.class);

            @Override
            public void onComplete() {
                LOG.debug(
                    "{} - Subscription/Unsubscription complete",
                    this);
            }

        }
    }

    /**
     * Contextual callback interface to receive status notifications for
     * subscription and unsubscription operations.
     *
     * <p>
     * Use this alternative to {@link SubscriptionCallback} to associate some
     * arbitrary context object with each call.
     *
     * @param <C> context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface SubscriptionContextCallback<C> extends
        com.pushtechnology.diffusion.client.features.ContextCallback<C> {

        /**
         * Called to indicate that the requested operation has been processed by
         * the server.
         *
         * @param context the context object supplied when making the call. May
         *        be {@code null}.
         */
        void onComplete(C context);

        /**
         * Default implementation of {@link SubscriptionContextCallback}.
         * <P>
         * This simply logs onComplete calls at 'debug' level. This method may
         * be overridden to perform more specific processing.
         *
         * @param <C> context object type
         */
        class Default<C>
            extends
            com.pushtechnology.diffusion.client.features.ContextCallback.Default<C>
            implements SubscriptionContextCallback<C> {

            private static final Logger LOG =
                getLogger(SubscriptionContextCallback.Default.class);

            @Override
            public void onComplete(
                C context) {

                LOG.debug(
                    "{} - Subscription/Unsubscription complete, context={}",
                    this,
                    context);
            }

        }
    }

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribeByFilter(String, TopicSelector)} with a selector parsed
     * using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} and
     *         {@code VIEW_SESSION} global permissions;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code SELECT_TOPIC} permission for the
     *         path prefix of the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @since 6.0
     */
    CompletableFuture<SubscriptionByFilterResult> subscribeByFilter(
        String filter,
        String topics)
        throws IllegalArgumentException;

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     *
     * <p>
     * For each session that matches the filter, new subscriptions will be
     * established for existing topics that match the provided topic selector
     * and for which the sessions has {@code READ_TOPIC} permission. The topic
     * selector will be added to the topic selections of the subscribed session,
     * and re-evaluated when new topics are added or the session's security
     * roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the subscription request.
     *
     *         <p>
     *         If the subscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} and
     *         {@code VIEW_SESSION} permissions;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code SELECT_TOPIC} permission for the
     *         path prefix of the selector expression;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<SubscriptionByFilterResult> subscribeByFilter(
        String filter,
        TopicSelector topics);

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribeByFilter(String, TopicSelector, SubscriptionByFilterCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void subscribeByFilter(
        String filter,
        String topics,
        SubscriptionByFilterCallback callback)
        throws IllegalArgumentException;

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     *
     * <p>
     * For each session that matches the filter, new subscriptions will be
     * established for existing topics that match the provided topic selector
     * and for which the sessions has {@code READ_TOPIC} permission. The topic
     * selector will be added to the topic selections of the subscribed session,
     * and re-evaluated when new topics are added or the session's security
     * roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param filter the session filter expression
     *
     * @param topics identifies the topics to subscribe to
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void subscribeByFilter(
        String filter,
        TopicSelector topics,
        SubscriptionByFilterCallback callback);

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #subscribeByFilter(String, TopicSelector, Object, SubscriptionByFilterContextCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to subscribe to specified as a
     *        {@link TopicSelector} expression
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @param <C> context object type
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void subscribeByFilter(
        String filter,
        String topics,
        C context,
        SubscriptionByFilterContextCallback<C> callback)
        throws IllegalArgumentException;

    /**
     * Subscribe sessions that satisfy a given session filter to topics.
     *
     * <p>
     * For each session that matches the filter, new subscriptions will be
     * established for existing topics that match the provided topic selector
     * and for which the sessions has {@code READ_TOPIC} permission. The topic
     * selector will be added to the topic selections of the subscribed session,
     * and re-evaluated when new topics are added or the session's security
     * roles change.
     *
     * <p>
     * A session that does not have {@code SELECT_TOPIC} permission for a topic
     * cannot subscribe directly, but can be subscribed indirectly using this
     * method.
     *
     * @param filter the session filter expression
     *
     * @param topics identifies the topics to subscribe to
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods to indicate success or failure
     *
     * @param <C> context object type
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void subscribeByFilter(
        String filter,
        TopicSelector topics,
        C context,
        SubscriptionByFilterContextCallback<C> callback);

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribeByFilter(String, TopicSelector)} with a selector
     * parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from specified as a
     *        {@link TopicSelector} expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} and
     *         {@code VIEW_SESSION} permissions;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @since 6.0
     */
    CompletableFuture<SubscriptionByFilterResult> unsubscribeByFilter(
        String filter,
        String topics)
        throws IllegalArgumentException;

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from specified as a
     *        {@link TopicSelector} expression
     *
     * @return a CompletableFuture that completes when the server has processed
     *         the unsubscription request.
     *
     *         <p>
     *         If the unsubscription was accepted, the CompletableFuture will
     *         complete successfully with a {@link SubscriptionByFilterResult}.
     *
     *         <p>
     *         Otherwise, the CompletableFuture will complete exceptionally with
     *         a {@link CompletionException}. Common reasons for failure, listed
     *         by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link InvalidFilterException} &ndash; if the filter is
     *         invalid;
     *
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have {@code MODIFY_SESSION} and
     *         {@code VIEW_SESSION} global permissions;
     *
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     *
     * @since 6.0
     */
    CompletableFuture<SubscriptionByFilterResult> unsubscribeByFilter(
        String filter,
        TopicSelector topics);

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribeByFilter(String, TopicSelector, SubscriptionByFilterCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe specified as a
     *        {@link TopicSelector} expression
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void unsubscribeByFilter(
        String filter,
        String topics,
        SubscriptionByFilterCallback callback)
        throws IllegalArgumentException;

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    void unsubscribeByFilter(
        String filter,
        TopicSelector topics,
        SubscriptionByFilterCallback callback);

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     *
     * <P>
     * This is equivalent to calling
     * {@link #unsubscribeByFilter(String, TopicSelector, SubscriptionByFilterCallback)}
     * with a selector parsed using {@link TopicSelectors#parse(String)}.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe specified as a
     *        {@link TopicSelector} expression
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @throws IllegalArgumentException if {@code topics} is not a valid
     *         selector expression
     *
     * @param <C> context object type
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void unsubscribeByFilter(
        String filter,
        String topics,
        C context,
        SubscriptionByFilterContextCallback<C> callback)
        throws IllegalArgumentException;

    /**
     * Unsubscribe sessions that satisfy a given session filter from topics.
     *
     * @param filter the session filter expression
     *
     * @param topics the topics to unsubscribe from
     *
     * @param context passed to the callback with the reply to allow requests
     *        and replies to be correlated. The caller may use any convenient
     *        object reference, including {@code null}
     *
     * @param callback provides callback methods indicating the status of this
     *        operation
     *
     * @param <C> context object type
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    <C> void unsubscribeByFilter(
        String filter,
        TopicSelector topics,
        C context,
        SubscriptionByFilterContextCallback<C> callback);

    /**
     * Callback interface for filtered subscriptions and unsubscriptions.
     *
     * @since 5.6
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface SubscriptionByFilterCallback extends Callback {

        /**
         * Called to indicate successful processing of the request at the
         * server.
         *
         * @param numberSelected the number of sessions that satisfied the
         *        filter and qualified for subscription/unsubscription. This
         *        will be 0 if no sessions satisfied the filter.
         */
        void onComplete(int numberSelected);

        /**
         * The filter was rejected. No sessions were subscribed/unsubscribed.
         *
         * @param errors the detail of why the filter was rejected
         */
        void onRejected(Collection<ErrorReport> errors);

        /**
         * Default implementation of {@link SubscriptionByFilterCallback}.
         * <p>
         * This logs onComplete calls at 'debug' level and onRejected calls at
         * 'error' level. This method may be overridden to perform more specific
         * processing.
         */
        class Default extends
            Callback.Default
            implements SubscriptionByFilterCallback {

            private static final Logger LOG =
                LoggerFactory
                    .getLogger(SubscriptionByFilterCallback.Default.class);

            @Override
            public void onComplete(int numberSelected) {
                LOG.debug(
                    "{} - Subscription/Unsubscription complete : {} sessions selected",
                    this,
                    numberSelected);
            }

            @Override
            public void onRejected(Collection<ErrorReport> errors) {
                LOG.error(
                    "{} - Subscription/Unsubscription rejected : {}",
                    this,
                    errors);
            }
        }

    }

    /**
     * Callback interface for filtered subscriptions and unsubscriptions that
     * have an associated context.
     *
     * @since 5.6
     * @param <C> the context object type
     *
     * @deprecated since 6.7
     *             <p>
     *             Methods that use callbacks are deprecated and will be removed
     *             in a future release. Use CompletableFuture variant instead.
     */
    @Deprecated
    interface SubscriptionByFilterContextCallback<C> extends ContextCallback<C> {

        /**
         * Called to indicate successful processing of the request at the
         * server.
         *
         * @param context the context object the application supplied when
         *        making the call; may be {@code null}
         *
         * @param numberSelected the number of sessions that satisfied the
         *        filter and qualified for subscription/unsubscription. This
         *        will be 0 if no sessions satisfied the filter.
         */
        void onComplete(C context, int numberSelected);

        /**
         * The filter was rejected. No sessions were subscribed/unsubscribed.
         *
         * @param context the context object supplied when making the call
         *
         * @param errors the detail of why the filter was rejected
         */
        void onRejected(C context, Collection<ErrorReport> errors);

        /**
         * Default implementation of {@link SubscriptionByFilterContextCallback}
         * .
         * <P>
         * This logs onComplete calls at 'debug' level and onRejected calls at
         * 'error' level. This method may be overridden to perform more specific
         * processing.
         *
         * @param <C> context object type
         */
        class Default<C>
            extends
            ContextCallback.Default<C>
            implements SubscriptionByFilterContextCallback<C> {

            private static final Logger LOG =
                getLogger(SubscriptionByFilterContextCallback.Default.class);

            @Override
            public void onComplete(C context, int numberSelected) {
                LOG.debug(
                    "{} - Subscription/Unsubscription complete : {} sessions selected, Context={}",
                    this,
                    numberSelected,
                    context);

            }

            @Override
            public void onRejected(C context, Collection<ErrorReport> errors) {
                LOG.error(
                    "{} - Subscription/Unsubscription rejected : {}, Context={}",
                    this,
                    errors,
                    context);
            }
        }
    }

    /**
     * A single request from a session to subscribe to routing topic.
     *
     * <p>
     * Each request received by a handler must either be {@link #route routed}
     * or {@link #defer deferred}.
     *
     * @deprecated since 6.7
     *             <p>
     *             Routing topics are deprecated. The more powerful
     *             {@link SessionTrees} feature should be used in their place.
     */
    @Deprecated
    interface RoutingSubscriptionRequest {

        /**
         * Identifies the session making the subscription request.
         *
         * @return the session id of the requesting session
         */
        SessionId getSessionId();

        /**
         * Identifies the subscribed routing topic.
         *
         * @return the topic path of the routing topic
         */
        String getTopicPath();

        /**
         * Defer the handling of this subscription request.
         *
         * <p>
         * When a request is deferred, the server will discard the request, and
         * the requesting session will not receive a subscription notification.
         *
         * <p>
         * An implementation can retain the requesting session's
         * {@link #getSessionId() session ID} and the routing topic's
         * {@link #getTopicPath() path} so a routing subscription can be
         * established at a later time using
         * {@link SubscriptionControl#subscribe(SessionId, String, SubscriptionCallback)}.
         *
         * <p>
         * A {@code RoutingSubscriptionRequest} can be used once. If
         * {@code route()} or {@code defer()} has already been called for this
         * {@code RoutingSubscriptionRequest}, calling this method will cause a
         * warning to be logged by the server, but otherwise have no effect.
         *
         * <h3>Behavior since 6.0</h3>
         *
         * <p>
         * From 6.0, this method returns a CompletableFuture and no longer
         * throws exceptions. All errors are reported through the
         * CompletableFuture result.
         *
         * @return a CompletableFuture that completes when the server has
         *         processed the defer instruction.
         *
         *         <p>
         *         If the instruction was processed, the CompletableFuture will
         *         complete successfully. The result type is any rather than
         *         Void to provide forward compatibility with future iterations
         *         of this API that may provide a non-null result with a more
         *         specific result type.
         *
         *         <p>
         *         Otherwise, the CompletableFuture will complete exceptionally
         *         with a {@link CompletionException}. Common reasons for
         *         failure, listed by the exception reported as the
         *         {@link CompletionException#getCause() cause}, include:
         *
         *         <ul>
         *         <li>{@link SessionException} &ndash; if {@code route()} or
         *         {@code defer()} has already been called for this
         *         {@code RoutingSubscriptionRequest};
         *
         *         <li>{@link SessionClosedException} &ndash; if the session is
         *         closed.
         *         </ul>
         */
        CompletableFuture<?> defer();

        /**
         * Resolve a subscription request by providing the path of a source
         * topic to which the requesting session will be subscribed via the
         * routing topic.
         *
         * <p>
         * The requesting session will receive a subscription notification with
         * the routing topic's path and the topic specification of the source
         * topic. If the source topic is stateful, the requesting session will
         * also be sent an update for the routing topic path with the current
         * value of the source topic. The existence of the source topic is
         * hidden from the requesting session. Updates to the source topic are
         * forwarded to the session as if they came from the routing topic.
         *
         * <p>
         * A session that does not have {@code SELECT_TOPIC} or
         * {@code READ_TOPIC} permission for the source topic cannot subscribe
         * directly, but can be subscribed indirectly using this method.
         *
         * @param topicPath the topic path of the source topic
         * @return a CompletableFuture that completes when the server has
         *         processed the routing instruction.
         *
         *         <p>
         *         If a routing subscription was established, the
         *         CompletableFuture will complete successfully. The result type
         *         is any rather than Void to provide forward compatibility
         *         with future iterations of this API that may provide a
         *         non-null result with a more specific result type.
         *
         *         <p>
         *         Otherwise, the CompletableFuture will complete exceptionally
         *         with a {@link CompletionException}. Common reasons for
         *         failure, listed by the exception reported as the
         *         {@link CompletionException#getCause() cause}, include:
         *
         *         <ul>
         *         <li>{@link RoutingSubscriptionException} &ndash; if there is
         *         no topic bound to {@code topicPath};
         *
         *         <li>{@link RoutingSubscriptionException} &ndash; if the topic
         *         bound to {@code topicPath} is itself a routing topic;
         *
         *         <li>{@link PermissionsException} &ndash; if the calling
         *         session does not have {@code READ_TOPIC} permission for the
         *         source topic bound to {@code topicPath};
         *
         *         <li>{@link PermissionsException} &ndash; if the calling
         *         session does not have {@code READ_TOPIC} permission for the
         *         routing topic bound to {@link #getTopicPath()};
         *
         *         <li>{@link SessionException} &ndash; if {@code route()} or
         *         {@code defer()} has already been called for this
         *         {@code RoutingSubscriptionRequest};
         *
         *         <li>{@link SessionClosedException} &ndash; if the session is
         *         closed.
         *         </ul>
         */
        CompletableFuture<?> route(String topicPath);

        /**
         * Resolve a subscription request by providing the path of a source
         * topic to which the requesting session will be subscribed via the
         * routing topic.
         *
         * <p>
         * The requesting session will receive a subscription notification with
         * the routing topic's path and the topic specification of the source
         * topic. If the source topic is stateful, the requesting session will
         * also be sent an update for the routing topic path with the current
         * value of the source topic. The existence of the source topic is
         * hidden from the requesting session. Updates to the source topic are
         * forwarded to the session as if they came from the routing topic.
         *
         * <p>
         * A session that does not have {@code SELECT_TOPIC} or
         * {@code READ_TOPIC} permission for the source topic cannot subscribe
         * directly, but can be subscribed indirectly using this method.
         *
         * <p>
         * The operation will fail and the callback with be
         * {@link SubscriptionCallback#onDiscard() discarded} in the following
         * cases:
         *
         * <ul>
         * <li>There is no source topic bound to {@code topicPath};
         * <li>The topic bound to {@code topicPath} is a routing topic;
         * <li>The calling session does not have {@code READ_TOPIC} permission
         * for the source topic bound to {@code topicPath};
         * <li>The calling session does not have {@code READ_TOPIC} permission
         * for the routing topic bound to {@link #getTopicPath()};
         * <li>{@code route()} or {@code defer()} has already been called for
         * this {@code RoutingSubscriptionRequest}.
         * </ul>
         *
         * @param topicPath the topic path of the resolved source topic
         *
         * @param callback provides callback methods indicating the status of
         *        this request.
         */
        void route(String topicPath, SubscriptionCallback callback);

        /**
         * Provide the path of a source topic to which the requesting session
         * will be subscribed via the routing topic, with a contextual object.
         *
         * <p>
         * The requesting session will receive a subscription notification with
         * the routing topic's path and the topic specification of the source
         * topic. If the source topic is stateful, the requesting session will
         * also be sent an update for the routing topic path with the current
         * value of the source topic. The existence of the source topic is
         * hidden from the requesting session. Updates to the source topic are
         * forwarded to the session as if they came from the routing topic.
         *
         * <p>
         * A session that does not have {@code SELECT_TOPIC} or
         * {@code READ_TOPIC} permission for the source topic cannot subscribe
         * directly, but can be subscribed indirectly using this method.
         *
         * <p>
         * The operation will fail and the callback with be
         * {@link SubscriptionCallback#onDiscard() discarded} in the following
         * cases:
         *
         * <ul>
         * <li>There is no source topic bound to {@code topicPath};
         * <li>The topic bound to {@code topicPath} is a routing topic;
         * <li>The calling session does not have {@code READ_TOPIC} permission
         * for the source topic bound to {@code topicPath};
         * <li>The calling session does not have {@code READ_TOPIC} permission
         * for the routing topic bound to {@link #getTopicPath()};
         * <li>{@code route()} or {@code defer()} has already been called for
         * this {@code RoutingSubscriptionRequest}.
         * </ul>
         *
         * @param topicPath the topic path of the resolved source topic
         *
         * @param context passed to the callback with the reply to allow
         *        requests and replies to be correlated. The caller may use any
         *        convenient object reference, including {@code null}
         *
         * @param callback provides callback methods indicating the status of
         *        this request.
         *
         * @param <C> context object type
         */
        <C> void route(
            String topicPath,
            C context,
            SubscriptionContextCallback<C> callback);

        /**
         * Handler for routing subscription requests.
         *
         * @deprecated since 6.7
         *             <p>
         *             Routing topics are deprecated. The more powerful
         *             {@link SessionTrees} feature should be used in their
         *             place.
         */
        @Deprecated
        interface RoutingHandler extends Stream {

            /**
             * A request to subscribe to a specific routing topic.
             *
             * @param request the request, which must be replied to or deferred
             */
            void onSubscriptionRequest(RoutingSubscriptionRequest request);

            /**
             * Abstract handler implementation.
             * <P>
             * The
             * {@link RoutingHandler#onSubscriptionRequest(SubscriptionControl.RoutingSubscriptionRequest)
             * onSubscriptionRequest} method must be implemented.
             */
            abstract class Default
                extends Stream.Default implements RoutingHandler {
            }
        }

        /**
         * Handler for routing subscription requests registered with the older,
         * callback-based
         * {@link SubscriptionControl#addRoutingSubscriptionHandler(String, SubscriptionControl.RoutingSubscriptionRequest.Handler)
         * addRoutingSubscriptionHandler(String, Handler)} method.
         *
         * @deprecated since 6.7
         *             <p>
         *             Routing topics are deprecated. The more powerful
         *             {@link SessionTrees} feature should be used in their
         *             place.
         */
        @Deprecated
        interface Handler extends TopicTreeHandler {

            /**
             * A request to subscribe to a specific routing topic.
             *
             * @param request the request, which must be replied to or deferred
             */
            void onSubscriptionRequest(RoutingSubscriptionRequest request);

            /**
             * Abstract handler implementation.
             * <P>
             * The
             * {@link Handler#onSubscriptionRequest(SubscriptionControl.RoutingSubscriptionRequest)
             * onSubscriptionRequest} method must be implemented.
             */
            abstract class Default
                extends TopicTreeHandler.Default implements Handler {
            }
        }
    }

    /**
     * Result used by CompletableFuture variants of
     * {@link SubscriptionControl#subscribeByFilter(String, String)
     * subscribeByFilter} and
     * {@link SubscriptionControl#unsubscribeByFilter(String, String)
     * unsubscribeByFilter}.
     *
     * @since 6.0
     */
    public interface SubscriptionByFilterResult {

        /**
         * @return number of sessions that satisfied the filter and qualified
         *         for subscription/unsubscription. This will be 0 if no
         *         sessions satisfied the filter.
         */
        int numberSelected();
    }
}
