/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/

package com.pushtechnology.diffusion.client.features.control.topics.views;

import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;

import com.pushtechnology.diffusion.client.features.ClusterRoutingException;
import com.pushtechnology.diffusion.client.features.ScriptException;
import com.pushtechnology.diffusion.client.features.Topics;
import com.pushtechnology.diffusion.client.features.control.RemoteServers;
import com.pushtechnology.diffusion.client.session.Feature;
import com.pushtechnology.diffusion.client.session.PermissionsException;
import com.pushtechnology.diffusion.client.session.Session;
import com.pushtechnology.diffusion.client.session.SessionClosedException;
import com.pushtechnology.diffusion.client.topics.TopicSelectors;
import com.pushtechnology.diffusion.client.topics.details.TopicSpecification;
import com.pushtechnology.diffusion.client.topics.details.TopicType;
import com.pushtechnology.diffusion.client.types.GlobalPermission;
import com.pushtechnology.diffusion.client.types.PathPermission;

/**
 * This feature allows a client session to manage topic views.
 *
 * <p>
 * A topic view maps one part of a server's topic tree to another. It
 * dynamically creates a set of <em>reference topics</em> from a set of
 * <em>source topics</em>, based on a declarative <em>topic view
 * specification</em>. The capabilities of topic views range from simple
 * mirroring of topics within the topic tree to advanced capabilities including
 * publication of partial values, expanding a single topic value into many
 * topics, changing topic values, inserting values from other topics, throttling
 * the rate of publication, and applying a fixed delay to the publication.
 *
 * <p>
 * A topic view can also map topics from another server (in a different
 * cluster). This capability is referred to as 'remote topic views'. The view
 * can specify the server that the <em>source topics</em> are hosted on in terms
 * of a <em>remote server</em> (see {@link RemoteServers} for details of how to
 * create and maintain remote servers).
 *
 * <p>
 * Each reference topic has a single source topic and has the same topic type as
 * its source topic. Reference topics are read-only (they cannot be updated),
 * nor can they be created or removed directly. Otherwise, they behave just like
 * standard topics. A client session can subscribe to a reference topic, and can
 * fetch the reference topic's current value if it has one.
 *
 * <p>
 * The source topics of a topic view are defined by a topic selector. One or
 * more reference topics are created for each source topic, according to the
 * topic view. If a source topic is removed, reference topics that are derived
 * from it will automatically be removed. If a topic is added that matches the
 * source topic selector of a topic view, corresponding reference topics will be
 * created. Removing a topic view will remove all of its reference topics.
 *
 * <h2 id="view-specification">Topic view specifications</h2>
 *
 * <p>
 * The following is a simple topic view specification that mirrors all topics
 * below the path <code>a</code> to reference topics below the path
 * <code>b</code>.
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt;
 * </pre>
 *
 * <p>
 * A topic view with this specification will map a source topic at the path
 * <code>a/x/y/z</code> to a reference topic at the path <code>b/x/y/z</code>.
 * The specification is simple, so the reference topic will exactly mirror the
 * source topic.
 *
 * <p>
 * A topic view specification comprises three main parts:
 * <ul>
 * <li>The <em>mapping</em> part which specifies the source topics to map from
 * and the mappings to target reference topics.
 * <li>Optional <em>transformations</em> which transform the topic value in some
 * way.
 * <li>Optional <em>options</em> which specify other changes that the view may
 * apply.
 * </ul>
 *
 * Mapping comprises:
 * <ul>
 * <li>The <em>source topic</em> clause identifying the source topics that the
 * view can apply to.
 * <li>The optional <em>from</em> clause which may identify a remote server that
 * hosts the source topics.
 * <li>The <em>path mapping</em> clause which determines how reference topic
 * paths are derived from the source topic paths, and when expanding to more
 * than one reference topic, from where the values are obtained.
 * </ul>
 * Transformations can be:
 * <ul>
 * <li><em>patch</em> transformation(s) specifying that a JSON patch is applied
 * to the reference topic value.
 * <li><em>process</em> transformations that allow conditional processing and/or
 * calculations to be applied to the reference topic value.
 * <li><em>insert</em> transformation(s) specifying that values from other
 * topics are inserted into the reference topic value.
 * </ul>
 * Options can be:
 * <ul>
 * <li>The <em>topic property mapping</em> clause determines how reference topic
 * properties are derived from source topic properties.
 * <li>The <em>value mapping</em> clause determines how reference topic values
 * are derived from source topic or expanded values.
 * <li>The <em>throttle</em> clause constrains the rate at which each reference
 * topic is updated when its source topic is updated.
 * <li>The <em>delay by</em> clause causes a change to a view's source topic to
 * be delayed by a fixed time before it is reflected in reference topics.
 * <li>The <em>separator</em> clause can define a replacement path separator for
 * values extracted using the scalar or expand directives.
 * <li>The <em>type</em> clause can specify that the reference topic that is
 * created is of a different {@link TopicType} from the selected source topic.
 * </ul>
 *
 * <h3>Mapping</h3>
 * <h4>Source topic clause</h4>
 * <p>
 * The source topic clause begins with the {@code map} keyword and is followed
 * by a topic selector. These topic selectors follow the same
 * {@link TopicSelectors#parse parsing rules} as other topic selectors.
 *
 * <p>
 * When evaluating a topic view, all topics in the topic tree that match the
 * source topic selector are considered (excluding {@link TopicType#ROUTING
 * ROUTING} topics). However, if a view specification uses some feature that can
 * only be applied to JSON topics then only JSON topics will be selected.
 *
 * <p>
 * Reference topics are valid source topics. In particular, chaining of topic
 * views is supported; that is, a reference topic created by one topic view can
 * be the source topic of another topic view. Additionally, a reference topic
 * can be the source topic of a routing topic subscription.
 *
 * <h4>From clause</h4>
 * <p>
 * The 'from' clause optionally follows the source topic clause. It begins with
 * the {@code from} keyword and is followed by a remote server name. The name
 * refers to a remote server created using the {@link RemoteServers} feature.
 *
 * <p>
 * The presence of the clause indicates that the source topics will be selected
 * from the specified server and not from the local server.
 *
 * <p>
 * Further details regarding the processing of remote topic views are given
 * below.
 *
 * <h4>Path mapping clause</h4>
 * <p>
 * The paths of reference topics are derived from the source topic according to
 * the path mapping clause. The path mapping allows the source topic path and
 * the value of the source topic to determine the path of the reference topic.
 * In addition the path mapping can include <em>expand</em> directives which
 * allow objects and arrays in JSON source topic values to be expanded to
 * produce many reference topics.
 *
 * <p>
 * A path mapping clause begins with the {@code to} keyword and is followed by a
 * path mapping template. A path mapping template is a topic path with embedded
 * <em>directives</em>. Directives are evaluated when creating the topic
 * reference and substituted into the topic path. Directives are delimited by
 * angle brackets ({@code <}, {@code >}) and consist of the name of the
 * directive and a list of parameters. The parameter list is comma-separated and
 * surrounded by parentheses ({@code (}, {@code )}).
 *
 * <p>
 * The following path mapping directives are supported:
 *
 * <dl>
 * <dt>Source path directives</dt>
 * <dd>Source path directives extract a portion of the source path and are
 * parameterized by the index of the start part of the source path and the
 * number of parts to include. The number of parts parameter is optional – if it
 * is missing, the selection extends to the end of the source path. The syntax
 * is <code>&lt;path(<em>start</em>, <em>number</em>)&gt;</code>, or
 * <code>&lt;path(<em>start</em>)&gt;</code> when the number of parts parameter
 * is omitted.
 * <p>
 * For example, given the source path {@code a/b/c/d}, the source path directive
 * {@code <path(1, 2)>} is mapped to the reference topic path {@code b/c}, and
 * the source path directive {@code <path(2)>} is mapped to the reference topic
 * path {@code c/d}.</dd>
 *
 * <dt>Source value ("scalar") directives</dt>
 * <dd>Source value directives are only applied to {@link TopicType#JSON JSON}
 * source topics or {@link TopicType#TIME_SERIES TIME_SERIES} source topics with
 * a {@code JSON} event type; if the path mapping contains a source value
 * directive, topics with other topic types matching the source topic selector
 * are ignored.
 * <p>
 * Source value directives use the keyword {@code scalar} and are parameterized
 * by a single <a href="https://tools.ietf.org/html/rfc6901"> JSON pointer</a>
 * that extracts a scalar value from the source (or current) value. A scalar
 * value is a string, a number, {@code true}, {@code false}, or {@code null},
 * that is, anything other than an array or a object. If the JSON pointer does
 * not refer to a scalar value in the source (or current) value, no reference
 * topic will be created. This includes cases where the JSON pointer refers to
 * an array or an object), or when no part of the source value is selected.
 * <p>
 * Deriving the reference topic paths from part of the source topic value
 * effectively creates a secondary index on the value. For source value
 * directives to work efficiently, the selected scalar values should be
 * relatively stable. If an update to the source topic changes the selected
 * scalar value, the corresponding reference topic will be removed and a new
 * reference topic will be created.
 * <p>
 * For example, given a source value of
 *
 * <pre>
 * {
 *   "account" : "1234",
 *   "balance" : { "amount" : 12.57, "currency" : "USD" }
 * }
 * </pre>
 *
 * and the source value directive
 * {@code currency/<scalar(/balance/currency)>/account/<scalar(/account)>}, the
 * reference topic path will be {@code currency/USD/account/1234}.
 *
 * <p>
 * If the extracted value is a string, it is copied literally to the reference
 * topic path. A value that contains path separators ({@code /}) will create a
 * reference topic path with more levels than the path mapping template. Use the
 * separator directive to replace path separators with an alternative string.
 *
 * <p>
 * An extracted value of {@code null} will be copied to the reference topic path
 * as the string {@code "null"}.</dd>
 *
 * <dt>Expand value directives</dt>
 *
 * <dd>Expand value directives are only applied to {@link TopicType#JSON JSON}
 * source topics; if the path mapping contains an expand value directive,
 * non-JSON topics matching the source topic selector are ignored.
 * <p>
 * Expand value directives use the keyword {@code expand} and are parameterized
 * by one or two <a href="https://tools.ietf.org/html/rfc6901"> JSON
 * pointers</a>.
 * <p>
 * The first pointer indicates the element within the value to be expanded, and
 * if omitted, the value is expanded from the root. Expansion of a source topic
 * indicates that every direct child of the element pointed to by the expand
 * pointer will be used to create a new reference topic (or provide input to
 * later expand or scalar directives). For example {@code <expand()>} would
 * expand every child item in the source value and {@code <expand(/account)>}
 * would expand every child of the {@code account} value in the source value.
 * The specified value could be an object, an array or even a scalar value, but
 * a scalar value would expand to only a single new value.
 * <p>
 * The optional second parameter of the expand directive specifies a pointer to
 * a scalar value within the expanded value which will be used to derive the
 * path fragment of the reference topic path. If the second pointer is not
 * specified or no scalar value is found for the pointer, the path fragment is
 * taken from the key (if the child value is an object) or the index (if the
 * child value is an array). Scalar child values will expand to a reference
 * topic but will not add anything to the generated path. For example
 * {@code <expand(,/name)>} would expand from the root of the source value and
 * each child value path fragment would be obtained from the scalar value with
 * the key {@code name}.
 * <p>
 * So if a source topic had a value of
 *
 * <pre>
 * {
 *     "values": [1, 5, 7]
 * }
 * </pre>
 *
 * a path mapping of {@code value<expand(/values)>} would expand the value to
 * the following reference topics:-
 * <p>
 * path {@code value0} with a value of {@code 1}<br>
 * path {@code value1} with a value of {@code 5}<br>
 * path {@code value2} with a value of {@code 7}<br>
 * <p>
 * Expand directives can be nested (i.e. there can be more than one expand
 * directive in a path mapping). In this case a second expand directive will use
 * the value from the previous expand as its source (root) value and not the
 * value of the source topic. This also applies to scalar directives that follow
 * an expand directive.
 * <p>
 * If expansion causes more than one mapping to the same topic path, only the
 * first encountered will be created and updated.
 * <p>
 * Expanding source topic values effectively creates secondary indices on the
 * value. For expanded value directives to work efficiently, the value selected
 * for expansion should be relatively stable in terms of the children it
 * contains. If an update to the source topic changes the children of the
 * expanded value, then corresponding reference topics will be removed and
 * created. Updates should generally be limited to changing values within the
 * expanded values.</dd>
 * </dl>
 *
 * <h3>Transformations</h3>
 * <p>
 * Transformations are specified after the mapping and before any options.
 * Transformations can only be applied to JSON topics.
 * <p>
 * Transformations are applied to the value extracted from the source topic in
 * the order specified. There can be any number of transformations interspersed
 * with one another and the value from one will be that which is input to the
 * next. The only restriction is that all <b>insert</b> transformations must
 * occur at the end, after any others.
 * <p>
 * A transformation is applied to the current value within a view processing
 * chain, so if a transformation occurs after an expand then it will be applied
 * to each expanded value.
 *
 * <h4>Process transformations</h4>
 * <p>
 * Process transformations may be used to apply conditional processing to a
 * value (optionally determining whether a reference topic is created) and/or
 * change the value in some way (for example, by applying some calculation to a
 * field within the value).
 * <p>
 * The format of a process transformation is:-
 * <p>
 * <code>
 * <b>process {</b><i>statement</i><b>}</b>
 * </code>
 * <p>
 * Where the <i>statement</i> can be:-
 * <ul>
 * <li>Operation(s). <br>
 * One or more operations separated by ';'.
 * <li>A conditional statement. <br>
 * Comprising one or more conditions with operations to perform if they are
 * satisfied.
 * </ul>
 * <p>
 * For example, the following topic view specification could be used to write a
 * field into the value of the reference topic :
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; process {set(/Name, 'John')};
 * </pre>
 *
 * The following example shows a simple conditional statement which would only
 * generate reference topics if the value of field <code>/Price</code> was
 * greater than 50 :
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; process {if '/Price gt 50' continue};
 * </pre>
 *
 * And the following shows a more complex statement which would set a field
 * according to the value of the input field <code>/Price</code> :
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; process {if '/Price lt 50' set(/Tier, 1) elseif '/Price gt 50' set(/Tier, 2)};
 * </pre>
 *
 * <dl>
 * <dt>Process operations</dt>
 * <dd>
 * <p>
 * The following operations are supported :
 * <table border = 1>
 * <tr>
 * <th style= "text-align:left;"><b>Operation</b></th>
 * <th style= "text-align:left;"><b>Description</b></th>
 * </tr>
 * <tr>
 * <tr>
 * <td style="text-align:left;"><b>set(</b><i>pointer</i>,
 * <i>value</i><b>)</b></td>
 * <td style="text-align:left;">Sets the field indicated by the JSON
 * <i>pointer</i> to an absolute <i>value</i>. If the field does not exist it
 * will be created. The <i>value</i> can be an integer (e.g. 123), a String
 * (e.g. &quot;XYZ&quot;), or a boolean (e.g. true or false).
 * <p>
 * For example :
 *
 * <pre>
 * set(/Name, &quot;John&quot;)
 * </pre>
 *
 * If a hierarchic pointer is specified, the parent object or array must
 * exist.</td>
 * </tr>
 * <tr>
 * <td style="text-align:left;"><b>set(</b><i>pointer</i>, <b>calc</b>
 * <i>calculation</i><b>)</b></td>
 * <td style="text-align:left;">Sets the field indicated by the JSON
 * <i>pointer</i> to a value which is the result of the specified
 * <i>calculation</i>. If the field does not exist it will be created. The
 * calculation can include fields within the input value. The calculation is
 * specified as a quoted string. See below for a detailed description of
 * calculations.
 * <p>
 * For example :
 *
 * <pre>
 * set(/DoubleValue, calc &quot;/Value * 2&quot;)
 * </pre>
 *
 * If a hierarchic pointer is specified, the parent object or array must
 * exist.</td>
 * </tr>
 * <tr>
 * <td style="text-align:left;"><b>remove(</b><i>pointer</i><b>)</b></td>
 * <td style="text-align:left;">Removes the json item at the specified
 * <i>pointer</i>. Unlike JSON patch, if the item does not exist the operation
 * does not fail.</td>
 * </tr>
 * <tr>
 * <td style="text-align:left;"><b>continue</b></td>
 * <td style="text-align:left;">This is a special operation that indicates that
 * the topic view evaluation should continue with the value as it is. This is
 * only for use with conditional statements as the default behavior of a
 * conditional statement is not to proceed if no condition is satisifed.</td>
 * </tr>
 * </table>
 * Operations can be chained by separating them with a ';' as shown in the
 * example below:-
 *
 * <pre>
 * set(/Amount, calc &quot;/Value * /Number&quot;); remove(/Value); remove (/Number)
 * </pre>
 *
 * In this case the operations are all performed on the original value, creating
 * a chain of deltas which are only applied to the original value at the end. If
 * the 'set' operation fails no reference topic would be generated, however,
 * processing will continue if the fields specified in the 'remove's are not
 * present.
 * <p>
 * </dd>
 *
 * <dt>Process calculations</dt>
 * <dd>
 * <p>
 * A calculation may be specified as the value of a <b>set</b> operation. A
 * calculation is a simple arithmetic calculation upon integer fields. If
 * applied to a non integer field the evaluation will not proceed. Floating
 * point calculations are not supported.
 * <p>
 * Arithmetic operators supported are <code>+</code>, <code>-</code> ,
 * <code>*</code> and <code>/</code>.
 * <p>
 * Examples of calculations are:-
 *
 * <pre>
 * set(/Value, calc &quot;/Value * 2&quot;)
 * set(/Result, calc &quot;/Value / 2&quot;)
 * set(/Bonus, calc &quot;/Salary + 1000&quot;)
 * set(/Bonus, calc &quot;/Salary + 1000 + /Age * 10&quot;)
 * </pre>
 *
 * Standard operator precedence is applied, so in the last example above we have
 * <code>(/Salary + 1000 + (/Age * 10))</code> not
 * <code>((/Salary + 1000 + /Age) * 10)</code>. Brackets may be used to override
 * this.
 * <p>
 * </dd>
 * <dt>Process conditional statements</dt>
 * <dd>
 * <p>
 * A conditional statement is made up of an <b>if</b> clause, optionally
 * followed by one or more <b>elseif</b> clauses and an optional final <b>else</b>
 * clause.
 * <p>
 * The <b>if</b> clause takes the form :
 *
 * <pre>
 * <b>if</b> <i>condition operation(s)</i>
 * </pre>
 *
 * Where the <i>condition</i> is a quoted string as described in detail below
 * and the <i>operation(s)</i> is as described previously. If the condition is
 * satisfied, the operations are applied to the value and the process is
 * complete. If the condition is not satisfied, processing moves on to any
 * <b>elseif</b> or <b>else</b> clauses that follow, but if there are none, the
 * topic view evaluation does not proceed and no reference topic is created.
 * <p>
 * An <b>elseif</b> (else if) clause takes the form :
 *
 * <pre>
 * <b>elseif</b> <i>condition operation(s)</i>
 * </pre>
 *
 * If the condition is satisfied, the operations are applied to the value and
 * the process is complete. If the condition is not satisfied, processing moves
 * on to any <b>elseif</b> or <b>else</b> clauses that follow, but if there are
 * none, the topic view evaluation does not proceed and no reference topic is
 * created. Note that <b>elseif</b> can be abbreviated to <b>elsf</b>.
 * <p>
 * An <b>else</b> clause takes the form :
 *
 * <pre>
 * <b>else</b> <i>operation(s)</i>
 * </pre>
 *
 * And will only be reached if no previous <b>if</b> or <b>elseif</b> conditions
 * were satisfied. If reached then the operations are applied to the value and
 * the topic view evaluation proceeds. The <b>continue</b> operation may be used
 * to proceed with an unchanged value.
 * <p>
 * </dd>
 * <dt>Process conditions</dt>
 * <dd>
 * <p>
 * A condition is of the form:
 *
 * <pre>
 * <i>pointer operator constant</i>
 * </pre>
 *
 * Where <i>pointer</i> is a JSON pointer, <i>operator</i> is a relational
 * operator and <i>constant</i> is a string, integer, or boolean value.
 *
 * for example:-
 *
 * <pre>
 * /Age &gt; 40
 * /Name = &quot;Bill&quot;
 * /Manager eq true
 * </pre>
 *
 * Operators allowed are:
 * <table border = 1>
 * <tr>
 * <th style= "text-align:center;">Operator</th>
 * <th style= "text-align:center;">Variant</th>
 * <th style= "text-align:center;">Description</th>
 * <th style= "text-align:center;">Supported JSON types</th>
 * </tr>
 * <tr>
 * <tr>
 * <td style="text-align:center;">=</td>
 * <td style="text-align:center;">eq</td>
 * <td style="text-align:left;">Equals</td>
 * <td style="text-align:left;">All</td>
 * </tr>
 * <tr>
 * <td style="text-align:center;">&gt;</td>
 * <td style="text-align:center;">gt</td>
 * <td style="text-align:left;">Greater than</td>
 * <td style="text-align:left;">Integer only</td>
 * </tr>
 * <tr>
 * <td style="text-align:center;">&lt;</td>
 * <td style="text-align:center;">lt</td>
 * <td style="text-align:left;">Less than</td>
 * <td style="text-align:left;">Integer only</td>
 * </tr>
 * <tr>
 * <td style="text-align:center;">!=</td>
 * <td style="text-align:center;">ne</td>
 * <td style="text-align:left;">Not equals</td>
 * <td style="text-align:left;">All</td>
 * </tr>
 * <tr>
 * <td style="text-align:center;">&gt;=</td>
 * <td style="text-align:center;">ge</td>
 * <td style="text-align:left;">Greater than or equal</td>
 * <td style="text-align:left;">Integer only</td>
 * </tr>
 * <tr>
 * <td style="text-align:center;">&lt;=</td>
 * <td style="text-align:center;">le</td>
 * <td style="text-align:left;">Less than or equal</td>
 * <td style="text-align:left;">Integer only</td>
 * </tr>
 * </table>
 * <p>
 * Compound conditions are supported by means of boolean operators:
 * <p>
 * <code>|</code> or <code>or</code><br>
 * <code>&amp;</code> or <code>and</code>
 * <p>
 * For example:
 *
 * <pre>
 * /Age = 50 or /Age &gt; 80
 * /Age gt 50 &amp; /Department eq &quot;Accounts&quot;
 * </pre>
 *
 * Normal boolean precedence applies but brackets can be used to control
 * precedence. For example:
 *
 * <pre>
 * (/Age &gt; 50 or /Department eq &quot;Accounts&quot;) and /Band &gt; 3
 * </pre>
 *
 * Boolean 'not' is also allowed :
 *
 * <pre>
 * not (/Age &lt; 65 or /Retired eq false)
 * </pre>
 *
 * </dd>
 * </dl>
 *
 * <h4>Patch transformations</h4>
 * <p>
 * Patch transformations indicate that a JSON patch is to be applied to the
 * value.
 * <p>
 * The format of a patch transformation is
 *
 * <pre>
 * <b>patch</b> 'patch string'
 * </pre>
 *
 * The patch string should be formatted according to the JSON Patch standard
 * (see <a href="https://tools.ietf.org/html/rfc6902"> RFC 6902: JavaScript
 * Object Notation (JSON) Patch</a>).
 * <p>
 * Patches are a sequence of JSON Patch operations contained in an array. They
 * are applied as an atomic update to the previous value if the resulting update
 * is successfully calculated. The following patch will check the value at a
 * specific key and update if the expected value is correct:
 * <p>
 * [{"op":"test", "path":"/price", "value" : 22}, {"op":"add", "path":"/price",
 * "value": 23}]
 * <p>
 * The available operations are:
 *
 * <ul>
 * <li><strong>Add:</strong> {@code {"op": "add", "path": "/a/b/c", "value": [
 * "foo", "bar" ]}}
 * <li><strong>Remove:</strong> {@code {"op": "remove", "path": "/a/b/c"}}
 * <li><strong>Replace:</strong> {@code {"op": "replace", "path": "/a/b/c",
 * "value": 43}}
 * <li><strong>Move:</strong> {@code {"op": "move", "from": "/a/b/c", "path":
 * "/a/b/d"}}
 * <li><strong>Copy:</strong> {@code {"op": "copy", "from": "/a/b/c", "path":
 * "/a/b/e"}}
 * <li><strong>Test:</strong> {@code {"op": "test", "path": "/a/b/c", "value":
 * "foo"}}
 * </ul>
 *
 * <p>
 * The test operation checks that the CBOR representation of the value of a
 * topic is identical to the value provided in the patch after converting it to
 * CBOR. If the value is represented differently as CBOR, commonly due to
 * different key ordering, then the patch will return the index of the failed
 * operation. e.g the values {@code {"foo": "bar", "count": 43}} and
 * {@code {"count": 43, "foo": "bar"}} are unequal despite semantic equality due
 * to the differences in a byte for byte comparison.
 * <p>
 * The following patch clause would add the 'price' field and remove the 'name'
 * field from an input JSON object.
 *
 * <pre>
 * patch '[{"op":"add", "path":"/price", "value" : 22}, {"op":"remove", "path":"/name"}]'
 * </pre>
 *
 * Patches can only be applied to JSON arrays or objects and if they fail to
 * apply, no resulting reference topic will be created by the view. If an update
 * patch fails, any previously created reference topic would be removed.
 *
 * <h4>Insert transformations</h4>
 * <p>
 * Insert transformations are used to insert a value from another topic into the
 * current value.
 * <p>
 * The name of the topic to insert from can be specified in a similar way to the
 * path mapping in that constants, path directives, and scalar directives (but
 * not expand directives) may be used.
 * <p>
 * The value from the insertion topic (if found) is inserted into the 'current'
 * value at a specified key position. The current value may be the source topic
 * value, the value output from expand directives (in which case the insertion
 * applies to each value), or the value from a previous transformation.
 * Insertion topics may be {@link TopicType#JSON JSON}, {@link TopicType#STRING
 * STRING}, {@link TopicType#INT64 INT64}, or {@link TopicType#DOUBLE DOUBLE}.
 * <p>
 * If, when evaluating a topic view, the insertion topic is not found (or the
 * specified key within it is not found), or it is of an unsupported topic type,
 * an optional default value may be inserted, but if no default is specified
 * then no insertion takes place and the value is passed to the derived
 * reference topic unchanged.
 * <p>
 * The values of the insertion topics are only taken at the point when the
 * source topic is evaluated against the topic view (i.e. when the source topic
 * is updated). Changes to the value of the insertion topic that occur in the
 * interim are not applied to the derived reference topics.
 * <p>
 * The format of an insert transformation is
 *
 * <pre>
 * <b>insert</b> <i>path</i> [<b>key</b> <i>fromKey</i>] <b>at</b> <i>insertionKey</i>
 * [<b>default</b> <i>defaultValue</i>]
 * </pre>
 *
 * The <i>path</i> is specified in exactly the same way as for the path mapping
 * clause, except it may not contain expand directives. <em>path</em> directives
 * operate on the path of the source topic, whereas <em>scalar</em> directives
 * operate on the 'current' value as defined previously.
 * <p>
 * <b>key</b> is optional and can specify a <i>fromKey</i> which is a JSON
 * pointer indicating the data within the insertion topic that is to be
 * inserted. If no key is specified then the whole of the insertion topic value
 * is inserted.
 * <p>
 * <b>at</b> specifies the <i>insertionKey</i> which is a JSON pointer
 * indicating where to insert the insertion topic value. If a value already
 * exists at the specified key then it is replaced otherwise it is inserted.
 * Insertion can only occur if the parent of the key exists in the value and is
 * of a compatible type. Array pointers may only be used to replace existing
 * entries or append one greater than the last entry. The special pointer value
 * <code>/-</code> may be used to append to the end of an existing array.
 * <p>
 * <b>default</b> is optional and may be used to specify a string
 * <i>defaultValue</i> to be inserted if the insertion topic does not exist, it
 * is of an unsupported topic type, or the specified key within it could not be
 * found.
 * <p>
 * The following insert transformation would cause the whole value of the topic
 * named AnyTopic to be inserted into the current value at key /T, assuming that
 * the current value is an object.
 *
 * <pre>
 * insert AnyTopic at /T
 * </pre>
 *
 * The following insert transformation would cause the whole value of the topic
 * named AnyTopic to be inserted into the current value at key /T/MyKey,
 * assuming that an object with the key T exists in the current value.
 *
 * <pre>
 * insert AnyTopic at /T/MyKey
 * </pre>
 *
 * The following insert transformation would cause the whole value of the topic
 * named AnyTopic to be appended to the array at the key T in the current value.
 *
 * <pre>
 * insert AnyTopic at /T/-
 * </pre>
 *
 * The following insert transformation would cause the value at the key name
 * within the topic named AnyTopic to be appended to the array at the key T in
 * the current value.
 *
 * <pre>
 * insert AnyTopic key /name at /T/-
 * </pre>
 *
 * In the above examples no insertion would take place if the insertion value
 * was not found, but a default value can be specified to insert into the
 * current value in this case.
 *
 * <pre>
 * insert AnyTopic at /T/- default "unknown"
 * </pre>
 *
 * The path of the topic to insert from can be built from parts of the source
 * topic path and/or scalar values within the current value. For example:
 *
 * <pre>
 * insert AC/&lt;path(1,1)&gt;/&lt;scalar(/myval)&gt; at /T
 * </pre>
 *
 * <h3>Options</h3>
 * <p>
 * Options are specified after the mapping and any transformations. Any number
 * of options may be specified but any particular option may only be specified
 * once.
 * <h4>Topic property mapping</h4>
 * <p>
 * The {@link TopicSpecification topic specification} of a reference topic is
 * derived from the topic specification of the source topics. A reference topic
 * has the same topic type as its source topic.
 *
 * <p>
 * The topic properties of a reference topic are derived from the source topic.
 * Some topic properties can be tuned using the topic property mapping option.
 * The following table describes the behavior for each topic property.
 *
 * <br>
 *
 * <table>
 * <tr>
 * <th style= "text-align:left;">Source topic property</th>
 * <th>Reference topic specification default</th>
 * <th>Can be set by topic property mapping?</th>
 * <th>Notes</th>
 * </tr>
 * <tr>
 * <tr>
 * <th style="text-align:left;">
 * {@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#COMPRESSION
 * COMPRESSION}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#CONFLATION
 * CONFLATION}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#DONT_RETAIN_VALUE
 * DONT_RETAIN_VALUE}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#OWNER
 * OWNER}</th>
 * <td>Not set</td>
 * <td>No</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#PERSISTENT
 * PERSISTENT}</th>
 * <td>Not set</td>
 * <td>No</td>
 * <td>Reference topics are not persisted. Topic views are persisted, so a
 * reference topic will be recreated on server restart if its source is
 * persistent.</td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#PRIORITY
 * PRIORITY}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#PUBLISH_VALUES_ONLY
 * PUBLISH_VALUES_ONLY}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#REMOVAL
 * REMOVAL}</th>
 * <td>Not set</td>
 * <td>No</td>
 * <td>Reference topics cannot be removed directly.</td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#SCHEMA
 * SCHEMA}</th>
 * <td>Copied from source topic specification</td>
 * <td>No</td>
 * <td>A {@link TopicType#RECORD_V2 RECORD_V2} reference topic has the same
 * schema as its source topic.</td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#TIDY_ON_UNSUBSCRIBE
 * TIDY_ON_UNSUBSCRIBE}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#TIME_SERIES_EVENT_VALUE_TYPE
 * TIME_SERIES_EVENT_VALUE_TYPE}</th>
 * <td>Copied from source topic specification</td>
 * <td>No</td>
 * <td>A {@link TopicType#TIME_SERIES TIME_SERIES} reference topic has the same
 * value type as its source topic.</td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#TIME_SERIES_RETAINED_RANGE
 * TIME_SERIES_RETAINED_RANGE}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes, with restrictions</td>
 * <td>A topic property mapping cannot increase the time series retained range
 * by overriding the {@code TIME_SERIES_RETAINED_RANGE} property. The retained
 * range of a reference time series topic will be constrained to be no greater
 * than that of its source topic.</td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#TIME_SERIES_SUBSCRIPTION_RANGE
 * TIME_SERIES_SUBSCRIPTION_RANGE}</th>
 * <td>Copied from source topic specification</td>
 * <td>Yes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <th style=
 * "text-align:left;">{@link com.pushtechnology.diffusion.client.topics.details.TopicSpecification#VALIDATE_VALUES
 * VALIDATE_VALUES}</th>
 * <td>Not set</td>
 * <td>No</td>
 * <td>A reference topic reflects updates to its source topic. It cannot reject
 * updates.</td>
 * </tr>
 * </table>
 *
 * <p>
 * A topic property option begins with the keywords {@code with properties} and
 * consists of a comma-separated list of topic property keys and values, each
 * separated by a colon. For example, the following topic view specification
 * maps all topics below the path <code>a</code> to reference topics below the
 * path <code>b</code>, and disables both conflation and compression for the
 * reference topics.
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; with properties CONFLATION:off, COMPRESSION:false
 * </pre>
 *
 * <h4>Topic value option</h4>
 * <p>
 * By default, a reference topic's value is a copy of the source topic value, or
 * part of the source value produced by an expand path mapping directive and/or
 * modified by transformations. For {@link TopicType#JSON JSON} source topics or
 * {@link TopicType#TIME_SERIES TIME_SERIES} topics with a {@code JSON} event
 * type, the value option can be applied to extract part of the resulting value
 * (the latest value in the case of TIME_SERIES topics).
 *
 * <p>
 * A topic value option begins with the keyword {@code as} and is followed by a
 * value directive. A value directive is delimited by angle brackets ({@code <},
 * {@code >}), and consists of the {@code value} keywords and a single JSON
 * pointer parameter. The JSON pointer selects the part of the current value to
 * copy.
 *
 * For example, given a current value of
 *
 * <pre>
 * {
 *   "account" : "1234",
 *   "balance" : { "amount" : 12.57, "currency" : "USD" }
 * }
 * </pre>
 *
 * and the value option {@code as <value(/balance)>}, the reference topic value
 * will be
 *
 * <pre>
 * {
 *   "amount" : 12.57,
 *   "currency" : "USD"
 * }
 * </pre>
 *
 * <p>
 * Value mappings that follow expand directives and/or transformations apply to
 * the current derived value and not the source topic value.
 *
 * <p>
 * Topic value mappings only alter the reference topic value; only the path
 * mapping determines whether a reference topic should exist. If the topic value
 * mapping's JSON pointer fails to select anything from the source topic value,
 * the reference topic will have the JSON value {@code null}.
 *
 * <p>
 * Topic value mappings are often used with path value mappings to avoid
 * repeating information in the path and the value. For example:
 *
 * <pre>
 * map ?accounts// to balances/&lt;scalar(/account)&gt; as &lt;value(/balance)&gt;
 * </pre>
 *
 * <h4>Throttle option</h4>
 * <p>
 * The throttle option can be used to constrain the rate at which a reference
 * topic is updated when its source topic is updated. The primary application of
 * a throttle option is to restrict the number of updates sent to reference
 * topic subscribers, reducing network utilization or the processing each
 * subscriber must do. Throttling also restricts the rate at which client
 * sessions can observe changes to reference topic values using the fetch API.
 *
 * <p>
 * The throttle option has the form
 * <code>throttle to <em>X</em> updates every <em>period</em></code>, where
 * <em>X</em> is a positive integer, and <em>period</em> is a positive integer
 * followed by a time unit which is one of <code>seconds</code>,
 * <code>minutes</code>, or <code>hours</code>.
 *
 * <p>
 * For example, the following topic view specification maps all topics below the
 * path <code>a</code> to reference topics below the path <code>b</code>, but
 * updates the value of each reference topic at most twice every five seconds:
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; throttle to 2 updates every 5 seconds
 * </pre>
 *
 * <p>
 * To improve readability, the throttle option allows <code>1 update</code> as
 * an alternative to <code>1 updates</code>, and <code>every second</code> as an
 * alternative to <code>every 1 seconds</code> (and so on, for other time
 * units). For example, the following topic view specification maps all topics
 * below the path <code>a</code> to reference topics below the path
 * <code>b</code>, but updates the value of each reference topic at most once
 * every hour:
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; throttle to 1 update every minute
 * </pre>
 *
 * <p>
 * The throttle option is only applied when a source topic is updated more
 * frequently than the configured rate. If a source topic is updated less
 * frequently, updates are passed on unconstrained. If the rate is exceeded, a
 * reference topic will not be updated again until the configured period has
 * expired. At this time, the reference topic will be updated based on the
 * source topic updates that happened in the interim, and a single value will be
 * published. Thus, the throttle option provides <em>topic-scoped
 * conflation</em>.
 *
 * <p>
 * The throttle option is ignored for time series topics because time series
 * updates do not support efficient conflation. Updates to source time series
 * topics are passed on immediately to the corresponding reference topics,
 * regardless of any throttle clause.
 *
 * <h4>Delay option</h4>
 * <p>
 * The delay option causes a change to a view's source topic to be delayed by a
 * fixed time before it is reflected in reference topics. Topic additions,
 * updates, and removals are all delayed. Delays can range from one second to
 * many days​.
 *
 * <p>
 * Such a publication delay is a useful way to devalue topic data so it can be
 * given away to non-paying users​.
 *
 * <p>
 * The delay option has the form <code>delay by <em>duration</em></code>, where
 * <em>duration</em> is a positive integer followed by a time unit which is one
 * of <code>seconds</code>, <code>minutes</code>, or <code>hours</code>.
 *
 * <p>
 * For example, the following topic view specification maps all topics below the
 * path <code>a</code> to reference topics below the path <code>b</code>, but
 * changes to a source topic are delayed by five minutes before they are
 * reflected in the corresponding reference topic.
 *
 * <pre>
 * map ?a// to b/&lt;path(1)&gt; delay by 5 minutes
 * </pre>
 *
 * <p>
 * Views with the delay option specified initially create reference topics in an
 * unpublished state. The topics are published once the delay time has expired.
 * A topic in the unpublished state prevents a lower priority topic view from
 * creating a reference topic with the same path. Sessions with the rights to
 * read the source topic can browse unpublished topics using the
 * {@link Topics.FetchRequest#withUnpublishedDelayedTopics
 * withUnpublishedDelayedTopics} fetch request option.
 *
 *
 * <h4>Separator option</h4>
 * <p>
 * Views can use the scalar and expand directives in path mappings to extract
 * text from the source value. By default, any {@code /} characters in the text
 * are interpreted as path separators and will introduce extra levels in
 * reference topic paths. If this is undesirable, the separator option can be
 * used to replace any {@code /} characters produced by the scalar and expand
 * directives with a constant string. The replacement can contain path
 * separators but must have no empty path segments ({@code //}).
 *
 * <p>
 * Here is an example replacing all path separators with an alternate character:
 *
 * <pre>
 * map ?a/path/ to b/&lt;scalar(/x/y)&gt; separator '%'
 * </pre>
 *
 * <p>
 * If the value at '/x/y' in the source topic was foo/bar, this would cause the
 * reference topic to be created at /b/foo%bar instead of /b/foo/bar.
 *
 * <h4>Preserve topics option</h4>
 * <p>
 * The default behavior for a topic view is that only the reference topics that
 * can be derived from the current value of the source topic are maintained.
 * This applies to views using directives that derive the path of the reference
 * topic(s) from a value within the source topic (e.g. scalar or expand). For
 * example if a view uses a scalar directive to determine the path of the
 * reference topic and the source topic is updated such that the selected scalar
 * value changes then the previously created reference topic will be removed and
 * a new one created. Similarly an expand directive may create many reference
 * topics from the source value but if the source is updated then only those
 * reference topics that still have a corresponding value within the source
 * value will be updated (or new ones added if not in the previous value),
 * whilst others will be removed.
 * <p>
 * There are situations where an application may not want reference topics to be
 * removed because of source value changes. In such cases the 'preserve topics'
 * option may be used to indicate that all reference topics created by a view
 * should be retained (and updated by later updates that apply to it) until
 * either the source topic or the creating view are removed.
 * <p>
 * For example:
 *
 * <pre>
 * map ?a/path/ to b/&lt;expand()&gt; preserve topics
 * </pre>
 *
 * <h4>Topic type option</h4>
 * <p>
 * The 'type' option can be used to specify the {@link TopicType} of the target
 * reference topic. If the current source value of type indicated by the source
 * topic's type can be converted to the target type, a reference topic of the
 * specified type will be created.
 * <p>
 * For example:
 *
 * <pre>
 * map ?a/ to b/&lt;path(1)&gt; type STRING
 * </pre>
 *
 * The specified type must be one of the supported target types
 * ({@link TopicType#STRING STRING}, {@link TopicType#INT64 INT64},
 * {@link TopicType#DOUBLE DOUBLE}, {@link TopicType#JSON JSON},
 * {@link TopicType#TIME_SERIES TIME_SERIES}, or {@link TopicType#BINARY
 * BINARY}), case insensitive.
 * <p>
 * The following table describes the supported conversions from the source topic
 * type (the left column) to the supported target types. The number in brackets
 * indicates a note at the foot of the table describing exactly how the
 * conversion is processed. Where there is no number, no conversion is necessary
 * and the derived source value is simply mapped to the target reference topic
 * as if the 'type' option was not specified. Where there is no entry for the
 * source topic type in the left column, no conversion is supported and topics
 * of unsupported types will not be processed by the topic view. An 'x'
 * indicates that the conversion is not supported.
 * <p>
 * In all cases the value being processed will be the 'current' value as derived
 * from other mappings within the topic view (e.g. expand) which is not
 * necessarily the value of the source topic. <br>
 * <table border = 1>
 * <tr>
 * <th style= "text-align:left;">Source topic</th>
 * <th style= "text-align:center;">{@link TopicType#STRING STRING}</th>
 * <th style= "text-align:center;">{@link TopicType#INT64 INT64}</th>
 * <th style= "text-align:center;">{@link TopicType#DOUBLE DOUBLE}</th>
 * <th style= "text-align:center;">{@link TopicType#JSON JSON}</th>
 * <th style= "text-align:center;">{@link TopicType#TIME_SERIES
 * TIME_SERIES}</th>
 * <th style= "text-align:center;">{@link TopicType#BINARY BINARY}</th>
 * </tr>
 * <tr>
 * <tr>
 * <th style="text-align:left;">{@link TopicType#STRING STRING}</th>
 * <td style="text-align:center;"></td>
 * <td style="text-align:center;">(1)</td>
 * <td style="text-align:center;">(1)</td>
 * <td style="text-align:center;">(2)</td>
 * <td style="text-align:center;">(3)</td>
 * <td style="text-align:center;">x</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link TopicType#INT64 INT64}</th>
 * <td style="text-align:center;">(4)</td>
 * <td style="text-align:center;"></td>
 * <td style="text-align:center;">(5)</td>
 * <td style="text-align:center;">(2)</td>
 * <td style="text-align:center;">(3)</td>
 * <td style="text-align:center;">x</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link TopicType#DOUBLE DOUBLE}</th>
 * <td style="text-align:center;">(4)</td>
 * <td style="text-align:center;">(6)</td>
 * <td style="text-align:center;"></td>
 * <td style="text-align:center;">(2)</td>
 * <td style="text-align:center;">(3)</td>
 * <td style="text-align:center;">x</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link TopicType#JSON JSON}</th>
 * <td style="text-align:center;">(7)</td>
 * <td style="text-align:center;">(7)</td>
 * <td style="text-align:center;">(7)</td>
 * <td style="text-align:center;"></td>
 * <td style="text-align:center;">(3)</td>
 * <td style="text-align:center;">x</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link TopicType#TIME_SERIES TIME_SERIES}</th>
 * <td style="text-align:center;">(8)</td>
 * <td style="text-align:center;">(8)</td>
 * <td style="text-align:center;">(8)</td>
 * <td style="text-align:center;">(8)</td>
 * <td style="text-align:center;"></td>
 * <td style="text-align:center;">(9)</td>
 * </tr>
 * <tr>
 * <th style="text-align:left;">{@link TopicType#BINARY BINARY}</th>
 * <td style="text-align:center;">x</td>
 * <td style="text-align:center;">x</td>
 * <td style="text-align:center;">x</td>
 * <td style="text-align:center;">x</td>
 * <td style="text-align:center;">(9)</td>
 * <td style="text-align:center;"></td>
 * </tr>
 * </table>
 * Conversion notes:-
 * <ol>
 * <li>{@link TopicType#STRING STRING} to number conversions will only occur if
 * the value of the string can be converted to the target number type. If the
 * string cannot be converted then no reference topic will be created.</li>
 * <li>Primitive types to {@link TopicType#JSON JSON} will result in a JSON
 * topic containing just the scalar value.</li>
 * <li>Conversions to {@link TopicType#TIME_SERIES TIME_SERIES} will result in a
 * time series topic with an event type matching the source topic. Every update
 * to the source topic will result in a new value being appended to the
 * reference time series topic. It is not possible to convert to a time series
 * topic with a different event type from the source topic.</li>
 * <li>Conversions from number types to {@link TopicType#STRING STRING} will
 * result in a simple string representation of the number in the reference topic
 * value.</li>
 * <li>{@link TopicType#INT64 INT64} to {@link TopicType#DOUBLE DOUBLE}
 * conversions perform a simple conversion. For example, 123 becomes 123.0.</li>
 * <li>{@link TopicType#DOUBLE DOUBLE} to {@link TopicType#INT64 INT64}
 * conversions perform rounding to the nearest integer value. For example 12.51
 * becomes 13.</li>
 * <li>{@link TopicType#JSON JSON} to primitive type conversions only occur if
 * the JSON value is a scalar which can be read as a string and converted to the
 * target type. Currently, only string and integer scalar values can be read as
 * a string (i.e. not doubles). The string representation of the value will be
 * converted in the same way as specified for {@link TopicType#STRING STRING} to
 * other primitive types. If the JSON value is a structure or cannot be
 * converted then no conversion takes place and no reference topic will be
 * created.</li>
 * <li>The conversion of {@link TopicType#TIME_SERIES TIME_SERIES} to other
 * types follows the same rules as for conversion from the source topic type
 * that matches the source time series topic's event value type. So if the time
 * series event type is 'double' then the conversion rules from source topic
 * type {@link TopicType#DOUBLE DOUBLE} to the target type will apply. Each
 * value appended to the source time series topic will result in an update to
 * the reference topic. If a failure to convert occurs at any point then the
 * reference topic would be removed and only recreated if a value is appended
 * that can be converted.</li>
 * <li>{@link TopicType#BINARY BINARY} to {@link TopicType#TIME_SERIES
 * TIME_SERIES}, and vice-versa is supported in the same way as for other time
 * series conversions.</li>
 * </ol>
 *
 * <h3>Escaping and quoting special characters</h3>
 * <p>
 * Each part of a topic view expression has characters with special
 * significance. Source topic clauses and path mapping clauses are delimited by
 * white space. Directives in path and topic property mapping clauses are
 * delimited by the {@code <} and {@code >} characters, and each directive
 * parameter is terminated by {@code ,} or {@code )}. Topic property mapping
 * clauses are delimited by white space, and the {@code :} and {@code ,}
 * characters.
 *
 * <p>
 * Sometimes a topic view must refer to or generate topics with paths that
 * containing special characters, or use a JSON pointer containing special
 * characters. The escape sequence {@code \x} can be used to literally insert
 * any character {@code x}, with a one exception: {@code \/} cannot be used in
 * path fragments since the path delimiter {@code /} is always significant.
 *
 * <p>
 * Here is an example topic view expression containing escape sequences. It maps
 * the topic path {@code a topic} a reference topic with the path
 * {@code another topic}.
 *
 * <pre>
 * map a\ topic to another\ topic
 * </pre>
 *
 * <p>
 * Here is an example with a source value directive that uses the JSON pointer
 * {@code /x()/y} to extract the target path from the source value. The
 * {@code )} character in the JSON pointer must be escaped so it is not treated
 * as the end of the parameter list.
 *
 * <pre>
 * map ?a// to &lt;scalar(/x(\)/y)&gt;
 * </pre>
 *
 * <p>
 * To insert {@code \}, the escape sequence {@code \\} must be used.
 *
 * <p>
 * There is no need to escape white space in JSON pointers directive parameters.
 * However, white space is significant. For example, the following expressions
 * have different topic value mapping clauses since the JSON pointer in the
 * second expression is {@code /x }; that is, it has a trailing space:
 *
 * <pre>
 * map a to b as &lt;value(/x)&gt;
 * map a to b as &lt;value(/x )&gt;
 * </pre>
 *
 * <p>
 * Instead of using escape sequences, white space characters can be included in
 * source topic clauses and path mapping clauses using quotes. A clause is
 * quoted by wrapping it in single quote ({@code '}) or double quote ({@code "})
 * characters. For example:
 *
 * <pre>
 * map "a topic" to "another topic"
 * </pre>
 *
 * <p>
 * Within a quoted clause, quotes of the same type must be escaped:
 *
 * <pre>
 * map 'alice\'s topic' to 'bob\'s topic'
 * </pre>
 *
 * <p>
 * For consistency, the values in topic property mapping clauses can be escaped
 * or quoted. However, there is no need to do so because none of the valid
 * values for the mappable properties contain special characters.
 *
 *
 * <h2><a id="path-conflicts">Dealing with topic path conflicts</a></h2>
 *
 * <p>
 * Reference topics have a lower priority than normal topics created through the
 * API, including replicas of normal topics created by topic replication or
 * fan-out. A reference topic will only be created if no topic or reference
 * topic is already bound to its derived topic path.
 *
 * <p>
 * Topic views have a precedence based on order of creation. If two topic views
 * define mappings the same topic path, the earliest-created topic view will
 * create a reference topic. If a topic view is updated, it retains its original
 * precedence.
 *
 * <h2>Remote topic views</h2>
 *
 * <p>
 * A remote topic view is one that specifies another server as the location of
 * the source topics using the <em>from</em> clause as shown in the example
 * below:
 *
 * <pre>
 * map ?a// from server1 to b/&lt;path(1)&gt;
 * </pre>
 * <p>
 * The server name ({@code server1} in this example) refers to the name of a
 * remote server created using the {@link RemoteServers} feature.
 *
 * <p>
 * In this case, upon establishing a successful connection with the remote
 * server indicated the topic view will create reference topics locally based
 * upon the topics selected by the topic view's selector at the remote server.
 * It is important to note that the selector <em>only</em> refers to topics that
 * match it at the remote server and not on the local server and there is no
 * reason why there could not be a source topic at the remote server that has
 * the same path as an entirely different topic on the local server.
 *
 * <p>
 * More than one topic view can specify the same remote server.
 *
 * <p>
 * A remote server only makes a physical connection when it is in use, therefore
 * the first topic view that specifies a remote server will cause it to
 * establish a connection. Similarly, if the last topic view that uses a remote
 * server is removed then the connection will be closed.
 *
 * <p>
 * It is not necessary for the named remote server definition to exist before
 * creating the topic view, as if it does not then the topic view will simply
 * remain dormant until the remote server is created and a successful connection
 * to the server specified in its URL is established. Similarly, if a remote
 * server that is in use by remote topic views is removed then all of the
 * reference topics created by the topic views will be removed and the topic
 * views will become dormant until the named remote server is created again or
 * the views are changed to name a different remote server.
 *
 * <p>
 * If a remote topic view selects a {@link TopicType#ROUTING ROUTING} topic at
 * the remote server then local mappings will only be performed if the routing
 * topic mapping at the remote server is able to establish a mapping for the
 * remote server connection. The mapping will be done as if from the resolved
 * routing topic.
 *
 * <p>
 * The rules of precedence for remote topic views are the same as for other
 * topic views. If the remote server for a remote topic view does not exist or
 * does not have an established connection then the remote topic view is not
 * evaluated (i.e. it is as if the source topics for the view did not exist),
 * but if the remote server later connects then the view will be evaluated and
 * rules of precedence will determine whether reference topic will replace those
 * created by earlier views.
 *
 * <h2>Topic view persistence and replication</h2>
 * <p>
 * Reference topics are neither replicated nor persisted. They are created and
 * removed based on their source topics. However, topic views are replicated and
 * persisted. A server that restarts will restore topic views during recovery.
 * Each topic view will then create reference topics based on the source topics
 * that have been recovered.
 *
 * <p>
 * The server records all changes to topic views in a persistent store. Topic
 * views are restored if the server is started.
 *
 * <p>
 * If a server belongs to a cluster, topic views (and remote servers) will be
 * replicated to each server in the cluster. Topic views are evaluated locally
 * within a server. Replicated topic views that select non-replicated source
 * topics can create different reference topics on each server in the cluster.
 * When remote topic views are in use, each server in the cluster will make a
 * connection to the specified remote server and will separately manage their
 * remote topic views.
 *
 * <p>
 * A view with a delay clause uses temporary storage to record delayed events.
 * If there is a high volume of updates, temporary per-server disk files will be
 * used to save server memory​. The storage is per-server, and does not survive
 * server restart. When a server is started, no data will be published by a view
 * with a delay clause until the delay time has expired.
 *
 * <h2>Access control</h2>
 * <p>
 * The following access control restrictions are applied:
 *
 * <ul>
 * <li>To {@link #listTopicViews() list the topic views}, a session needs the
 * {@link GlobalPermission#READ_TOPIC_VIEWS READ_TOPIC_VIEWS} global permission.
 *
 * <li>To {@link #createTopicView(String, String) create, replace}, or
 * {@link #removeTopicView(String) remove} a topic view, a session needs the
 * {@link GlobalPermission#MODIFY_TOPIC_VIEWS MODIFY_TOPIC_VIEWS} global
 * permission and {@link PathPermission#SELECT_TOPIC SELECT_TOPIC} permission
 * for the path prefix of the source topic selector.
 *
 * <li>Each topic view records the principal and security roles of the session
 * that created it as the <em>topic view security context</em>. When a topic
 * view is evaluated, this security context is used to constrain the creation of
 * reference topics. A reference topic will only be created if the security
 * context has {@link PathPermission#READ_TOPIC READ_TOPIC} permission for the
 * source topic path, and {@link PathPermission#MODIFY_TOPIC MODIFY_TOPIC}
 * permission for the reference topic path. The topic view security context is
 * copied from the creating session at the time the topic view is created or
 * replaced, and is persisted with the topic view. The topic view security
 * context is not updated if the roles associated with the session are changed.
 *
 * </ul>
 *
 * <h2>Accessing the feature</h2>
 * <p>
 * This feature may be obtained from a {@link Session session} as follows:
 *
 * <pre>
 * TopicViews topicViews = session.feature(TopicViews.class);
 * </pre>
 *
 * <p>
 * This feature is also extended by the {@link Topics Topics} feature. This
 * means is it possible to use the methods described here through the
 * {@link Topics Topics} feature.
 *
 * @author DiffusionData Limited
 * @since 6.3
 */
public interface TopicViews extends Feature {
    /**
     * Create a new named topic view.
     * <p>
     * If a view with the same name already exists the new view will update the
     * existing view.
     *
     * @param name the name of the view. If the name is empty, the operation
     *        will throw an InvalidArgumentException.
     * @param specification the <a href="./TopicViews.html#view-specification">
     *        specification</a> of the topic view
     * @return a CompletableFuture that completes when a response is received
     *         from the server, returning the topic view created by the
     *         operation.
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link ScriptException} &ndash; if {@code specification} is
     *         invalid;
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have MODIFY_TOPIC_VIEW permission or appropriate
     *         path prefix permissions;
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     */
    CompletableFuture<TopicView> createTopicView(String name,
        String specification);

    /**
     * List all the topic views that have been created.
     *
     * @return a CompletableFuture that completes when a response is received
     *         from the server, returning a list of views sorted by their
     *         creation order.
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have READ_TOPIC_VIEW permission or appropriate
     *         path prefix permissions;
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     */
    CompletableFuture<List<TopicView>> listTopicViews();

    /**
     * Get a named Topic View.
     * <p>
     * If the named view does not exist the completable future will complete
     * successfully with a null result.
     *
     * @param name the name of the view
     * @return a CompletableFuture that completes when a response is received
     *         from the server, returning a named view if it exists
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have READ_TOPIC_VIEW permission or appropriate
     *         path prefix permissions;
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     */
    CompletableFuture<TopicView> getTopicView(String name);

    /**
     * Remove a named topic view if it exists.
     * <p>
     * If the named view does not exist the completable future will complete
     * successfully.
     *
     * @param name the name of the view
     * @return a CompletableFuture that completes when a response is received
     *         from the server.
     *
     *         <p>
     *         If the task fails, the CompletableFuture will complete
     *         exceptionally with a {@link CompletionException}. Common reasons
     *         for failure, listed by the exception reported as the
     *         {@link CompletionException#getCause() cause}, include:
     *
     *         <ul>
     *         <li>{@link ClusterRoutingException} &ndash; if the operation
     *         failed due to a transient cluster error;
     *         <li>{@link PermissionsException} &ndash; if the calling
     *         session does not have MODIFY_TOPIC_VIEW permission or appropriate
     *         path prefix permissions;
     *         <li>{@link SessionClosedException} &ndash; if the session is
     *         closed.
     *         </ul>
     */
    CompletableFuture<?> removeTopicView(String name);
}
