/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.session;

import java.net.SocketAddress;
import java.util.List;

import javax.net.ssl.SSLContext;

import com.pushtechnology.diffusion.client.session.proxy.HTTPProxyAuthentication;
import com.pushtechnology.diffusion.client.session.reconnect.ReconnectionStrategy;
import com.pushtechnology.diffusion.client.session.retry.RetryStrategy;

/**
 * Session Attributes.
 * <P>
 * This encapsulates the basic attributes of a {@link Session}. These attributes
 * may be set (unless default values are sufficient) on the
 * {@link SessionFactory} used to create sessions.
 *
 * @author DiffusionData Limited
 * @since 5.0
 */
public interface SessionAttributes {

    /**
     * The default connection timeout = 2 seconds.
     */
    int DEFAULT_CONNECTION_TIMEOUT = 2000;

    /**
     * The default reconnection timeout = 60 seconds.
     *
     * @since 5.5
     */
    int DEFAULT_RECONNECTION_TIMEOUT = 60000;

    /**
     * The default write timeout for blocking writes = 2 seconds.
     */
    int DEFAULT_WRITE_TIMEOUT = 2000;

    /**
     * The default maximum message size in bytes. The maximum message size
     * limits the size of received messages. This default value is
     * is {@link Integer#MAX_VALUE}, so the message size is effectively
     * unlimited.
     */
    int DEFAULT_MAXIMUM_MESSAGE_SIZE = Integer.MAX_VALUE;

    /**
     * The minimum value that the maximum message size can be set to in bytes.
     */
    int MAXIMUM_MESSAGE_SIZE_MIN = 1024;

    /**
     * The default size of socket and message input buffers in bytes.
     */
    int DEFAULT_INPUT_BUFFER_SIZE = 1024 * 128;

    /**
     * The default size of socket and message output buffers in bytes.
     */
    int DEFAULT_OUTPUT_BUFFER_SIZE = 1024 * 128;

    /**
     * The default recovery buffer size in messages.
     *
     * @since 5.8
     */
    int DEFAULT_RECOVERY_BUFFER_SIZE = 128;

    /**
     * The default maximum outbound queue size in messages.
     *
     * @since 5.9
     */
    int DEFAULT_MAXIMUM_QUEUE_SIZE = 1000;

    /**
     * The default request path.
     *
     * @since 6.0
     */
    String DEFAULT_REQUEST_PATH = "/diffusion";

    /**
     * Returns the URL used to create the session.
     *
     * @return the URL
     */
    String getServerURL();

    /**
     * Returns the connection timeout value.
     *
     * @see SessionFactory #connectionTimeout(int)
     *
     * @return connection timeout in milliseconds
     */
    int getConnectionTimeout();

    /**
     * Returns the input buffer size for socket connection buffers and message
     * receiving buffers.
     *
     * @see SessionFactory#inputBufferSize(int)
     *
     * @return the input buffer size in bytes
     */
    int getInputBufferSize();

    /**
     * Returns the output buffer size for socket connection buffers and message
     * sending buffers.
     *
     * @see SessionFactory#outputBufferSize(int)
     *
     * @return the output buffer size in bytes
     */
    int getOutputBufferSize();

    /**
     * Returns the write timeout value.
     *
     * @see SessionFactory#writeTimeout(int)
     *
     * @return write timeout in milliseconds
     */
    int getWriteTimeout();

    /**
     * Returns the initial retry strategy.
     *
     * @see SessionFactory#initialRetryStrategy(RetryStrategy)
     *
     * @return the initial retry strategy used by the session
     */
    RetryStrategy getInitialRetryStrategy();

    /**
     * Returns the reconnection strategy.
     *
     * @see SessionFactory#reconnectionStrategy(ReconnectionStrategy)
     *
     * @return the reconnection strategy to be used by the session
     *
     * @since 5.5
     */
    ReconnectionStrategy getReconnectionStrategy();

    /**
     * Returns the reconnection timeout.
     *
     * @see SessionFactory#reconnectionTimeout(int)
     *
     * @return the reconnection timeout in milliseconds
     *
     * @since 5.5
     */
    int getReconnectionTimeout();

    /**
     * Returns the local socket address.
     *
     * @see SessionFactory#localSocketAddress(SocketAddress)
     *
     * @return the local socket address or null if none has been specified
     */
    SocketAddress getLocalSocketAddress();

    /**
     * Returns the SSL context for secure connections.
     *
     * @see SessionFactory#sslContext(SSLContext)
     *
     * @return the SSL context
     */
    SSLContext getSslContext();

    /**
     * Returns the maximum message size that will be accepted by this session.
     *
     * @see SessionFactory#maximumMessageSize(int)
     *
     * @return the maximum message size
     */
    int getMaximumMessageSize();

    /**
     * Returns the proxy host name used to route a connection to the server via
     * an HTTP proxy.
     *
     * @see SessionFactory#httpProxy(String, int)
     *
     * @return the proxy host
     *
     * @since 5.1
     */
    String getHttpProxyHost();

    /**
     * Returns the proxy port used to route a connection to the server via an
     * HTTP proxy.
     *
     * @see SessionFactory#httpProxy(String, int)
     *
     * @return the proxy port
     *
     * @since 5.1
     */
    int getHttpProxyPort();

    /**
     * Returns the proxy authenticator used to route a connection to the server
     * via an HTTP proxy.
     *
     * @see SessionFactory#httpProxy(String, int, HTTPProxyAuthentication)
     *
     * @return the proxy authenticator
     *
     * @since 5.1
     */
    HTTPProxyAuthentication getHttpProxyAuth();

    /**
     * Returns the host name or IP of the server the session will connect to.
     *
     * @see SessionFactory#serverHost(String)
     *
     * @return the host name (or IP address)
     *
     * @since 5.8
     */
    String getServerHost();

    /**
     * Returns the port of the server the session will connect to.
     *
     * @see SessionFactory#serverPort(int)
     *
     * @return the port
     *
     * @since 5.8
     */
    int getServerPort();

    /**
     * Returns the transports the session will use to connect to the server.
     *
     * @see SessionFactory#transports
     *
     * @return the transports in order of preference
     *
     * @since 5.8
     */
    List<Transport> getRequestedTransports();

    /**
     * Indicates whether the session will use transport layer security to
     * connect to Diffusion.
     *
     * @see SessionFactory#secureTransport(boolean)
     *
     * @return if the session will use TLS
     *
     * @since 5.8
     */
    boolean isTransportSecure();

    /**
     * Returns the path used for any HTTP requests made by the session.
     *
     * @see SessionFactory#requestPath(String)
     *
     * @return the path used in request URLs
     * @since 5.8
     */
    String getRequestPath();

    /**
     * Returns the maximum number of sent messages that can be recovered on
     * reconnection.
     *
     * @see SessionFactory#recoveryBufferSize(int)
     *
     * @return the size of the buffer used to store sent messages for sessions
     *         that support reconnection
     *
     * @since 5.8
     */
    int getRecoveryBufferSize();

    /**
     * Returns the maximum size of the outbound message queue for the
     * connection.
     *
     * @see SessionFactory#maximumQueueSize(int)
     *
     * @return the maximum number of messages that may be queued on the outbound
     *         message queue
     *
     * @since 5.9
     */
    int getMaximumQueueSize();

    /**
     * The transports available to Java clients.
     *
     * @since 5.8
     * @see SessionFactory
     */
    enum Transport {
        /**
         * WebSocket transport.
         */
        WEBSOCKET,
        /**
         * HTTP long polling transport.
         */
        HTTP_POLLING;
    }
}
