/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.session.reconnect;

/**
 * Interface that defines a reconnection strategy.
 * <P>
 * A reconnection strategy will be applied when the session enters the
 * {@link com.pushtechnology.diffusion.client.session.Session.State#RECOVERING_RECONNECT
 * RECOVERING_RECONNECT} state, allowing the session to attempt to reconnect and
 * recover its previous state.
 *
 * @author DiffusionData Limited
 * @since 5.5
 */
public interface ReconnectionStrategy {

    /**
     * This method is called before each reconnection attempt. The strategy
     * should either allow the reconnection attempt to proceed by calling
     * {@link ReconnectionAttempt#start}, or abort the reconnection by calling
     * {@link ReconnectionAttempt#abort}.
     * <P>
     * Strategies should avoid blocking operations. The
     * {@link ReconnectionAttempt} object can be called either directly or
     * asynchronously (for example, by a scheduled timer) depending on the
     * strategy.
     *
     * @param reconnection A single reconnection attempt that may be either
     *        started or aborted
     */
    void performReconnection(ReconnectionAttempt reconnection);

    /**
     * A single reconnection attempt that can either be started or aborted.
     */
    interface ReconnectionAttempt {

        /**
         * Start the reconnection attempt.
         */
        void start();

        /**
         * Abort the reconnection attempt. This will cause the session to
         * transition to
         * {@link com.pushtechnology.diffusion.client.session.Session.State#CLOSED_FAILED
         * CLOSED_FAILED}.
         */
        void abort();
    }
}
