/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.session.retry;

import java.util.Objects;

import com.pushtechnology.diffusion.client.session.SessionEstablishmentTransientException;

/**
 * Defines a retry strategy.
 * <P>
 * A retry strategy will be applied when an initial to attempt to open a session
 * fails with a {@link SessionEstablishmentTransientException}.
 * <p>
 * The strategy is defined in terms of the number of milliseconds between
 * retries and the maximum number of retries to attempt.
 *
 * @author DiffusionData Limited
 * @since 6.9
 */
public final class RetryStrategy {

    /**
     * The retry strategy that indicates that no retry is to be attempted.
     */
    public static final RetryStrategy NO_RETRY = new RetryStrategy();

    private final long interval;
    private final int attempts;

    /**
     * Creates a new retry strategy object.
     *
     * @param interval the number of milliseconds before the first retry and
     *        between subsequent retries
     *
     * @param attempts the number of retry attempts
     */
    public RetryStrategy(long interval, int attempts) {

        if (interval < 1) {
            throw new IllegalArgumentException("Invalid interval");
        }
        this.interval = interval;

        if (attempts < 1) {
            throw new IllegalArgumentException("Invalid attempts");
        }
        this.attempts = attempts;

    }

    /**
     * Creates a new retry strategy that will retry indefinitely at the
     * specified interval.
     *
     * @param interval the number of milliseconds before the first retry and
     *        between subsequent retries
     */
    public RetryStrategy(long interval) {
        if (interval < 1) {
            throw new IllegalArgumentException("Invalid interval");
        }
        this.interval = interval;
        this.attempts = Integer.MAX_VALUE;
    }

    /**
     * No retry constructor.
     */
    private RetryStrategy() {
        interval = 0;
        attempts = 0;
    }

    /**
     * Returns the time interval between attempts.
     *
     * @return the interval in ms.
     */
    public long getInterval() {
        return interval;
    }

    /**
     * Returns the number of retry attempts.
     *
     * @return the number of retry attempts, where {@code Integer.MAX_VALUE}
     * means infinite retries.
     */
    public int getAttempts() {
        return attempts;
    }

    @Override
    public int hashCode() {
        return Objects.hash(interval, attempts);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof RetryStrategy)) {
            return false;
        }
        final RetryStrategy other = (RetryStrategy) obj;
        return interval == other.interval && attempts == other.attempts;
    }

    @Override
    public String toString() {
        if (attempts == 0) {
            return "RetryStrategy [No retry]";
        }
        else if (attempts == Integer.MAX_VALUE) {
            return "RetryStrategy [interval=" + interval + "ms" +
                ", attempts=unlimited]";
        }
        else {
            return "RetryStrategy [interval=" + interval + "ms, attempts=" +
                attempts + "]";
        }
    }

}
