/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.topics;

/**
 * A {@code TopicSelector} is a value that identifies one or more topics.
 *
 * <p>
 * Depending on where it is used, a selector may be evaluated by either the
 * server or the client library. For example, the server evaluates the selector
 * used to subscribe to a topic. In contrast, the selector used to register a
 * topic handler is evaluated by the client. Applications do not normally need
 * to evaluate selectors, but may do so using {@link #selects(String) selects}.
 * <p>
 * A client may receive a selector providing the context for certain operations.
 * For example, to allow it to veto subscriptions. Limited ability to query the
 * selector's {@link #getType() type}, associated {@link #getExpression()
 * expression}, and {@link TopicPathSelector#getPath() topic path} is provided.
 * </p>
 *
 * <p>
 * Selectors may be compared using {@link Object#equals(Object) equals}.
 * Selectors of different types are never equal.
 * </p>
 *
 * Selectors may be created using {@link TopicSelectors}.
 *
 * @author DiffusionData Limited
 * @since 5.0
 * @see TopicSelectors
 */
public interface TopicSelector {

    /**
     * Prefix used for {@link TopicSelector.Type#PATH PATH} expressions.
     */
    char PATH_PREFIX = '>';

    /**
     * Prefix used for {@link TopicSelector.Type#SPLIT_PATH_PATTERN
     * SPLIT_PATH_PATTERN} expressions.
     */
    char SPLIT_PATH_PATTERN_PREFIX = '?';

    /**
     * Prefix used for {@link TopicSelector.Type#FULL_PATH_PATTERN
     * FULL_PATH_PATTERN} expressions.
     */
    char FULL_PATH_PATTERN_PREFIX = '*';

    /**
     * Prefix used for {@link TopicSelector.Type#SELECTOR_SET SET} expressions.
     */
    char SELECTOR_SET_PREFIX = '#';

    /**
     * The type of this selector.
     *
     * @return The selector type.
     */
    Type getType();

    /**
     * The expression associated with this selector.
     *
     * <p>
     * See {@link TopicSelectors#parse(String)} for the expression format for
     * {@code PATH}, {@code SPLIT_PATH_PATTERN}, {@code FULL_PATH_PATTERN}, and
     * {@code SELECTOR_SET} selectors. Expressions for these types of selectors
     * can be re-parsed, and {@code selector.equals(}
     * {@code Diffusion.topicSelectors().parse(selector.getExpression())}
     * {@code ) == true} .
     * </p>
     *
     * @return The expression.
     */
    String getExpression();

    /**
     * Get the topic path prefix from this selector pattern.
     *
     * <p>
     * Returns the largest fixed topic path that begins the selector expression.
     * For {@code PATH} selectors, this is the entire path. For
     * {@code SPLIT_PATH_PATTERN} or {@code FULL_PATH_PATTERN} selectors, this
     * is a topic path up to, but not including, the first part of the path that
     * contains a regular expression.
     * <p>
     * For {@code SELECTOR_SET}s, this method will return the largest prefix
     * that is common to all included selectors.
     * <p>
     * If there is no common prefix, an empty string will be returned.
     *
     * @return the topic path prefix
     * @since 5.1
     */
    String getPathPrefix();

    /**
     * Evaluate this selector against a topic path.
     *
     * @param topicPath The topic path.
     * @return {@code true} If this selector selects {@code topicPath}.
     */
    boolean selects(String topicPath);

    /**
     * Topic selector type.
     */
    enum Type {

        /**
         * The selector is a simple topic path.
         *
         * @see TopicSelectors#parse(String)
         */
        PATH(PATH_PREFIX),

        /**
         * The selector is a split-path pattern.
         *
         * @see TopicSelectors#parse(String)
         */
        SPLIT_PATH_PATTERN(SPLIT_PATH_PATTERN_PREFIX),

        /**
         * The selector is a full-path pattern.
         *
         * @see TopicSelectors#parse(String)
         */
        FULL_PATH_PATTERN(FULL_PATH_PATTERN_PREFIX),

        /**
         * The selector is a composite of other selectors.
         *
         * @see TopicSelectors#anyOf(TopicSelector...)
         */
        SELECTOR_SET(SELECTOR_SET_PREFIX);

        private final char expressionPrefix;

        Type(char expressionPrefix) {
            this.expressionPrefix = expressionPrefix;
        }

        /**
         * The expression prefix character used for this type.
         *
         * @return The prefix.
         * @see TopicSelectors#parse(String)
         */
        public char getExpressionPrefix() {
            return expressionPrefix;
        }
    }
}
