/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.client.topics.details;

import com.pushtechnology.diffusion.client.features.TimeSeries;
import com.pushtechnology.diffusion.client.features.control.topics.SessionTrees;
import com.pushtechnology.diffusion.client.features.control.topics.SubscriptionControl;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.DataTypes;
import com.pushtechnology.diffusion.datatype.recordv2.RecordV2;
import com.pushtechnology.diffusion.datatype.recordv2.RecordV2DataType;

/**
 * The topic type determines the type of the data values a topic publishes to
 * subscribers.
 *
 * <h3>Source Topics</h3>
 *
 * Most topics are source topics. The characteristics of each type of source
 * topic are summarized in the following table.
 *
 * <table>
 * <tr>
 * <th>Topic type</th>
 * <th>State</th>
 * <th>Data type</th>
 * </tr>
 * <tr>
 * <th>{@link #STRING}</th>
 * <td>Single scalar value.</td>
 * <td>String.</td>
 * </tr>
 * <tr>
 * <th>{@link #INT64}</th>
 * <td>Single scalar value.</td>
 * <td>64-bit integer.</td>
 * </tr>
 * <tr>
 * <th>{@link #DOUBLE}</th>
 * <td>Single scalar value.</td>
 * <td>Double precision floating point number.</td>
 * </tr>
 * <tr>
 * <th>{@link #BINARY}</th>
 * <td>Single scalar value.</td>
 * <td>Arbitrary binary data.</td>
 * </tr>
 * <tr>
 * <th>{@link #RECORD_V2}</th>
 * <td>Single composite value.</td>
 * <td>{@link RecordV2} &ndash; Diffusion-specific data type. A list of records,
 * each composed of field values, with an optional schema.</td>
 * </tr>
 * <tr>
 * <th>{@link #JSON}</th>
 * <td>Single composite value.</td>
 * <td>{@link com.pushtechnology.diffusion.datatype.json.JSON JSON}, backed by
 * CBOR-format binary.</td>
 * </tr>
 * <tr>
 * <th>{@link #TIME_SERIES}</th>
 * <td>Append-only log of events.</td>
 * <td>{@link com.pushtechnology.diffusion.client.features.TimeSeries.Event
 * TimeSeries.Event} containing a value of a {@link DataTypes well-known data
 * type}.</td>
 * </tr>
 *
 * </table>
 *
 * <h3>Routing Topics (deprecated)</h3>
 *
 * <p>
 * A {@link #ROUTING} topic can have a different source topic for each
 * subscription.
 * <p>
 * Routing topics republish values from source topics. The data type is
 * inherited from the source topic.
 * <p>Routing topics are deprecated in favor of {@link SessionTrees}.
 *
 * @author DiffusionData Limited
 * @since 5.0
 * @see TopicSpecification
 */
public enum TopicType {

    /**
     * Routing Topic.
     * <P>
     * A topic that can reference different source topics for different
     * sessions.
     * <P>
     * Each subscription to a routing topic is routed to a source topic. Updates
     * to the source topic are routed back to the subscriber and appear to come
     * from the routing topic.
     * <P>
     * The result is that a session may subscribe to a topic which is in reality
     * supported by another topic and the mapping of the routing topic to the
     * actual topic can be different for each session.
     * <P>
     * As an example, you may wish for all sessions to simply subscribe to a
     * topic called "Prices" but depending upon the client type the actual topic
     * could differ (Prices/Discount, Prices/Standard etc).
     * <P>
     * An instance of this topic may map any number of sessions to any number of
     * different source topics.
     * <P>
     * From the point of view of a session subscribing to such a topic, a
     * routing topic appears to be a normal topic but it has no state of its own
     * and cannot be updated.
     * <P>
     * The mapping of sessions to source topics is performed by a control client
     * session using the {@link SubscriptionControl} feature. When a session
     * subscribes to the routing topic the control client is requested to
     * provide the topic that the client is to be subscribed to. If there is no
     * control client available to handle subscriptions at the time a session
     * subscribed, the session will not be subscribed to the topic.
     * <P>
     * Alternatively, the routing can be determined by a user-written Java class
     * (deployed on the server) which will be invoked to define the mapping of
     * the topic to another data topic when a session subscribes.
     * <P>
     * When a source topic is removed that is mapped to from a routing topic
     * then any session that were mapped to that source topic will be
     * unsubscribed from the routing topic.
     *
     * @deprecated since 6.7
     *             <p>
     *             Routing topics are deprecated. The more powerful
     *             {@link SessionTrees} feature should be used in their place.
     */
    @Deprecated
    ROUTING,

    /**
     * Topic that stores and publishes JSON (JavaScript Object Notation) values.
     * Based on the {@link DataTypes#json JSON data type}.
     * <p>
     * Supports delta-streams.
     *
     * @since 5.7
     */
    JSON,

    /**
     * Topic that stores and publishes binary values. Based on the
     * {@link DataTypes#binary binary data type}.
     * <p>
     * Supports delta-streams.
     *
     * @since 5.7
     */
    BINARY,

    /**
     * Topic that stores and publishes data in the form of records and fields.
     * Based on the {@link RecordV2DataType RecordV2} data type.
     * <p>
     * Supports delta-streams.
     *
     * @since 6.0
     */
    RECORD_V2,

    /**
     * Topic that stores and publishes IEEE 754 double-precision floating point
     * numbers. Based on the {@link DataTypes#doubleFloat() double data type}.
     * <p>
     * Supports null Double values.
     * <p>
     * The topic does not support delta-streams — only complete values are
     * transmitted.
     *
     * @since 6.0
     */
    DOUBLE,

    /**
     * Topic that stores and publishes 64-bit integer values. Based on the
     * {@link DataTypes#int64() int64 data type}.
     * <p>
     * Supports null Long values.
     * <p>
     * The topic does not support delta-streams — only complete values are
     * transmitted.
     *
     * @since 6.0
     */
    INT64,

    /**
     * Topic that stores and publishes String values. Based on the
     * {@link DataTypes#string() string data type}.
     * <p>
     * Supports null String values.
     * <p>
     * Supports delta-streams.
     *
     * @since 6.0
     */
    STRING,

    /**
     * Time Series Topic.
     *
     * <p>
     * A <em>time series</em> is a sequence of events. Each event contains a
     * value and has server-assigned metadata comprised of a sequence number,
     * timestamp, and author.
     * <p>
     * A time series topic allows sessions to access a time series that is
     * maintained by the server. A time series topic has an associated
     * {@link DataType event data type}, such as {@code Binary}, {@code String},
     * or {@code JSON}, that determines the type of value associated with each
     * event.
     *
     * <h4>Retained range</h4>
     *
     * <p>
     * The {@link TopicSpecification#TIME_SERIES_SUBSCRIPTION_RANGE
     * TIME_SERIES_SUBSCRIPTION_RANGE} property configures the range of historic
     * events retained by a time series topic. If the property is not specified,
     * a time series topic will retain the ten most recent events.
     *
     * <h4>Subscription range</h4>
     *
     * <p>
     * The {@link TopicSpecification#TIME_SERIES_SUBSCRIPTION_RANGE
     * TIME_SERIES_SUBSCRIPTION_RANGE} property configures a time series topic
     * to send a range of historic events from the end of the time series to new
     * subscribers. This is a convenient way to synchronize new subscribers
     * without requiring the use of a {@link TimeSeries#rangeQuery() range
     * query}.
     * <p>
     * By default, new subscribers will be sent the latest event if delta
     * streams are enabled and no events if delta streams are disabled. See the
     * description of <em>Subscription range</em> in the {@link TimeSeries time
     * series feature} documentation.
     *
     *
     * <h4>Mandatory properties</h4>
     * <p>
     * The {@link TopicSpecification#TIME_SERIES_EVENT_VALUE_TYPE
     * TIME_SERIES_EVENT_VALUE_TYPE} property must be provided when creating a
     * time series topic.
     *
     *
     * @since 6.0
     * @see TimeSeries
     */
    TIME_SERIES,

    /**
     * A topic type that is unsupported by the session.
     *
     * @since 6.1
     */
    UNKNOWN_TOPIC_TYPE,
}
