/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

/**
 * Read-only interface for values that are internally represented as binary
 * data.
 *
 * <p>
 * This interface provides access to copies of the binary data, making instances
 * effectively immutable. Methods of derived interfaces and classes that relax
 * this restriction and expose access to the internal data should be clearly
 * documented.
 *
 * @author DiffusionData Limited
 * @since 5.7
 */
public interface Bytes {

    /**
     * @return the length of the data in bytes
     */
    int length();

    /**
     * Returns a stream that provides the data.
     *
     * @return a new input stream over the bytes. Implementations may share the
     *         underlying state with the {@code Bytes}. This minimises copying
     *         at the expense of retaining potentially inaccessible data for
     *         longer.
     */
    InputStream asInputStream();

    /**
     * Returns a copy of the data as a byte array.
     *
     * @return a copy of the binary data
     */
    byte[] toByteArray();

    /**
     * Write the bytes to an output stream.
     * <P>
     * This method avoids the allocation and copying costs of producing an
     * {@link #toByteArray() intermediate byte array}.
     *
     * @param target the target output stream
     * @throws IOException if writing to target causes an I/O error
     */
    void copyTo(OutputStream target) throws IOException;
}