/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype;

import java.io.IOException;
import java.io.OutputStream;

import com.pushtechnology.diffusion.client.features.Messaging;
import com.pushtechnology.diffusion.client.features.Messaging.RequestHandler;
import com.pushtechnology.diffusion.client.features.Messaging.RequestStream;
import com.pushtechnology.diffusion.client.features.TimeSeries.RangeQuery;
import com.pushtechnology.diffusion.client.features.Topics.FetchRequest;
import com.pushtechnology.diffusion.client.features.Topics.ValueStream;

/**
 * A data type is specified for a particular class (its <em>value class</em>).
 * It provides methods to convert values of the value class to and
 * from binary. Diffusion provides several {@link DataTypes implementations}.
 *
 * <p>
 * Value types should provide a {@link Object#toString() string representation}
 * and define reasonable {@link Object#equals(Object) equality}. The value type
 * implementations used by Diffusion-provided data types do so.
 * <p>
 * A data type can optionally support incremental changes to values, represented
 * by one or more types of <em>delta</em>. For each type of delta it supports,
 * the data type provides an implementation of {@link DeltaType} via
 * {@link #deltaType(String)} and {@link #deltaType(Class)}.
 *
 * @author DiffusionData Limited
 * @since 5.7
 * @param <V> value type
 * @see DeltaType
 */
public interface DataType<V> {

    /**
     * Returns the external type identifier.
     *
     * @return a unique name identifying this data type
     */
    String getTypeName();

    /**
     * Serialize a value to binary.
     *
     * @throws IOException if the OutputStream does
     * @throws IllegalArgumentException if the implementation does not support
     *         the provided value
     */
    void writeValue(V value, OutputStream out)
        throws IOException, IllegalArgumentException;

    /**
     * Returns the serialized form of {@code value} as a {@link Bytes}.
     *
     * @throws IllegalArgumentException if the implementation does not support
     *         the provided value
     * @since 6.0
     */
    Bytes toBytes(V value) throws IllegalArgumentException;

    /**
     * Create a value from binary.
     *
     * <p>
     * Implementations can choose not to fully validate values when they are
     * read, but instead defer parsing until it is required. See
     * {@link #validate(Object)}.
     *
     * @param bytes The binary data. The implementation may re-use the array to
     *        avoid copying so the caller must ensure the array is not modified.
     * @param offset start of the data within bytes
     * @param length length of the data within bytes
     * @throws InvalidDataException If {@code bytes} does not represent a valid
     *         {@code V}. The implementation may defer validation; see
     *         {@link #validate(Object)}.
     * @throws IndexOutOfBoundsException if either {@code offset} or
     *         {@code length} is negative, or
     *         {@code offset + length > bytes.length}
     */
    V readValue(byte[] bytes, int offset, int length)
        throws InvalidDataException, IndexOutOfBoundsException;

    /**
     * Create a value from binary. Equivalent to
     * {@link #readValue(byte[], int, int) readValue(in, 0, in.length)}.
     *
     * @param bytes The binary data. The implementation may re-use the array to
     *        avoid copying so the caller must ensure the array is not modified.
     * @throws InvalidDataException If {@code bytes} does not represent a valid
     *         {@code V}. The implementation may defer validation; see
     *         {@link #validate(Object)}.
     */
    V readValue(byte[] bytes) throws InvalidDataException;

    /**
     * Create a value from binary. Equivalent to
     * {@link #readValue(byte[]) readValue(bytes.toByteArray())}.
     *
     * @param bytes the binary data
     * @throws InvalidDataException If {@code bytes} does not represent a valid
     *         {@code V}. The implementation may defer validation; see
     *         {@link #validate(Object)}.
     */
    V readValue(Bytes bytes) throws InvalidDataException;

    /**
     * Check whether a value is valid.
     *
     * <p>
     * A {@code DataType} implementation is not required to check the binary
     * data supplied to {@link #readValue(byte[], int, int)} can be parsed as a
     * value of type {@code V} if doing so is unnecessarily costly. Instead a
     * value may simply wrap the binary data, and lazily deserialize as
     * required. This method can be used the check the value at a later time.
     *
     * @throws InvalidDataException if the value is invalid
     */
    void validate(V value) throws InvalidDataException;

    /**
     * Create a value of a compatible class from a binary.
     *
     * <p>
     * If {@code valueType} is incompatible with this data type, this method
     * will throw an IllegalArgumentException. Compatibility can be tested with
     * {@link #canReadAs(Class)}.
     *
     * @param <T> type of {@code classOfT}
     * @param classOfT the type of the result
     * @param bytes The binary data. The implementation may re-use the array to
     *        avoid copying so the caller must ensure the array is not modified.
     * @param offset start of the data within bytes
     * @param length length of the data within bytes
     * @throws InvalidDataException If {@code bytes} does not represent a valid
     *         {@code V}, and by extension cannot be provided as a {@code T}.
     *         The implementation may defer validation; see
     *         {@link #validate(Object)}.
     * @throws IllegalArgumentException if {@code classOfT} is
     *         incompatible with this data type
     * @throws IndexOutOfBoundsException if either {@code offset} or
     *         {@code length} is negative, or
     *         {@code offset + length > bytes.length}
     * @since 6.0
     */
    <T> T readAs(Class<T> classOfT, byte[] bytes, int offset, int length)
        throws InvalidDataException, IllegalArgumentException, IndexOutOfBoundsException;

    /**
     * Create a value of a compatible class from binary. Equivalent to
     * {@link #readAs(Class, byte[], int, int) readAs(classOfT, in, 0,
     * in.length)}.
     *
     * @param <T> type of {@code classOfT}
     * @param classOfT the type of the result
     * @param bytes The binary data. The implementation may re-use the array to
     *        avoid copying so the caller must ensure the array is not modified.
     * @throws InvalidDataException If {@code bytes} does not represent a valid
     *         {@code V}, and by extension cannot be provided as a {@code T}.
     *         The implementation may defer validation; see
     *         {@link #validate(Object)}.
     * @throws IllegalArgumentException if {@code classOfT} is
     *         incompatible with this data type
     * @since 6.0
     */
    <T> T readAs(Class<T> classOfT, byte[] bytes)
        throws InvalidDataException, IllegalArgumentException;

    /**
     * Create a value of a compatible class from binary. Equivalent to
     * {@link #readAs(Class, byte[]) readAs(classOfT, bytes.toByteArray())}.
     *
     * @param <T> type of {@code classOfT}
     * @param classOfT the type of the result
     * @param bytes the binary data
     * @throws InvalidDataException If {@code bytes} does not represent a valid
     *         {@code V}, and by extension cannot be provided as a {@code T}.
     *         The implementation may defer validation; see
     *         {@link #validate(Object)}.
     * @throws IllegalArgumentException if {@code classOfT} is
     *         incompatible with this data type
     * @since 6.0
     */
    <T> T readAs(Class<T> classOfT, Bytes bytes)
        throws InvalidDataException, IllegalArgumentException;

    /**
     * Test whether this data type is compatible with {@code classOfT}.
     * Compatibility with a {@code classOfT} means that any valid binary
     * representation of a {@code V} can be
     * {@link #readAs(Class, byte[], int, int) read as} an instance of
     * {@code classOfT}.
     *
     * <p>
     * Every data type should be compatible with the following:
     *
     * <ul>
     * <li>{@code Class<V>} &ndash; the class corresponding to implementation's
     * value type. For a data type with a value type of {@code X},
     * {@code readAs(X.class, bytes)} is equivalent to {@code readValue(bytes)}.
     * <li>{@code Class<? super V>} &ndash; any super type of the class
     * corresponding to implementation's value type. Consequently every data
     * type is compatible with {@code Class<Object>}.
     * <li>{@code Class<Bytes>}.
     * </ul>
     *
     * <p>
     * Compatibility is an asymmetric relationship. For example, the string data
     * type is compatible with {@code Class<JSON>}, but the JSON data type is
     * incompatible with {@code Class<<String>}.
     *
     * <p>
     * In addition to compatibility with {@code Class<V>} and
     * {@code Class<Bytes>}, the standard data types implement compatibility
     * according to the type hierarchy shown in the following diagram:
     *
     * <p>
     * <img src="doc-files/CompatibleTypeHierarchy.png" alt="Compatible type
     * hierarchy implemented by standard data types.">
     *
     * <p>
     * For example, the string data type is compatible with
     * {@code Class<String>}, {@code Class<JSON>} and {@code Class<Bytes>}, as
     * well as super types of {@code Class<String>} such as
     * {@code Class<Object>} and {@code Class<CharSequence>}.
     *
     * <p>
     * Data type compatibility is used in several places in the Diffusion API:
     * <ul>
     * <li>{@link ValueStream Value streams} only receive updates from topics
     * with compatible data types, and convert received values appropriately.
     * <li>{@link FetchRequest#withValues} constrains a fetch operation to
     * retrieve data from topics with compatible data types, and convert the
     * values appropriately.
     * <li>{@link RangeQuery#as} converts the values resulting from time series
     * range query appropriately; the query will fail if the time series topic
     * data type is an incompatible data type.
     * <li>Values sent using the {@link Messaging} feature will not be delivered
     * if the recipient {@link RequestStream request stream} or
     * {@link RequestHandler request handler} is incompatible with the value
     * data type.
     * </ul>
     *
     * @param <T> type of {@code classOfT}
     * @param classOfT the type to check
     * @return true if a binary representation created by this data type can
     *         read as an instance of classOfT
     * @since 6.0
     */
    <T> boolean canReadAs(Class<T> classOfT);

    // PlantUML source for Javadoc diagram.
    // @formatter:off
    // @startuml
    // skinparam monochrome true
    // skinparam backgroundcolor transparent
    // hide circle
    // hide members
    //
    // JSON -up-> Bytes
    // String -up-> JSON
    // Int64 -up-> JSON
    // Double -up-> JSON
    // Binary -up-> Bytes
    // RecordV2 -up-> Bytes
    // @enduml

    /**
     * Obtain a {@link DeltaType} by name.
     *
     * @param name the name, as returned by {@link DeltaType#getName()}
     * @return the delta type
     * @throws IllegalArgumentException if this data type does not provide a
     *         {@code DeltaType} with the name {@code typeName}
     * @see DeltaType
     */
    DeltaType<V, ?> deltaType(String name);

    /**
     * Obtain a {@link DeltaType} by class.
     *
     * @param deltaClass the class
     * @return the delta type
     * @throws IllegalArgumentException if this data type does not provide a
     *         {@code DeltaType} for deltas of class {@code deltaClass}
     * @throws IllegalArgumentException if the data type provides more than one
     *         {@code DeltaType} for deltas of class {@code deltaClass}, but
     *         none is preferred
     * @see DeltaType
     */
    <D> DeltaType<V, D> deltaType(Class<D> deltaClass);

    /**
     * Returns the binary delta type for this data type, if any.
     *
     * <p>This method behaves similarly to {@link #deltaType(Class)
     * deltaType(BinaryDelta.class)} except it returns {@code null} if this
     * data type does not support binary deltas rather than throwing
     * {@code IllegalArgumentException}.
     *
     * @return the delta type, or null if none
     * @since 6.3
     */
    DeltaType<V, BinaryDelta> binaryDeltaType();
}
