/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype;

import java.io.IOException;
import java.io.OutputStream;

/**
 * Optional extension provided by {@link DataType} implementations that support
 * incremental changes to values.
 *
 * <p>
 * A data type can optionally support incremental changes to values, represented
 * by one or more types of <em>delta</em>. A delta is the difference between two
 * values. For large or composite values that change in small steps, it is more
 * efficient to transmit an initial value followed by a delta for each change
 * than to transmit a complete value for each change.
 *
 * <p>
 * Each implementation specifies a <em>value</em> type and a <em>delta</em>
 * type. Two values, {@code oldValue} and {@code newValue}, can be compared
 * using {@link #diff(Object, Object)} to produce a delta. The delta can be
 * later applied to {@code oldValue} to re-create {@code newValue} using
 * {@link #apply(Object, Object)}.
 *
 * <p>
 * Delta types should provide a {@link Object#toString() string representation}.
 * {@link Object#equals(Object) Equality} is optional for delta types. The delta
 * type implementations used by Diffusion-provided data types implement both.
 *
 * <h3>Deferred parsing</h3>
 * <p>
 * Implementations can choose not to fully validate values and deltas when they
 * are read, but instead defer parsing until it is required. Consequently, all
 * methods that accept values and deltas can throw {@link InvalidDataException}.
 *
 * @author DiffusionData Limited
 * @since 5.7
 * @param <V> value class
 * @param <D> delta class
 */
public interface DeltaType<V, D> {

    /**
     * Returns the external identifier for this delta type.
     *
     * @return a unique name within the scope of the data type that provided
     *         this delta type
     *
     * @see DataType#deltaType(String)
     */
    String getName();

    /**
     * Serialize a delta to binary.
     */
    void writeDelta(D delta, OutputStream out) throws IOException;

    /**
     * Returns the serialized form of {@code delta} as a {@link Bytes}.
     *
     * @since 6.0
     */
    Bytes toBytes(D delta);

    /**
     * Create a delta from binary.
     *
     * <p>
     * Implementations can choose not to fully validate deltas when they are
     * read, but instead defer parsing until it is required.
     *
     * @param in the binary data. The implementation may re-use the array to
     *        avoid copying so the caller must ensure the array is not modified.
     * @throws InvalidDataException if {@code in} does not represent a valid
     *         delta
     * @throws IndexOutOfBoundsException if either {@code offset} or
     *         {@code length} is negative, or
     *         {@code offset + length > bytes.length}
     */
    D readDelta(byte[] in, int offset, int length)
        throws InvalidDataException, IndexOutOfBoundsException;

    /**
     * Create a delta from binary. Equivalent to
     * {@link #readDelta(byte[], int, int) readDelta(in, 0, in.length)}.
     */
    D readDelta(byte[] in) throws InvalidDataException;

    /**
     * Create a delta from binary. Equivalent to
     * {@link #readDelta(byte[]) readValue(bytes.toByteArray())}.
     *
     * @param in the binary data
     * @throws InvalidDataException if {@code in} does not represent a valid
     *         delta
     * @since 6.0
     */
    D readDelta(Bytes in) throws InvalidDataException;

    /**
     * Create a delta from two values.
     *
     * <p>
     * If there are many differences between oldValue and newValue, the result
     * might require more bytes to transmit than the new value, or be
     * computationally expensive to apply. In this case, it is better to discard
     * oldValue and publish newValue in its place. This can be checked using
     * {@link #isValueCheaper(Object, Object)}.
     *
     * <p>
     * The implementation can return the special constant {@link #noChange} to
     * indicate the oldValue and newValue are equivalent and there is no change
     * to publish.
     *
     * @return a delta representing the difference between oldValue and newValue
     *
     * @throws InvalidDataException if {@code oldValue} or {@code newValue} is
     *         invalid
     */
    D diff(V oldValue, V newValue) throws InvalidDataException;

    /**
     * Apply a delta to a value.
     *
     * @return the new value. {@code oldValue} will be returned if delta has no
     *         effect.
     *
     * @throws InvalidDataException if {@code oldValue} or {@code delta} is
     *         invalid
     */
    V apply(V oldValue, D delta) throws InvalidDataException;

    /**
     * Calculate if {@code value} is <em>cheaper</em> than {@code delta}. The
     * result is typically determined by the length of the serialized form but
     * can also consider the complexity of the delta.
     *
     * @return true if {@code value} is considered cheaper than {@code delta}
     *
     * @throws InvalidDataException if {@code value} or {@code delta} is invalid
     */
    boolean isValueCheaper(V value, D delta) throws InvalidDataException;

    /**
     * Constant returned by {@link #diff(Object, Object)} to indicate
     * {@code oldValue} and {@code newValue} are equivalent. The result is
     * guaranteed to be equal only to itself.
     */
    D noChange();
}
