/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.binary;

import com.pushtechnology.diffusion.datatype.BinaryDelta;
import com.pushtechnology.diffusion.datatype.Bytes;
import com.pushtechnology.diffusion.datatype.InvalidDataException;

/**
 * A read-only binary value with support for binary deltas.
 *
 * <P>
 * Values are effectively immutable. Instances can be backed by
 * {@link com.pushtechnology.diffusion.datatype.binary.BinaryDataType#readValue
 * user-supplied byte arrays}. Once a Binary has been constructed around an
 * array, care must be taken not to modify the data in the array because doing
 * so would violate immutability.
 *
 * @author DiffusionData Limited
 * @since 5.7
 * @see BinaryDataType
 */
public interface Binary extends Bytes {

    /**
     * Apply a binary delta to this value to create a new value.
     *
     * <p>
     * Convenient equivalent to
     * {@link com.pushtechnology.diffusion.datatype.DeltaType#apply(Object, Object)
     * Diffusion.dataTypes().binary().binaryDeltaType().apply(this, delta)}.
     *
     * @throws InvalidDataException if the delta is invalid
     */
    Binary apply(BinaryDelta delta) throws InvalidDataException;

    /**
     * Compare this value with an earlier version to calculate a binary delta.
     *
     * <p>
     * Convenient equivalent to
     * {@link com.pushtechnology.diffusion.datatype.DeltaType#diff(Object, Object)
     * Diffusion.dataTypes().binary().binaryDeltaType().diff(original, this)}.
     *
     * @param original the original Binary value to compare with this value
     * @return a binary delta representing the difference between original and
     *         this Binary
     */
    BinaryDelta diff(Binary original);
}