/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.binary;

import com.pushtechnology.diffusion.datatype.BinaryDelta;
import com.pushtechnology.diffusion.datatype.Bytes;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.DeltaType;
import com.pushtechnology.diffusion.datatype.json.JSON;

/**
 * Binary data type.
 *
 * <p>
 * {@link Binary} values can be used to store and transmit arbitrary
 * information. The responsibility for formatting and interpreting the
 * information belongs solely to the application. Before using Binary for a
 * topic, consider other data types such as {@link JSON}; these may provide a
 * simpler interface for your application.
 *
 * <p>
 * The implementation provides support for {@link #binaryDeltaType() binary
 * deltas}.
 *
 * @author DiffusionData Limited
 * @since 5.7
 * @see com.pushtechnology.diffusion.datatype.DataTypes#binary()
 */
public interface BinaryDataType extends DataType<Binary> {

    /**
     * Returns support for binary deltas.
     *
     * <p>
     * Equivalent to calling {@link #deltaType(Class)
     * deltaType(BinaryDelta.class)}.
     */
    @Override
    DeltaType<Binary, BinaryDelta> binaryDeltaType();

    /**
     * {@inheritDoc}
     * <p>
     * There are no invalid Binary instances, so this method does not throw
     * {@code InvalidDataException}.
     */
    @Override
    Binary readValue(byte[] bytes, int offset, int length)
        throws IndexOutOfBoundsException;

    /**
     * {@inheritDoc}
     * <p>
     * There are no invalid Binary instances, so this method does not throw
     * {@code InvalidDataException}.
     */
    @Override
    Binary readValue(byte[] bytes);

    /**
     * {@inheritDoc}
     * <p>
     * There are no invalid Binary instances, so this method does not throw
     * {@code InvalidDataException}.
     */
    @Override
    Binary readValue(Bytes bytes);

    /**
     * {@inheritDoc}
     * <p>
     * There are no invalid Binary instances, so this method does not throw
     * {@code InvalidDataException}.
     */
    @Override
    <T> T readAs(Class<T> classOfT, byte[] bytes, int offset, int length)
        throws IllegalArgumentException, IndexOutOfBoundsException;

    /**
     * {@inheritDoc}
     * <p>
     * There are no invalid Binary instances, so this method does not throw
     * {@code InvalidDataException}.
     */
    @Override
    <T> T readAs(Class<T> classOfT, byte[] bytes)
        throws IllegalArgumentException;

    /**
     * {@inheritDoc}
     * <p>
     * There are no invalid Binary instances, so this method does not throw
     * {@code InvalidDataException}.
     */
    @Override
    <T> T readAs(Class<T> classOfT, Bytes bytes)
        throws IllegalArgumentException;
}
