/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.json;

import com.pushtechnology.diffusion.datatype.BinaryDelta;
import com.pushtechnology.diffusion.datatype.Bytes;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.DeltaType;
import com.pushtechnology.diffusion.datatype.InvalidDataException;

/**
 * JSON data type.
 *
 * <p>
 * For efficiency, the JSON value is serialized in binary form following the
 * CBOR specification.
 *
 * <p>
 * The implementation provides support for {@link #binaryDeltaType() binary
 * deltas}.
 *
 * @author DiffusionData Limited
 * @since 5.7
 * @see com.pushtechnology.diffusion.datatype.DataTypes#json()
 */
public interface JSONDataType extends DataType<JSON> {

    /**
     * Returns support for binary deltas.
     *
     * <p>
     * Equivalent to calling {@link #deltaType(Class)
     * deltaType(BinaryDelta.class)}.
     */
    @Override
    DeltaType<JSON, BinaryDelta> binaryDeltaType();

    /**
     * Parse a JSON string.
     *
     * <p>
     * Precision for numeric types is lost in the translation to the internal
     * CBOR binary form and non-significant white space is not preserved. It is
     * not guaranteed that {@code fromJsonString(json).toJsonString()} equals
     * {@code json}.
     *
     * @throws InvalidDataException if json does not represent a single JSON
     *         value
     */
    JSON fromJsonString(String json) throws InvalidDataException;

    /**
     * Parse a value from CBOR-format binary.
     *
     * <p>
     * JSON instances defer parsing of binary data until required so this method
     * does not throw {@code InvalidDataException}. If the data is not valid, an
     * {@code InvalidDataException} may be thrown when a {@link JSON} method is
     * called.
     *
     * @param bytes the binary data. The implementation re-uses the array to avoid
     *        copying. The caller must ensure the array is not modified.
     * @see DataType#validate
     * @throws IndexOutOfBoundsException if either {@code offset} or
     *         {@code length} is negative, or
     *         {@code offset + length > bytes.length}
     */
    @Override
    JSON readValue(byte[] bytes, int offset, int length)
        throws IndexOutOfBoundsException;

    /**
     * Parse a value from CBOR-format binary. Equivalent to
     * {@link #readValue(byte[], int, int) readValue(in, 0, in.length)}.
     *
     * @param bytes the binary data. The implementation re-uses the array to
     *        avoid copying. The caller must ensure the array is not modified.
     */
    @Override
    JSON readValue(byte[] bytes);

    /**
     * Parse a value from CBOR-format binary. Equivalent to
     * {@link #readValue(byte[]) readValue(bytes.toByteArray())}.
     */
    @Override
    JSON readValue(Bytes bytes);
}
