/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.json;

import java.util.Map;

/**
 * Structural delta type for {@link JSON}.
 *
 * <p>
 * A JSONDelta describes the differences between two JSON values. Unlike a
 * {@link com.pushtechnology.diffusion.datatype.BinaryDelta binary delta}, a
 * structural delta can be queried to determine its effect. The
 * {@link #removed()} and {@link #inserted()} methods provide full details of
 * the differences between the two values.
 *
 * <p>
 * An instance can be created from two JSON values using
 * {@link JSON#diff(JSON)}.
 *
 * <p>
 * JSONDeltas are useful for identifying small changes to complex JSON values.
 * Here's any example of how to use this class to filter interesting changes in
 * a {@link com.pushtechnology.diffusion.client.features.Topics.ValueStream
 * ValueStream}.
 *
 * <pre>
 * public class ExampleStream implements ValueStream&lt;JSON&gt; {
 *   public void onValue(String topicPath, JSON oldValue, JSON newValue) {
 *     JSONDelta delta = newValue.diff(oldValue);
 *
 *     if (!delta.removed().intersection("/address").isEmpty() ||
 *         !delta.inserted().intersection("/address").isEmpty()) {
 *
 *         // The "address" field has changed.
 *         processAddress(newValue);
 *     }
 *   }
 *
 *   // ...
 * }
 * </pre>
 *
 * @author DiffusionData Limited
 * @since 5.7
 */
public interface JSONDelta {

    /**
     * Returns the parts of the first JSON value not found in the second JSON
     * value.
     *
     * @return the removed parts. The JSON Pointer references used for the keys
     *         are relative to first JSON value.
     */
    ChangeMap removed();

    /**
     * Returns the parts of the second JSON value not found in the first JSON
     * value.
     *
     * @return the removed parts. The JSON Pointer references used for the keys
     *         are relative to second JSON value.
     */
    ChangeMap inserted();

    /**
     * Returns whether the two JSON values used to create this instance are
     * different. It is equivalent to {!@code inserted().isEmpty() ||
     * !removed().isEmpty()}.
     *
     * @return true if this delta has an effect
     */
    boolean hasChanges();

    /**
     * An unmodifiable map describing the changes to a JSON value.
     *
     * <p>
     * The {@link JSONDelta#inserted()} method returns a
     * {@code ChangeMap} describing the parts of the second JSON value not found
     * in the first JSON value. Similarly, {@link JSONDelta#removed()}
     * returns a {@code ChangeMap} describing the parts of the first JSON value
     * not found in the second JSON value.
     *
     * <p>
     * The map contains an entry for each change, as follows:
     * <ul>
     * <li>The key is a <a href= "https://tools.ietf.org/html/rfc6901">JSON
     * Pointer</a> syntax reference locating the change in the complete value.
     * Since a JSON value is a list of zero or more data items, the reference
     * always begins with an array index. For example, the first part is
     * identified by the JSON Pointer {@code /0}.
     * <li>The value is part of the complete value. It is returned as a
     * {@link JSON} that can be parsed independently as a unit, or
     * {@link JSON#toJsonString() converted to JSON}.
     * </ul>
     *
     * <p>
     * Instances cannot be modified. Destructive methods such as
     * {@link Map#put(Object, Object)} throw
     * {@link UnsupportedOperationException} if called.
     *
     * <p>
     * An {@link IllegalArgumentException} will be thrown if an invalid JSON
     * pointer expression is passed to {@link #get(Object)},
     * {@link #containsKey(Object)}, {@link #descendants(String)}, or
     * {@link #intersection(String)}. This only occurs if the expression does
     * not start with {@code /} and is not empty.
     *
     */
    interface ChangeMap extends Map<String, JSON> {
        /**
         * Returns a view of the portion of this map whose keys are descendants
         * of {@code pointer}. If {@code pointer} is contained in this map, it
         * will be included in the result.
         *
         * @throws IllegalArgumentException if pointer is an invalid JSON
         *         Pointer expression
         */
        ChangeMap descendants(String pointer);

        /**
         * Returns a view of the portion of this map whose keys are descendants
         * or parents of {@code pointer}. If {@code pointer} is contained in
         * this map, it will be included in the result.
         *
         * <p>This method can be used to determine whether a structural
         * delta affects a particular part of a JSON value. For example:
         *
         * <pre>
         * if (!structuralDelta.removed().intersection("/contact/address").isEmpty()) {
         *   // The structural delta removes elements that affect '/contact/address'.
         * }
         * if (!structuralDelta.inserted().intersection("/contact/address").isEmpty()) {
         *   // The structural delta inserts elements that affect '/contact/address'.
         * }
         * </pre>
         *
         * @throws IllegalArgumentException if pointer is an invalid JSON
         *         Pointer expression
         */
        ChangeMap intersection(String pointer);
    }
}
