/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2;

import com.pushtechnology.diffusion.datatype.BinaryDelta;
import com.pushtechnology.diffusion.datatype.DataType;
import com.pushtechnology.diffusion.datatype.DeltaType;
import com.pushtechnology.diffusion.datatype.recordv2.model.MutableRecordModel;
import com.pushtechnology.diffusion.datatype.recordv2.model.RecordModel;
import com.pushtechnology.diffusion.datatype.recordv2.schema.Schema;
import com.pushtechnology.diffusion.datatype.recordv2.schema.SchemaBuilder;
import com.pushtechnology.diffusion.datatype.recordv2.schema.SchemaParseException;

/**
 * Record-based data type.
 * <P>
 * This provides the ability to handle data in Diffusion proprietary 'record'
 * format. This format allows string data to be organized into 'records' which
 * in turn are made up of 'fields'. Each field is a string but may be handled as
 * either an integer or a decimal field if required.
 * <P>
 * This format is compatible with legacy RECORD topics (removed in release 6.2)
 * and therefore can be used to migrate client applications from using such
 * topics with minimal change to the application.
 * <P>
 * The data can either be free format or constrained by a {@link Schema}.
 * <P>
 * In free format mode, no schema is associated with the data and the data will
 * contain zero or more records, each comprising zero or more fields. In this
 * mode the meaning of each field is entirely up to the application and no
 * validation will be performed by Diffusion, either in the client library, or
 * at the server. To write free format records, a {@link RecordV2Builder} can be
 * used to create a {@link RecordV2} object. Such a builder may be created using
 * the {@link #valueBuilder()} method.
 * <P>
 * When using a {@link Schema} then the permitted records and fields are defined
 * by the schema. The schema names the records and the fields within them and
 * provides a mechanism for direct access to the fields. The schema is also used
 * to validate the data to ensure it complies with the schema definition.
 * <P>
 * In schema mode, data can be created and updated using a
 * {@link MutableRecordModel} which allows records and fields to be conveniently
 * set and updated by name. A base model can be created from a schema using the
 * {@link Schema#createMutableModel()} method. The model can at any time be used
 * to create a new {@link RecordV2} object. A consumer of a {@link RecordV2}
 * value can read it as a {@link RecordModel} by simply using the
 * {@link RecordV2#asModel(Schema)} method to produce an immutable
 * representation of the data. When creating the data using a
 * {@link MutableRecordModel} then the model ensures that the data is valid and
 * therefore there is no need for the server or the consuming client to validate
 * the data.
 * <P>
 * Schemas can be parsed from JSON strings or more simply using a
 * {@link SchemaBuilder} obtained using the {@link #schemaBuilder()} method. A
 * schema can be bound to a {@link RecordV2} data type instance using the method
 * {@link #withSchema(Schema)}. This method will return a new RecordV2DataType
 * instance with the schema bound to it for validation.
 * <P>
 * A {@link RecordV2} object can only be validated within the context of a
 * {@link Schema}. For this reason if the {@link DataType#validate(Object)
 * validate} method is called on a dataType that has no bound schema, it will
 * always succeed.
 *
 * @author DiffusionData Limited
 * @since 6.0
 */
public interface RecordV2DataType extends DataType<RecordV2> {

    /**
     * Returns support for binary deltas.
     *
     * <p>
     * Equivalent to calling {@link #deltaType(Class)
     * deltaType(BinaryDelta.class)}.
     */
    @Override
    DeltaType<RecordV2, BinaryDelta> binaryDeltaType();

    /**
     * Parse a value from a {@link RecordV2} format byte array.
     * <P>
     * {@link RecordV2} instances can only be invalid in the context of a schema
     * so this method does not throw {@code InvalidDataException}.
     *
     * @param in the binary data. The implementation re-uses the array to avoid
     *        copying. The caller must ensure the array is not modified.
     *
     * @param offset the start position of the data to be read within the byte
     *        array
     *
     * @param length the length of the data to be read within the byte array
     *
     * @throws IndexOutOfBoundsException if either {@code offset} or
     *         {@code length} is negative, or
     *         {@code offset + length > bytes.length}
     */
    @Override
    RecordV2 readValue(byte[] in, int offset, int length)
        throws IndexOutOfBoundsException;

    /**
     * Parse a value from a {@link RecordV2} format byte array.
     * <P>
     * Equivalent to {@code readValue(in, 0, in.length)}.
     *
     * @param in the binary data. The implementation re-uses the array to avoid
     *        copying. The caller must ensure the array is not modified.
     */
    @Override
    RecordV2 readValue(byte[] in);

    /**
     * Parse a schema from a JSON string.
     *
     * @param json string containing a schema definition
     *
     * @return schema
     *
     * @throws SchemaParseException if a schema parsing error occurs.
     */
    Schema parseSchema(String json) throws SchemaParseException;

    /**
     * Creates a new {@link RecordV2Builder}.
     * <P>
     * Such a builder may be used to generate free format {@link RecordV2}
     * format value that is not constrained by a {@link Schema}.
     *
     * @return a new records builder
     */
    RecordV2Builder valueBuilder();

    /**
     * Creates a new schema builder.
     *
     * @return a new schema builder
     */
    SchemaBuilder schemaBuilder();

    /**
     * Bind a specific schema to a {@link RecordV2DataType} instance.
     *
     * @param schema to bind to the data type
     * @return a new {@link RecordV2DataType} instance bound to a specific
     *         schema
     */
    RecordV2DataType withSchema(Schema schema);

}
