/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2.model;

import java.math.RoundingMode;

import com.pushtechnology.diffusion.datatype.recordv2.RecordV2;
import com.pushtechnology.diffusion.datatype.recordv2.schema.Schema;
import com.pushtechnology.diffusion.datatype.recordv2.schema.SchemaViolationException;

/**
 * This is a mutable data model of {@link RecordV2} data based upon a
 * {@link Schema}.
 * <P>
 * An initial version of such a model can be created from a schema using the
 * {@link Schema#createMutableModel()} method. A model created in this way will
 * have all mandatory fields set to default values.
 * <P>
 * The model may then be updated as required and then at any time a
 * {@link RecordV2} object can be generated from the current state using
 * the {@link MutableRecordModel#asValue() asValue}
 * method. The {@link RecordV2} object may then be used to update a
 * topic.
 * <P>
 * When values for integer or decimal type fields are supplied the values are
 * validated and normalized. All number values will have any insignificant
 * leading zeroes removed. A decimal value will also be rounded to its specified
 * scale using {@link RoundingMode#HALF_UP half-up} rounding.
 * <P>
 * All mutator methods return the model so that calls can be chained.
 *
 * @author DiffusionData Limited
 * @since 6.0
 */
public interface MutableRecordModel extends RecordModel {

    /**
     * Sets a specified field value.
     *
     * @param recordName the name of the record containing the field
     *
     * @param recordIndex the index of the record containing the field
     *
     * @param fieldName the name of the field
     *
     * @param fieldIndex the index of the field
     *
     * @param value the new value
     *
     * @return this model
     *
     * @throws SchemaViolationException if the details given conflict with the
     *         schema or the value is incompatible with the schema field type
     *
     * @throws IndexOutOfBoundsException if an index is out of bounds
     */
    MutableRecordModel set(
        String recordName,
        int recordIndex,
        String fieldName,
        int fieldIndex,
        String value) throws SchemaViolationException,
        IndexOutOfBoundsException;

    /**
     * Sets a specified field value.
     * <P>
     * This allows an item to be addressed using a key of the form
     * recordName(recordIndex).fieldName(fieldIndex). Indexes may be omitted in
     * which case 0 is assumed. The record part may also be omitted in which
     * case the first occurrence of the first record is assumed.
     *
     * @param key the field key
     *
     * @param value the field value
     *
     * @return this model
     *
     * @throws SchemaViolationException if the key does not address a valid
     *         field
     *
     * @throws IndexOutOfBoundsException if a specified index is out of bounds
     *
     * @throws IllegalArgumentException of the key format is invalid
     *
     * @throws NumberFormatException if a specified index is not a valid number
     */
    MutableRecordModel set(
        String key,
        String value) throws SchemaViolationException,
        IndexOutOfBoundsException, IllegalArgumentException,
        NumberFormatException;

    /**
     * Adds new values to the end of a variable length field list.
     * <P>
     * This can only be used for a variable multiplicity field which can only be
     * the last field in a record and therefore the field does not need to be
     * named.
     *
     * @param recordName the name of the record
     *
     * @param recordIndex the index identifying the occurrence of the record
     *
     * @param values the values to add
     *
     * @return this model
     *
     * @throws SchemaViolationException if details conflict with the schema,
     *         possibly because the last field of the record is not a variable
     *         multiplicity field or the maximum number of occurrences for the
     *         field would be breached. This could also occur if one of the
     *         values is incompatible with the field type.
     *
     * @throws IndexOutOfBoundsException if the record index is out of bounds
     */
    MutableRecordModel add(
        String recordName,
        int recordIndex,
        String... values) throws SchemaViolationException,
        IndexOutOfBoundsException;

    /**
     * Adds new values to the end of a variable length field list.
     * <P>
     * This is a convenience method for adding to the end of the last record and
     * is therefore useful when there is only one record type.
     *
     * @param values the values
     *
     * @return this model
     *
     * @throws SchemaViolationException if details conflict with the schema,
     *         possibly because the last field of the last record is not a
     *         variable multiplicity field or the maximum number of occurrences
     *         for the field would be breached. This could also occur if one of
     *         the values is incompatible with the field type.
     */
    MutableRecordModel add(String... values)
        throws SchemaViolationException;

    /**
     * Adds a new initialized record occurrence to the end of a variable
     * multiplicity record list.
     * <P>
     * As the only variable multiplicity record can be the last one there is no
     * need to name the record. This method will add to the list of occurrences
     * of the last defined record. The record will be initialized with default
     * values appropriate to the schema definition and may then have individual
     * field items set separately.
     *
     * @return this model
     *
     * @throws SchemaViolationException if the last or only record is not a
     *         variable repeating record or has already reached the maximum
     *         number of occurrences
     */
    MutableRecordModel addRecord() throws SchemaViolationException;

    /**
     * Removes the specified occurrence of a variable multiplicity record.
     * <P>
     * A variable multiplicity record must be the last or only record within a
     * schema and therefore the record name is not required.
     *
     * @param index the index of the record to remove
     *
     * @return this model
     *
     * @throws SchemaViolationException if the last or only record is not a
     *         variable multiplicity record or can not be removed as it would
     *         violate the minimum number of occurrences
     *
     * @throws IndexOutOfBoundsException if the specified index is out of bounds
     */
    MutableRecordModel removeRecord(int index)
        throws SchemaViolationException, IndexOutOfBoundsException;

    /**
     * Removes the specified occurrence of a variable multiplicity field.
     * <P>
     * A variable multiplicity field must be the last or only field within a
     * record and therefore the field name is not required.
     *
     * @param recordName the name of the record
     *
     * @param recordIndex the record index
     *
     * @param fieldIndex the index of the field to remove
     *
     * @return this model
     *
     * @throws SchemaViolationException if the record is not known or the last
     *         or only field within the record is not a variable repeating field
     *         or can not be removed as it would violate the minimum number of
     *         occurrences
     *
     * @throws IndexOutOfBoundsException if one of the specified indexes is out
     *         of bounds
     */
    MutableRecordModel removeField(
        String recordName,
        int recordIndex,
        int fieldIndex) throws SchemaViolationException,
        IndexOutOfBoundsException;

    /**
     * Removes all optional instances of a variable multiplicity record.
     * <P>
     * As a variable repeating record can only be the last or only record then
     * the record name does not need to be specified.
     * <P>
     * This will only remove record occurrences down to the minimum number of
     * occurrences specified by the schema.
     * <P>
     * If the last or only record is not defined as variable multiplicity, this
     * would have no effect.
     *
     * @return this model
     */
    MutableRecordModel clearVariableRecords();

    /**
     * Remove all optional instances of a variable multiplicity field.
     * <P>
     * As a variable repeating field can only be the last or only field within a
     * record then the field name does not need to be specified.
     * <P>
     * This will only remove field occurrences down to the minimum number of
     * occurrences specified by the schema.
     * <P>
     * If the last or only field within the record is not defined as variable
     * multiplicity, this would have no effect.
     *
     * @param recordName the name of the record
     *
     * @param recordIndex the index of the record
     *
     * @return this model
     *
     * @throws SchemaViolationException if the record is not known.
     *
     * @throws IndexOutOfBoundsException if the specified index is out
     *         of bounds
     */
    MutableRecordModel clearVariableFields(String recordName, int recordIndex)
        throws SchemaViolationException, IndexOutOfBoundsException;

}
