/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2.model;

import com.pushtechnology.diffusion.datatype.recordv2.RecordV2;
import com.pushtechnology.diffusion.datatype.recordv2.schema.Schema;
import com.pushtechnology.diffusion.datatype.recordv2.schema.SchemaViolationException;

/**
 * {@link RecordV2} data model.
 * <P>
 * A read only model can be created from any {@link RecordV2} object
 * using the {@link RecordV2#asModel(Schema) asModel} method. The model
 * then provides direct access to the fields within the data. Fields may be
 * accessed either by explicitly specifying the record and field occurrence or
 * by specifying a key of the form:
 *
 * <pre>
 * recordName(recordIndex).fieldName(fieldIndex)
 * </pre>
 *
 * Indexes start from 0 and if omitted then 0 is assumed. The record name may
 * also be omitted, in which case the first record definition. This form of
 * addressing is useful when there is only one record definition.
 * <P>
 * So valid keys would be: <table>
 * <tr>
 * <td><code>Address(4).AddressLine(3)</code></td>
 * <td>The 4th <code>AddressLine</code> occurrence within the 5th
 * <code>Address</code> record</td>
 * </tr>
 * <tr>
 * <td><code>Address.Name</code></td>
 * <td>The first (or only) <code>Name</code> field within the first (or only)
 * <code>Address</code> record</td>
 * </tr>
 * <tr>
 * <td><code>AddressLine(1)</code></td>
 * <td>The 2nd <code>AddressLine</code> field within the first (or only) record
 * </td>
 * </tr>
 * <tr>
 * <td><code>Name</code></td>
 * <td>The first (or only) <code>Name</code> field within the first (or only)
 * record</td>
 * </tr>
 * </table>
 * <P>
 * The {@link #recordCount} and {@link #fieldCount} methods are useful for
 * determining the actual number of occurrences of variable multiplicity items.
 *
 * @author DiffusionData Limited
 * @since 6.0
 */
public interface RecordModel {

    /**
     * Get a field value.
     *
     * @param recordName the name of the record
     *
     * @param recordIndex the index of the record
     *
     * @param fieldName the name of the field
     *
     * @param fieldIndex the index of the field
     *
     * @return the field value
     *
     * @throws SchemaViolationException if the named record or field is not
     *         defined by the schema
     *
     * @throws IndexOutOfBoundsException if either the record or field index is
     *         out of bounds
     */
    String get(
        String recordName,
        int recordIndex,
        String fieldName,
        int fieldIndex) throws SchemaViolationException,
        IndexOutOfBoundsException;

    /**
     * Get a field value.
     * <P>
     * This allows an item to be addressed using a key of the form
     * recordName(recordIndex).fieldName(fieldIndex). Indexes may be omitted in
     * which case 0 is assumed. The record part may also be omitted in which
     * case the first occurrence of the first record is assumed.
     *
     * @param key the field key
     *
     * @return the field value
     *
     * @throws SchemaViolationException if the key does not address a valid
     *         field
     *
     * @throws IndexOutOfBoundsException if an index is out of bounds
     *
     * @throws IllegalArgumentException if the key format is invalid
     *
     * @throws NumberFormatException if an index is not a valid number
     */
    String get(String key) throws SchemaViolationException,
        IndexOutOfBoundsException, IllegalArgumentException,
        NumberFormatException;

    /**
     * Returns the actual number of occurrences of a named record.
     * <P>
     * If the record is not variable, this is the same as the defined number of
     * occurrences in the schema.
     *
     * @param recordName the record name
     *
     * @return the actual number of occurrences of the record
     *
     * @throws SchemaViolationException if the record is not defined in the
     *         schema
     */
    int recordCount(String recordName) throws SchemaViolationException;

    /**
     * Returns the actual number of occurrences of a named field within a
     * specified record occurrences.
     * <P>
     * For all but variable fields this simply returns the schema defined number
     * of occurrences of the field.
     *
     * @param recordName the record name
     *
     * @param recordIndex the record index
     *
     * @param fieldName the field name
     *
     * @return the actual number of occurrences of the field
     *
     * @throws SchemaViolationException if the record or field name are not
     *         defined in the schema
     *
     * @throws IndexOutOfBoundsException if the record index is out of bounds
     */
    int fieldCount(String recordName, int recordIndex, String fieldName)
        throws SchemaViolationException, IndexOutOfBoundsException;

    /**
     * Creates an immutable {@link RecordV2} object generated from the
     * model.
     *
     * @return new immutable {@link RecordV2} object
     */
    RecordV2 asValue();

}
