/*******************************************************************************
 * Copyright (c) 2023 DiffusionData Ltd., All Rights Reserved.
 *
 * Use is subject to license terms.
 *
 * NOTICE: All information contained herein is, and remains the
 * property of Push Technology. The intellectual and technical
 * concepts contained herein are proprietary to Push Technology and
 * may be covered by U.S. and Foreign Patents, patents in process, and
 * are protected by trade secret or copyright law.
 *******************************************************************************/
package com.pushtechnology.diffusion.datatype.recordv2.schema;

import com.pushtechnology.diffusion.datatype.recordv2.RecordV2DataType;

/**
 * Used to build an immutable {@link Schema}.
 * <P>
 * A schema defines the records and fields that may occur in a RecordV2 topic
 * value.
 * <P>
 * The schema must declare at least one record type and every record must have
 * at least one field type declared.
 * <P>
 * Every record type and field type has a 'multiplicity' which defines the
 * number of times that the record or field may occur within the data.
 * Multiplicity is specified as a 'minimum' and 'maximum' number of occurrences
 * or where the minimum and maximum are the same (fixed multiplicity) then the
 * multiplicity may be specified as a single 'occurs' value. If the minimum and
 * maximum are different, this is referred to a 'variable' multiplicity. Only
 * the last record declared or the last field within a record may have variable
 * multiplicity. The maximum value may be declared as -1 to indicate that the
 * record or field can have an unlimited number of occurrences.
 * <P>
 * The builder is used to add a record definition followed by the fields within
 * it. After all fields have been added to a record another may then be added,
 * and so on, and then finally {@link #build()} is called to create an immutable
 * schema object.
 * <P>
 * Every call returns the builder instance allowing calls to be chained, for
 * example:
 *
 * <pre>
 * Schema schema = builder.record(&quot;R1&quot;).string(&quot;S1&quot;).string(&quot;S2&quot;, 1, 5)
 *     .record(&quot;R2&quot;, 0, -1).decimal(&quot;D&quot;, 5).build();
 * </pre>
 *
 * A builder is obtained using the
 * {@link RecordV2DataType#schemaBuilder()} method.
 *
 * @author DiffusionData Limited
 * @since 6.0
 */
public interface SchemaBuilder {

    /**
     * Add a new single occurrence record to the schema.
     * <P>
     * This is the equivalent to calling {@link #record(String, int)
     * record(name, 1)}
     *
     * @param name the record name
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder record(String name) throws SchemaViolationException;

    /**
     * Add a new fixed multiplicity record to the schema.
     * <P>
     * This is the equivalent to calling {@link #record(String, int, int)
     * record(name, occurs, occurs)} .
     *
     * @param name the record name
     *
     * @param occurs the number of times the record is to occur. This must be a
     *        positive value.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder record(String name, int occurs)
        throws SchemaViolationException;

    /**
     * Add a new record to the schema.
     * <P>
     * The record added must not have the same name as a previously added
     * record.
     * <P>
     * A record may not be added after a record with variable multiplicity.
     * <P>
     * A record may not be added directly after another record which has had no
     * fields added.
     *
     * @param name the record name
     *
     * @param min the minimum number of occurrences of the record. This must not
     *        be negative.
     *
     * @param max the maximum number of occurrences of the record. This must
     *        either be -1 to indicate an unlimited number or it must be a
     *        positive number greater than or equal to {@code min}.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder record(String name, int min, int max)
        throws SchemaViolationException;

    /**
     * Add a single occurrence string field to the current record.
     * <P>
     * This is the equivalent of calling {@link #string(String, int)
     * string(name, 1)}.
     *
     * @param name field name
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder string(String name) throws SchemaViolationException;

    /**
     * Add a fixed multiplicity string field to the current record.
     * <P>
     * This is the equivalent of calling {@link #string(String, int, int)
     * string(name, occurs, occurs)}.
     *
     * @param name field name
     *
     * @param occurs the fixed number of times the field should occur within the
     *        record. This must be positive.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder string(String name, int occurs)
        throws SchemaViolationException;

    /**
     * Add a string field to the current record.
     * <P>
     * A field may not be added after a field that has variable multiplicity
     * (min != max).
     *
     * @param name field name. This must not be the same as any field already
     *        added to the record.
     *
     * @param min the minimum number of times the field may occur within the
     *        record. This must not be negative.
     *
     * @param max the maximum number of times that the field may occur within
     *        the record. This must either be -1 (indicating no upper limit) or
     *        a positive value that is not less than min.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder string(String name, int min, int max)
        throws SchemaViolationException;

    /**
     * Add a single occurrence integer field to the current record.
     * <P>
     * This is the equivalent of calling {@link #integer(String, int)
     * integer(name, 1)}.
     *
     * @param name field name
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder integer(String name) throws SchemaViolationException;

    /**
     * Add a fixed multiplicity integer field to the current record.
     * <P>
     * This is the equivalent of calling {@link #integer(String, int, int)
     * integer(name, occurs, occurs)}.
     *
     * @param name field name
     *
     * @param occurs the fixed number of times the field should occur within the
     *        record. This must be positive.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder integer(String name, int occurs)
        throws SchemaViolationException;

    /**
     * Add an integer field to the current record.
     * <P>
     * A field may not be added after a field that has variable multiplicity
     * (min != max).
     *
     * @param name field name. This must not be the same as any field already
     *        added to the record.
     *
     * @param min the minimum number of times the field may occur within the
     *        record. This must not be negative.
     *
     * @param max the maximum number of times that the field may occur within
     *        the record. This must either be -1 (indicating no upper limit) or
     *        a positive value that is not less than min.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder integer(String name, int min, int max)
        throws SchemaViolationException;

    /**
     * Add a single occurrence decimal field to the current record.
     * <P>
     * This is the equivalent of calling {@link #decimal(String, int, int)
     * decimal(name, scale, 1)}.
     *
     * @param name field name
     *
     * @param scale the scale of the field (the number of decimal places). This
     *        must be a positive value.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder decimal(String name, int scale)
        throws SchemaViolationException;

    /**
     * Add a fixed multiplicity decimal field to the current record.
     * <P>
     * This is the equivalent of calling {@link #decimal(String, int, int, int)
     * decimal(name, scale, occurs, occurs)}.
     *
     * @param name field name
     *
     * @param scale the scale of the field (the number of decimal places). This
     *        must be a positive value.
     *
     * @param occurs the fixed number of times the field should occur within the
     *        record. This must be positive.
     *
     * @return this builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder decimal(String name, int scale, int occurs)
        throws SchemaViolationException;

    /**
     * Add a decimal field to the current record.
     * <P>
     * A field may not be added after a field that has variable multiplicity
     * (min != max).
     *
     * @param name field name. This must not be the same as any field already
     *        added to the builder.
     *
     * @param scale the scale of the field (the number of decimal places). This
     *        must be a positive value.
     *
     * @param min the minimum number of times the field may occur within the
     *        record. This must not be negative.
     *
     * @param max the maximum number of times that the field may occur within
     *        the record. This must either be -1 (indicating no upper limit) or
     *        a positive value that is not less than min.
     *
     * @return the builder
     *
     * @throws SchemaViolationException if schema rules are violated
     */
    SchemaBuilder decimal(String name, int scale, int min, int max)
        throws SchemaViolationException;

    /**
     * Build an immutable Schema.
     * <P>
     * At least one record with at least one field must have been added to the
     * builder.
     *
     * @return a new immutable schema object representing the current state of
     *         the builder
     *
     * @throws SchemaViolationException if no record have been specified to the
     *         builder.
     */
    Schema build() throws SchemaViolationException;

}
