package com.flutterwave.services;

import com.flutterwave.bean.ListResponse;
import com.flutterwave.bean.Response;
import org.apache.http.NameValuePair;
import org.apache.http.message.BasicNameValuePair;
import org.json.JSONObject;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import static com.flutterwave.bean.ChargeTypes.*;
import static com.flutterwave.client.Utility.get;
import static com.flutterwave.client.Utility.post;
import static com.flutterwave.utility.Properties.getProperty;

/**
 * APIs endpoints that are important for general transaction management and operations.
 * Some of these operations include transaction queries, confirming transaction fees, resending failed webhook responses and managing refunds.
 * @author Cleopatra Douglas
 */
public class Transactions {
    private String ERROR = "Error processing request, please check logs";

    /**
     * Query the final status of a transaction. This can be used to check transactions of all payment types after they have been attempted.
     * @param id int This is a unique transaction identifier generated by our systems. It is returned in the initiate charge response as data.id
     * @return Response
     */
    public Response runVerifyTransaction(int id) {
        return Optional.of(get(getProperty("VERIFY_TRANSACTION_ENDPOINT") + id + "/verify",
                        VERIFY_TRANSACTION, null))
                .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    /**
     * Query previously initiated transactions. You can do a single or bulk query with the endpoint depending on your use case.
     * @return ListResponse
     */
    public ListResponse runGetTransactions(Optional<String> from, Optional<String> to, Optional<String> page,
                                           Optional<String> customer_email,Optional<String> status,Optional<String> tx_ref,
                                           Optional<String> customer_fullname, Optional<String> currency) {

        List<NameValuePair> nameValuePairs = new ArrayList<>();
        from.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("from", from.get())));
        to.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("to", to.get())));
        page.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("page", page.get())));
        customer_email.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("customer_email", customer_email.get())));
        status.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("status", status.get())));
        tx_ref.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("tx_ref", tx_ref.get())));
        customer_fullname.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("customer_fullname", customer_fullname.get())));
        currency.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("currency", currency.get())));

        return Optional.of(get(getProperty("VERIFY_TRANSACTION_ENDPOINT"),
                        GET_TRANSACTION, nameValuePairs))
                .map(ListResponse::toListResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    /**
     * Query the fees expected to be paid for a particular transaction. This endpoint only returns fees for collections i.e. inflows.
     * @param amount BigDecimal This is the amount of the product or service to be charged from the customer.
     * @param currency String This is the specified currency to charge in.
     * @return Response
     */
    public Response runGetTransactionsFees(BigDecimal amount, String currency) {
        List<NameValuePair> nameValuePairs = new ArrayList<>();
        nameValuePairs.add(new BasicNameValuePair("amount", amount.toString()));
        nameValuePairs.add(new BasicNameValuePair("currency", currency));

        return Optional.of(get(getProperty("VERIFY_TRANSACTION_ENDPOINT") + "/fee", GET_TRANSACTION, nameValuePairs))
                .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    /**
     * Resend webhooks from failed sending queues to your server.
     * @param id int This is a unique transaction identifier generated by our systems. It is returned in the initiate charge response as data.id
     * @param wait Optional  This parameter would hold for the hook response and return what you respond with as the response. The expected value is 1.
     * @return Response
     */
    public Response runResendWebhook(int id, Optional<Integer> wait) {
        List<NameValuePair> nameValuePairs = new ArrayList<>();
        wait.ifPresent(s -> nameValuePairs.add(new BasicNameValuePair("wait", wait.toString())));

        return Optional.of(post(getProperty("VERIFY_TRANSACTION_ENDPOINT") + id + "/resend-hook",
                        null,
                        GET_TRANSACTION,
                        nameValuePairs))
                .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    /**
     * View the timeline for a transaction. A transaction timeline is a list of events that happened to a selected transaction.
     * Some key events include: loading the payment modal, Switching payment methods and Entering details in modal fields.
     * Using the response, you can get insights into the transactions and payment behaviour of users.
     * @param id int This is a unique transaction identifier generated by our systems. It is returned in the initiate charge and verify transaction responses as data.id
     * @return ListResponse
     */
    public ListResponse runViewTimeline(int id) {
        return Optional.of(get(getProperty("VERIFY_TRANSACTION_ENDPOINT") + id + "/events",
                        GET_TRANSACTION, null))
                .map(ListResponse::toListResponse).orElseThrow(() -> new RuntimeException(ERROR));
    }

    public class Refunds {
        public ListResponse runGet(Optional<String> from, Optional<String> to, Optional<String> status,
                                   Optional<String> currency,Optional<String> id) {

            List<NameValuePair> nameValuePairs = new ArrayList<>();
            from.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("from", from.get())));
            to.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("to", to.get())));
            currency.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("currency", currency.get())));
            id.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("id", id.get())));
            status.ifPresent(String -> nameValuePairs.add(new BasicNameValuePair("status", status.get())));

            return Optional.of(get(getProperty("REFUND_ENDPOINT"),
                            REFUND, nameValuePairs))
                    .map(ListResponse::toListResponse).orElseThrow(() -> new RuntimeException(ERROR));
        }

        public Response runRefund(int id, Optional<BigDecimal> amount) {
            return Optional.of(post(getProperty("VERIFY_TRANSACTION_ENDPOINT") + id + "/refund",
                            new JSONObject().put("amount", amount).toString(), REFUND, null))
                    .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
        }

        public Response runGetDetails(int id) {
            return Optional.of(get(getProperty("VERIFY_TRANSACTION_ENDPOINT") + id + "/refund",
                            REFUND, null))
                    .map(Response::toResponse).orElseThrow(() -> new RuntimeException(ERROR));
        }


    }
}
