package com.github.developframework.jsonview.core;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.github.developframework.jsonview.core.element.JsonviewConfiguration;
import com.github.developframework.jsonview.core.xml.ConfigurationSource;
import com.github.developframework.jsonview.core.xml.FileConfigurationSource;
import com.github.developframework.jsonview.core.xml.JsonviewConfigurationSaxReader;
import lombok.Getter;

import java.util.HashSet;
import java.util.Objects;
import java.util.Set;

/**
 * jsonview factory
 * 
 * @author qiuzhenhao
 *
 */
public class JsonviewFactory {

	@Getter
	private JsonviewConfiguration jsonviewConfiguration;

	public JsonviewFactory(String... configs) {
		this(new ObjectMapper(), configs);
	}

	public JsonviewFactory(Set<String> configs) {
		this(new ObjectMapper(), configs);
	}

	/**
	 * constructor for string array
	 *
	 * @param objectMapper objectMapper
	 * @param configs configuration filename array
	 */
	public JsonviewFactory(ObjectMapper objectMapper, String... configs) {
		Objects.requireNonNull(objectMapper);
		Objects.requireNonNull(configs);
		Set<ConfigurationSource> sources = new HashSet<>();
		for (String config : configs) {
			sources.add(new FileConfigurationSource(config));
		}
		JsonviewConfigurationSaxReader reader = new JsonviewConfigurationSaxReader(sources);
		jsonviewConfiguration = reader.readConfiguration();
		jsonviewConfiguration.setObjectMapper(objectMapper);
	}

	/**
	 * constructor for string set
	 *
	 * @param objectMapper objectMapper
	 * @param configs configuration filename set
	 */
	public JsonviewFactory(ObjectMapper objectMapper, Set<String> configs) {
		Objects.requireNonNull(objectMapper);
		Objects.requireNonNull(configs);
		Set<ConfigurationSource> sources = new HashSet<>();
		for (String config : configs) {
			sources.add(new FileConfigurationSource(config));
		}
		JsonviewConfigurationSaxReader reader = new JsonviewConfigurationSaxReader(sources);
		jsonviewConfiguration = reader.readConfiguration();
		jsonviewConfiguration.setObjectMapper(objectMapper);
	}

	/**
	 * constructor for jsonviewConfiguration
	 *
	 * @param objectMapper objectMapper
	 * @param jsonviewConfiguration jsonviewConfiguration
	 */
	public JsonviewFactory(ObjectMapper objectMapper, JsonviewConfiguration jsonviewConfiguration) {
		Objects.requireNonNull(jsonviewConfiguration);
		Objects.requireNonNull(objectMapper);
		jsonviewConfiguration.setObjectMapper(objectMapper);
		this.jsonviewConfiguration = jsonviewConfiguration;
	}

	/**
	 * Create a json creator, use the default objectMapper object
	 * 
	 * @return JsonCreator instance
	 */
	public JsonCreator getJsonCreator() {
		return new JsonCreatorImpl(jsonviewConfiguration);
	}

	/**
	 * get objectMapper
	 * 
	 * @return objectMapper
	 */
	public ObjectMapper getObjectMapper() {
		return jsonviewConfiguration.getObjectMapper();
	}
}
