package com.github.jonpereiradev.diffobjects.builder;


import com.github.jonpereiradev.diffobjects.DiffException;
import com.github.jonpereiradev.diffobjects.strategy.DiffMetadata;

import java.util.Map;


/**
 * Responsible to map a class and fields to be able to generate diffs.
 *
 * @author Jonathan Pereira
 * @see DiffInstanceBuilder
 * @see DiffMappingBuilder
 * @see DiffConfiguration
 * @since 1.0
 */
class DiffQueryBuilderImpl implements DiffQueryBuilder {

    private final DiffMetadata diffMetadata;
    private final Map<String, DiffMetadata> metadatas;
    private final DiffMappingAllBuilder diffMappingAllBuilder;

    DiffQueryBuilderImpl(String field, Map<String, DiffMetadata> metadatas, DiffMappingAllBuilder diffMappingAllBuilder) {
        this.diffMetadata = metadatas.get(field);

        if (this.diffMetadata == null) {
            throw new DiffException("No field \"" + field + "\" mapped in builder. You need to map the field before query.");
        }

        this.diffMappingAllBuilder = diffMappingAllBuilder;
        this.metadatas = metadatas;
    }

    /**
     * Define a property for the last mapping.
     *
     * @param key the identifier of the property.
     * @param value the value of the property.
     * @return the instance of this mapping.
     */
    @Override
    public DiffQueryBuilder property(String key, String value) {
        diffMetadata.getProperties().put(key, value);
        return this;
    }

    /**
     * Removes the property from the mapping.
     *
     * @return the instance of this builder.
     */
    @Override
    public DiffMappingAllBuilder unmapping() {
        metadatas.remove(diffMetadata.getProperties().get("field"));
        return diffMappingAllBuilder;
    }

    /**
     * Gets the configuration instance to get the configuration generated by this instance instance.
     *
     * @return a configuration instance instance.
     */
    @Override
    public DiffConfiguration configuration() {
        return new DiffConfigurationImpl(metadatas);
    }
}
