package com.github.jonpereiradev.diffobjects.builder;


import com.github.jonpereiradev.diffobjects.comparator.DiffComparator;
import com.github.jonpereiradev.diffobjects.strategy.DiffMetadata;


/**
 * Responsible to map a class and fields to be able to generate diffs.
 *
 * @author Jonathan Pereira
 * @see DiffInstanceBuilder
 * @see DiffMappingBuilder
 * @see DiffConfiguration
 * @since 1.0
 */
final class DiffQueryMappingBuilderImpl<T> implements DiffQueryMappingBuilder<T> {

    private final DiffMetadata diffMetadata;
    private final DiffMappingBuilderImpl<T> diffMappingBuilder;

    DiffQueryMappingBuilderImpl(DiffMetadata diffMetadata, DiffMappingBuilderImpl<T> diffMappingBuilder) {
        this.diffMetadata = diffMetadata;
        this.diffMappingBuilder = diffMappingBuilder;
    }

    /**
     * Define a property for the last mapping.
     *
     * @param key the identifier of the property.
     * @param value the value of the property.
     *
     * @return the instance of this mapping.
     */
    @Override
    public DiffQueryMappingBuilder<T> property(String key, String value) {
        diffMetadata.getProperties().put(key, value);
        return this;
    }

    /**
     * Removes the property from the mapping.
     *
     * @return the instance of this builder.
     */
    @Override
    public DiffMappingBuilder<T> unmapping() {
        diffMappingBuilder.getMetadatas().remove(diffMetadata.getProperties().get("field"));
        return diffMappingBuilder;
    }

    /**
     * Maps the getter of the field for the class.
     *
     * @param field name of the field that will me used to find the getter method.
     *
     * @return the instance of this mapping.
     */
    @Override
    public DiffQueryMappingBuilder<T> mapping(String field) {
        return diffMappingBuilder.mapping(field);
    }

    /**
     * Maps a new getter of the field for the class.
     *
     * @param field name of the field that will me used to find the getter method.
     * @param comparator implementation that define how two objects will be check for equality.
     *
     * @return the instance of this mapping.
     */
    @Override
    public <F> DiffQueryMappingBuilder<T> mapping(String field, Class<F> fieldClass, DiffComparator<F> comparator) {
        return diffMappingBuilder.mapping(field, fieldClass, comparator);
    }

    /**
     * Gets the configuration instance to get the configuration generated by this instance instance.
     *
     * @return a configuration instance instance.
     */
    @Override
    public DiffConfiguration configuration() {
        return new DiffConfigurationImpl(diffMappingBuilder.getMetadatas());
    }

}
