package cdc.issues.api;

import java.util.ArrayList;
import java.util.List;

import cdc.util.lang.Checks;

/**
 * Base class used to describe an Issue.
 * <p>
 * <b>Node:</b> This class can be specialized if necessary.
 * <p>
 * An issue may have several targets.<br>
 * For example, when there is a compliance issue between several things,
 * one can not tell which one is at fault.
 *
 * @author Damien Carbonne
 *
 * @param <T> The issue type.
 */
public class Issue<T> {
    private final T type;
    private final IssueLevel level;
    private final String description;
    private final IssueLocation[] locations;

    protected Issue(T type,
                    IssueLevel level,
                    String description,
                    List<IssueLocation> locations) {
        this.type = Checks.isNotNull(type, "type");
        this.level = Checks.isNotNull(level, "level");
        this.description = Checks.isNotNull(description, "description");
        this.locations = locations.toArray(new IssueLocation[locations.size()]);
    }

    public T getType() {
        return type;
    }

    public IssueLevel getLevel() {
        return level;
    }

    public String getDescription() {
        return description;
    }

    public IssueLocation[] getLocations() {
        return locations.clone();
    }

    public IssueLocation getLocationAt(int index) {
        return locations[index];
    }

    public <L extends IssueLocation> L getLocationAt(int index,
                                                     Class<L> cls) {
        return cls.cast(getLocationAt(index));
    }

    public static <T> Builder<T> builder(Class<T> typeClass) {
        return new Builder<>();
    }

    public abstract static class AbstractIssueBuilder<B extends AbstractIssueBuilder<B, I, T>, I extends Issue<T>, T> {
        protected T type;
        protected IssueLevel level;
        protected String description;
        protected final List<IssueLocation> locations = new ArrayList<>();

        protected AbstractIssueBuilder() {
        }

        protected abstract B self();

        public B type(T type) {
            this.type = type;
            return self();
        }

        public B level(IssueLevel level) {
            this.level = level;
            return self();
        }

        public B description(String description) {
            this.description = description;
            return self();
        }

        public B addLocation(IssueLocation location) {
            this.locations.add(location);
            return self();
        }

        public abstract I build();
    }

    public static class Builder<T> extends AbstractIssueBuilder<Builder<T>, Issue<T>, T> {
        @Override
        protected Builder<T> self() {
            return this;
        }

        @Override
        public Issue<T> build() {
            return new Issue<>(type,
                               level,
                               description,
                               locations);
        }
    }
}