package cdc.issues.api.locations;

import cdc.issues.api.IssueLocation;
import cdc.util.lang.Checks;

/**
 * Implementation of IssueLocation dedicated to workbooks.
 *
 * @author Damien Carbonne
 */
public class WorkbookIssueLocation implements IssueLocation {
    private final String systemId;
    private final String sheetName;
    private final int rowNumber;
    private final String columnName;

    private WorkbookIssueLocation(String systemId,
                                  String sheetName,
                                  int rowNumber,
                                  String columnName) {
        this.systemId = Checks.isNotNull(systemId, "systemId");
        this.sheetName = sheetName;
        this.rowNumber = rowNumber;
        this.columnName = columnName;
        if (rowNumber <= 0 && columnName != null) {
            throw new IllegalArgumentException("Non compliant column number of column name.");
        }
        if (rowNumber > 0 && sheetName == null) {
            throw new IllegalArgumentException("Sheet name is missing.");
        }
    }

    @Override
    public String getTargetId() {
        return getSystemId();
    }

    @Override
    public String getPath() {
        if (isCellLocation()) {
            return getSheetName() + ":" + getRowNumber() + ":" + getColumnName();
        } else {
            return getSheetName() == null ? "" : getSheetName();
        }
    }

    public String getSystemId() {
        return systemId;
    }

    public String getSheetName() {
        return sheetName;
    }

    public int getRowNumber() {
        return rowNumber;
    }

    public String getColumnName() {
        return columnName;
    }

    public boolean isCellLocation() {
        return rowNumber > 0;
    }

    public boolean isSheetLocation() {
        return sheetName != null && !isCellLocation();
    }

    @Override
    public String toString() {
        if (isCellLocation()) {
            return getSystemId()
                    + ":" + getSheetName()
                    + ":" + getRowNumber()
                    + ":" + getColumnName();
        } else {
            return getSystemId()
                    + ":" + getSheetName();
        }
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String systemId;
        private String sheetName;
        private int rowNumber = 0;
        private String columnName;

        private Builder() {
            super();
        }

        public Builder systemId(String systemId) {
            this.systemId = systemId;
            return this;
        }

        public Builder sheetName(String sheetName) {
            this.sheetName = sheetName;
            return this;
        }

        public Builder rowNumber(int rowNumber) {
            this.rowNumber = rowNumber;
            return this;
        }

        public Builder columnName(String columnName) {
            this.columnName = columnName;
            return this;
        }

        public WorkbookIssueLocation build() {
            return new WorkbookIssueLocation(systemId,
                                             sheetName,
                                             rowNumber,
                                             columnName);
        }
    }
}