package cdc.issues.io;

import java.io.File;
import java.io.IOException;
import java.util.EnumSet;
import java.util.Set;

import cdc.issues.answers.IssuesAndAnswers;
import cdc.util.events.ProgressController;
import cdc.util.lang.Checks;

@FunctionalInterface
public interface IssuesReader {

    /**
     * Class used to configure writing of issues.
     */
    public static class Settings {
        /**
         * Enumeration of possible hints.
         */
        public enum Hint {
            /** If set, answers are not loaded. */
            NO_ANSWERS
        }

        private final Set<Hint> hints;

        protected Settings(Set<Hint> hints) {
            this.hints = Checks.isNotNull(hints, "hints");
        }

        /**
         * @return The set of enabled hints.
         */
        public Set<Hint> getHints() {
            return hints;
        }

        public static Builder builder() {
            return new Builder();
        }

        public static class Builder {
            private final Set<Hint> hints = EnumSet.noneOf(Hint.class);

            protected Builder() {
            }

            public Builder hint(Hint hint) {
                this.hints.add(hint);
                return this;
            }

            public Settings build() {
                return new Settings(hints);
            }
        }
    }

    public static final Settings ANSWERS =
            Settings.builder()
                    .build();

    public static final Settings NO_ANSWERS =
            Settings.builder()
                    .hint(Settings.Hint.NO_ANSWERS)
                    .build();

    public IssuesAndAnswers load(Settings settings,
                                 ProgressController controller) throws IOException;

    public static IssuesAndAnswers load(File file,
                                        Settings settings,
                                        ProgressController controller,
                                        IssuesIoFactoryFeatures features) throws IOException {
        final IssuesIoFactory factory = new IssuesIoFactory(features);
        final IssuesReader reader = factory.createIssuesReader(file);
        return reader.load(settings, controller);
    }
}