package cdc.issues.rules;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import cdc.issues.Issue;
import cdc.issues.IssuesHandler;
import cdc.issues.locations.Location;
import cdc.util.lang.CollectionUtils;

/**
 * Interface implemented by classes that can detect issues.
 *
 * @author Damien Carbonne
 *
 * @param <T> The data type onto which issue can be detected.
 */
public interface IssuesDetector<T> {

    /**
     * Interface implemented by classes that can describe
     * and create an IssuesDetector.
     *
     * @author Damien Carbonne
     *
     * @param <T> The data type onto which issue can be detected.
     */
    public interface Factory<T> {
        /**
         * @return The class of data that can be analyzed.
         */
        public Class<T> getDataClass();

        /**
         * @return The set of all supported Rules.
         */
        public Set<Rule> getSupportedRules();

        /**
         * Creates and configures an IssuesDetector to analyze some rules.
         *
         * @param project The project for which detection is run.
         * @param snapshot The snapshot for which detection is run.
         * @param configuredRules The configured rules for which analysis must be performed.<br>
         *            Rules must be a subset of supported rules and associated
         *            params must comply with expected params.
         * @return A newly created IssuesDetector, configured with
         *         {@code configuredRules} and {@code rules}.
         */
        public IssuesDetector<T> create(String project,
                                        String snapshot,
                                        Set<ConfiguredRule> configuredRules);

        /**
         * Creates and configures an IssuesDetector to analyze some rules.
         *
         * @param project The project for which detection is run.
         * @param snapshot The snapshot for which detection is run.
         * @param configuredRules The configured rules for which analysis must be performed.<br>
         *            Rules must be a subset of supported rules and associated
         *            params must comply with expected params.
         * @return A newly created IssuesDetector, configured with
         *         {@code configuredRules} and {@code rules}.
         */
        public default IssuesDetector<T> create(String project,
                                                String snapshot,
                                                ConfiguredRule... configuredRules) {
            final Set<ConfiguredRule> set = new HashSet<>();
            Collections.addAll(set, configuredRules);
            return create(project, snapshot, set);
        }

        /**
         * Creates and configures an IssuesDetector to analyze some rules.
         *
         * @param project The project for which detection is run.
         * @param snapshot The snapshot for which detection is run.
         * @param configuredRule The configured rule for which analysis must be performed.<br>
         *            The rule must be one of supported rules and params must
         *            comply with {@link Rule#getParams()}.
         * @return A newly created IssuesDetector, configured with
         *         {@code configuredRules}.
         */
        public default IssuesDetector<T> create(String project,
                                                String snapshot,
                                                ConfiguredRule configuredRule) {
            final Set<ConfiguredRule> configuredRules = new HashSet<>();
            configuredRules.add(configuredRule);
            return create(project, snapshot, configuredRules);
        }
    }

    public abstract static class AbstractFactory<T> implements Factory<T> {
        private final Class<T> dataClass;
        private final Set<Rule> rules;

        protected AbstractFactory(Class<T> dataClass,
                                  Rule... rules) {
            this.dataClass = dataClass;
            this.rules = Collections.unmodifiableSet(CollectionUtils.toSet(rules));
        }

        @Override
        public final Class<T> getDataClass() {
            return dataClass;
        }

        @Override
        public final Set<Rule> getSupportedRules() {
            return rules;
        }
    }

    /**
     * @return The factory that was used to create this IssuesDetector.
     */
    public Factory<T> getFactory();

    /**
     * @return The class of data that can be analyzed.
     */
    public default Class<T> getDataClass() {
        return getFactory().getDataClass();
    }

    /**
     * @return The project name of analyzed data.
     */
    public String getProject();

    /**
     * @return The snapshot name for which analysis is done.
     */
    public String getSnapshot();

    /**
     * @return The set of enabled rules.
     */
    public Set<Rule> getEnabledRules();

    /**
     * @return The set of enabled ConfiguredRules.
     */
    public Set<ConfiguredRule> getEnabledConfiguredRules();

    /**
     * Analyzes one data.
     *
     * @param data The data to analyze.
     * @param locations The data locations. Must <em>NOT</em> be empty.
     * @param issuesHandler The issues handler that must be used to store issues.
     */
    public void analyze(T data,
                        List<Location> locations,
                        IssuesHandler<Issue> issuesHandler);

    /**
     * Analyzes one data.
     *
     * @param data The data to analyze.
     * @param location The data location.
     * @param issuesHandler The issues handler that must be used to store issues.
     */
    public default void analyze(T data,
                                Location location,
                                IssuesHandler<Issue> issuesHandler) {
        final List<Location> locations = new ArrayList<>();
        locations.add(location);
        analyze(data, locations, issuesHandler);
    }

    public static String toString(IssuesDetector<?> detector) {
        final StringBuilder builder = new StringBuilder();
        builder.append("IssuesDetector<")
               .append(detector.getDataClass().getSimpleName())
               .append(">(");
        boolean first = true;
        for (final ConfiguredRule crule : detector.getEnabledConfiguredRules()) {
            if (first) {
                first = false;
            } else {
                builder.append(",");
            }
            builder.append(crule);
        }
        builder.append(")");
        return builder.toString();
    }
}