package cdc.issues;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;

import cdc.issues.locations.Location;
import cdc.issues.rules.RuleId;
import cdc.util.lang.Checks;

/**
 * Base class used to describe an Issue (more exactly an occurrence of
 * detection of an issue).
 * <p>
 * <b>Node:</b> This class can be specialized if necessary.
 * <p>
 * An issue may have several targets.<br>
 * For example, when there is a compliance issue between several things,
 * one can not tell which one is at fault.
 *
 * @author Damien Carbonne
 */
public class Issue {
    /**
     * Comparator of Issues sing timestamp.
     */
    public static final Comparator<Issue> TIMESTAMP_COMPARATOR =
            Comparator.comparing(Issue::getTimestamp);

    private final Instant timestamp;
    private final IssueId id;
    private final String snapshot;
    private final IssueSeverity severity;
    private final String description;
    private final Params metas;

    /**
     * Creates an Issue.
     *
     * @param timestamp The optional timestamp.
     * @param domain The rule domain.
     * @param name The rule name.
     * @param params The rule parameters.
     * @param project The project name.
     * @param locations The target locations.
     * @param snapshot The issue snapshot.
     * @param severity The issue severity.
     * @param description The issue description.
     * @param metas The issue meta data.
     */
    protected Issue(Instant timestamp,
                    String domain,
                    String name,
                    Params params,
                    String project,
                    List<? extends Location> locations,
                    String snapshot,
                    IssueSeverity severity,
                    String description,
                    Params metas) {
        this.timestamp = timestamp == null
                ? Instant.now()
                : timestamp;
        this.id = IssueId.builder()
                         .domain(domain)
                         .name(name)
                         .params(params)
                         .project(project)
                         .locations(locations)
                         .build();
        this.snapshot = snapshot;
        this.severity = Checks.isNotNull(severity, "severity");
        this.description = Checks.isNotNull(description, "description");
        this.metas = Checks.isNotNull(metas, "metas");
    }

    protected Issue(String domain,
                    String name,
                    Params params,
                    String project,
                    List<? extends Location> locations,
                    String snapshot,
                    IssueSeverity severity,
                    String description,
                    Params metas) {
        this(null,
             domain,
             name,
             params,
             project,
             locations,
             snapshot,
             severity,
             description,
             metas);
    }

    protected Issue(Instant timestamp,
                    String domain,
                    Enum<?> name,
                    Params params,
                    String project,
                    List<? extends Location> locations,
                    String snapshot,
                    IssueSeverity severity,
                    String description,
                    Params metas) {
        this(timestamp,
             domain,
             name.name(),
             params,
             project,
             locations,
             snapshot,
             severity,
             description,
             metas);
    }

    protected Issue(String domain,
                    Enum<?> name,
                    Params params,
                    String project,
                    List<? extends Location> locations,
                    String snapshot,
                    IssueSeverity severity,
                    String description,
                    Params metas) {
        this(null,
             domain,
             name,
             params,
             project,
             locations,
             snapshot,
             severity,
             description,
             metas);
    }

    /**
     * @return The Instant at which this issue was created.
     */
    public final Instant getTimestamp() {
        return timestamp;
    }

    /**
     * @return The id of this issue.
     */
    public IssueId getId() {
        return id;
    }

    /**
     * @return The RuleId of this issue.
     */
    public RuleId getRuleId() {
        return id.getRuleId();
    }

    /**
     * @return The domain of the rule of this issue.
     */
    public String getDomain() {
        return id.getDomain();
    }

    /**
     * @return The name of the rule of this rule.
     */
    public String getName() {
        return id.getName();
    }

    public <T extends Enum<T>> T getName(Class<T> typeClass) {
        return id.getName(typeClass);
    }

    /**
     * @return The rule parameters of this issue.
     */
    public Params getParams() {
        return id.getParams();
    }

    /**
     * @return The project of this issue.
     *         May be {@code null}.
     */
    public String getProject() {
        return id.getProject();
    }

    /**
     * @return The snapshot of this issue.
     *         May be {@code null}.
     */
    public String getSnapshot() {
        return snapshot;
    }

    /**
     * @return The severity of this issue.
     */
    public final IssueSeverity getSeverity() {
        return severity;
    }

    /**
     * @return The description of this issue.
     */
    public final String getDescription() {
        return description;
    }

    /**
     * @return The meta data associated to this issue.
     */
    public Params getMetas() {
        return metas;
    }

    /**
     * @return The target locations of this issue.
     */
    public Location[] getLocations() {
        return id.getLocations();
    }

    /**
     * @return The number of target locations of this issue.
     */
    public final int getNumberOfLocations() {
        return id.getLocations().length;
    }

    public Location getLocationAt(int index) {
        return id.getLocations()[index];
    }

    public <L extends Location> L getLocationAt(int index,
                                                Class<L> cls) {
        return cls.cast(getLocationAt(index));
    }

    @Override
    public int hashCode() {
        return Objects.hash(timestamp,
                            id,
                            severity,
                            description,
                            metas);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (object == null || getClass() != object.getClass()) {
            return false;
        }
        final Issue other = (Issue) object;
        return Objects.equals(this.timestamp, other.timestamp)
                && Objects.equals(this.id, other.id)
                && Objects.equals(this.snapshot, other.snapshot)
                && this.severity == other.severity
                && Objects.equals(this.description, other.description)
                && Objects.equals(this.metas, other.metas);
    }

    @Override
    public String toString() {
        return getTimestamp()
                + " - " + getDomain()
                + " - " + getName()
                + " - " + getParams()
                + " - " + getProject()
                + " - " + getSnapshot()
                + " - " + getSeverity()
                + " - " + getDescription()
                + " - " + Arrays.toString(getLocations())
                + " - " + getMetas();
    }

    public static Builder<?> builder() {
        return new Builder<>();
    }

    public static class Builder<B extends Builder<B>> {
        protected Instant timestamp = null;
        protected String domain;
        protected String name;
        protected Params params = Params.NO_PARAMS;
        protected String project;
        protected String snapshot;
        protected IssueSeverity severity;
        protected String description = "";
        protected final List<Location> locations = new ArrayList<>();
        protected Params metas = Params.NO_PARAMS;

        protected Builder() {
        }

        @SuppressWarnings("unchecked")
        protected B self() {
            return (B) this;
        }

        /**
         * Sets the issue timestamp.
         * <p>
         * <b>WARNING:</b> this should only be used to reconstruct an issue -from a file, stream, ...).
         *
         * @param timestamp The timestamp.
         * @return This builder.
         */
        public B timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return self();
        }

        /**
         * Sets the issue RuleId.
         * <p>
         * This is equivalent to setting its domain and name.
         *
         * @param ruleId The rule id.
         * @return This builder.
         */
        public B ruleId(RuleId ruleId) {
            this.domain = ruleId.getDomain();
            this.name = ruleId.getName();
            return self();
        }

        /**
         * Sets the issue domain.
         *
         * @param domain The domain.
         * @return This builder.
         */
        public B domain(String domain) {
            this.domain = domain;
            return self();
        }

        /**
         * Sets the issue name.
         *
         * @param name The name.
         * @return This builder.
         */
        public B name(String name) {
            this.name = name;
            return self();
        }

        /**
         * Sets the issue name and its severity if the {@code name} implements
         * {@link IssueSeverityItem} and current severity is {@code null}.
         *
         * @param name The name.
         * @return This builder.
         */
        public B name(Enum<?> name) {
            this.name = name.name();
            if (severity == null && name instanceof IssueSeverityItem) {
                severity(((IssueSeverityItem) name).getSeverity());
            }
            return self();
        }

        public B params(Params params) {
            this.params = params;
            return self();
        }

        /**
         * Sets the effective parameters that have been used to configure the issue detection.
         *
         * @param project The project name.
         * @return This builder.
         */
        public B project(String project) {
            this.project = project;
            return self();
        }

        /**
         * Adds an issue location.
         *
         * @param location The location.
         * @return This builder.
         */
        public B addLocation(Location location) {
            this.locations.add(location);
            return self();
        }

        /**
         * Sets the issue locations.
         *
         * @param locations The locations
         * @return This builder.
         */
        public B locations(Location... locations) {
            this.locations.clear();
            Collections.addAll(this.locations, locations);
            return self();
        }

        /**
         * Sets the issue locations.
         *
         * @param locations The locations
         * @return This builder.
         */
        public B locations(List<Location> locations) {
            this.locations.clear();
            this.locations.addAll(locations);
            return self();
        }

        /**
         * Sets the issue snapshot.
         *
         * @param snapshot The snapshot identifier.
         * @return This builder.
         */
        public B snapshot(String snapshot) {
            this.snapshot = snapshot;
            return self();
        }

        /**
         * Sets the issue severity.
         *
         * @param severity The severity.
         * @return This builder.
         */
        public B severity(IssueSeverity severity) {
            this.severity = severity;
            return self();
        }

        /**
         * Sets the issue description.
         *
         * @param description The description.
         * @return This builder.
         */
        public B description(String description) {
            Checks.isNotNull(description, "description");
            this.description = description;
            return self();
        }

        /**
         * Sets the issue meta data.
         *
         * @param metas The meta data.
         * @return This builder.
         */
        public B metas(Params metas) {
            this.metas = metas;
            return self();
        }

        public Issue build() {
            return new Issue(timestamp,
                             domain,
                             name,
                             params,
                             project,
                             locations,
                             snapshot,
                             severity,
                             description,
                             metas);
        }
    }
}